# -*- coding: utf-8 -*-
import os
import json
import itertools
import contextlib
import collections

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search import bisect
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.wizard import parameters as wp
from sandbox.projects.release_machine.components.configs.begemot import BegemotCfg


class Version(parameters.SandboxIntegerParameter):
    name = "version"
    required = True
    description = "Wizard release version"


class UseAppHost(parameters.SandboxBoolParameter):
    name = "use_apphost"
    description = "Beta uses apphost"


def _last_commit_to_branch(version, component='wizard'):
    return bisect.branch_last_rev_first_minor_last_minor('arcadia:/arc/branches/{}/stable-{}'.format(component, version))[0]


def _get_wizard_build_info(kind, rev_a, rev_b):
    return iter(bisect.get_build_info(
        BegemotCfg.Testenv.trunk_db, 'BUILD_WIZARD_' + kind.upper(), rev_a, rev_b
    ))


def _load_testenv_wizard_builds(rev_a, rev_b):
    sources = {
        resource_types.REMOTE_WIZARD: _get_wizard_build_info('executable', rev_a, rev_b),
        resource_types.WIZARD_CONFIG: _get_wizard_build_info('config', rev_a, rev_b),
        resource_types.WIZARD_SHARD:  _get_wizard_build_info('data', rev_a, rev_b),
    }
    while True:
        parts = {t: next(s) for t, s in sources.items()}
        while True:
            expect = max(p['revision'] for p in parts.values())
            for t, p in parts.items():
                if p['revision'] < expect or p['task_status'] != 'SUCCESS':
                    parts[t] = next(sources[t])
                    break
            else:
                break
        resources = {}
        for t, p in parts.items():
            res = channel.sandbox.list_resources(t, task_id=p['task_id'])
            if not res:
                eh.check_failed('build task {} did not build {}'.format(p['task_id'], t))
            resources[t] = res[0].id
        v = next(iter(parts.values()))
        v['wizard_resources'] = resources
        yield v


@contextlib.contextmanager
def _temporary_sync(task, resources):
    synced = []
    try:
        for r in resources:
            synced.append(task.sync_resource(r))
        yield synced
    finally:
        for r in synced:
            parent = os.path.normpath(os.path.join(r, '..'))
            paths.add_write_permissions_for_path(parent, False)
            paths.add_write_permissions_for_path(r)
            paths.remove_path(r)
            paths.remove_write_permissions_for_path(parent, False)


class BisectWizardDiffReason(bisect.Bisect):
    """
        Ищем номер комита в wizard, вызвавший diff в выдаче на указанном запросе
        (+ указывается позиция в которой изменился документ).
        Для определения дипазона ревизий в которых нужно искать diff указывается номер версии wizard.
    """

    type = "BISECT_WIZARD_DIFF_REASON"
    required_ram = 50 * 1024
    input_parameters = bisect.Bisect.input_parameters + (Version, wp.RuntimeData, UseAppHost)

    def on_execute(self):
        self._doccache = {}
        version = self.ctx[Version.name]
        old = _last_commit_to_branch(version - 1)
        new = _last_commit_to_branch(version)
        self.set_bisect_info(bisect.BisectInfo.version, version)
        self.set_bisect_info(bisect.BisectInfo.revisions, '{}:{}'.format(old, new))
        self.bisect(list(_load_testenv_wizard_builds(old, new)))

    def get_docs(self, row, requrl):
        resources = row['wizard_resources']
        revision = row['revision']
        if revision in self._doccache:
            return self._doccache[revision]
        runtime = self.sync_resource(self.ctx[wp.RuntimeData.name])
        variants = []
        with _temporary_sync(self, [
            resources[resource_types.REMOTE_WIZARD],
            resources[resource_types.WIZARD_CONFIG],
            resources[resource_types.WIZARD_SHARD],
        ]) as (binary, config, shard):
            paths.make_folder('wizard_tmp', delete_content=True)
            with sc.Wizard(self.abs_path('wizard_tmp'), binary, 32123, sc.sconf.SearchConfig, config, shard, runtime, False) as component:
                if self.ctx.get(UseAppHost.name):
                    param = 'srcrwr=WIZARD%3A{}%3A{}'.format(self.local_host, int(component.port) + 1)
                else:
                    param = 'wizhosts={}%3A{}'.format(self.local_host, component.port)
                for _ in range(10):
                    rsp = json.loads(self.read(requrl + '&waitall=da&json_dump=searchdata&' + param))
                    variants.append(self._get_docs(rsp))
        docs = [collections.Counter(xs).most_common(1)[0][0] for xs in itertools.izip(*variants)]
        self._doccache[revision] = docs
        return docs

    def _get_docs(self, response):
        result = []
        for doc in response['searchdata']['docs']:
            doc_data = doc['url']
            for attr in self.doc_attrs:
                doc_data += ' {}={}'.format(attr, repr(doc.get(attr)))
            result.append(doc_data)
        return result


__Task__ = BisectWizardDiffReason
