import os

from sandbox import sdk2
import sandbox.projects.common.wizard.utils as w_utils

from .build_tasks import (
    BuildBegemotData,
    BuildFastDataDeployer,
    BuildExecutable,
    BuildBegemotBstr,
    CollectStaticFiles,
)

from sandbox.projects.common import binary_task
from sandbox.projects.common.build.KosherYaMake import KosherYaMake
from sandbox.projects.websearch.begemot import AllBegemotServices
from sandbox.projects.websearch.begemot import BW_RESOURCES
from sandbox.projects.websearch.begemot.common import Begemots
from sandbox.projects.websearch.begemot.resources import BEGEMOT_SHARD_UPDATER
from sandbox.projects.websearch.upper.resources import FastDataDeployer


EXECUTABLE_TARGETS = [
    'BegemotExecutable', 'MegamindExecutable', 
    'AdvqExecutable', 'BegginsExecutable', 
    'RequestInitExecutable', 'Bstr'
]


class BuildBegemotCommon(KosherYaMake, binary_task.LastBinaryTaskRelease):

    class Requirements(sdk2.Requirements):
        pass

    class Parameters(KosherYaMake.Parameters):
        executable_param = binary_task.binary_release_parameters(stable=True)
        with sdk2.parameters.RadioGroup("Build mode") as mode:
            mode.values["build_data"] = mode.Value("Begemot Data", default=True)
            mode.values["build_executable"] = mode.Value("Begemot Executable")
            mode.values["build_bstr"] = mode.Value("Begemot Bstr")
            mode.values["build_fast_data_deployer"] = mode.Value("Fast Data Deployer")
            mode.values["collect_files"] = mode.Value("Collect static files")

        with mode.value["build_data"]:
            with sdk2.parameters.CheckGroup('Shard names to build. If you don\'t build fresh, select one shard') as shards_to_build:
                for shard_name in sorted(Begemots.keys()):
                    shard = Begemots[shard_name]
                    setattr(
                        shards_to_build.values, shard_name,
                        shards_to_build.Value(shard_name, checked=shard.release_fresh),
                    )
            build_fresh = sdk2.parameters.Bool("Build fresh data instead of slow data", default=False)
            all_in_one_config = sdk2.parameters.Bool("Build only one fast build config with all rules (actual for fresh)", default=False,)
            build_test_shard = sdk2.parameters.Bool("Apply testpatches to data", default=False)
            use_full_shard_build = sdk2.parameters.Bool("Full shard build", default=False)
            use_fast_build = sdk2.parameters.Bool("Fast Build", default=False)
            force_rebuild = sdk2.parameters.Bool("Rebuild all rules in fast build mode", default=False)
            fast_build_cache = sdk2.parameters.String("Fast build cache path in yt", default="//home/search-runtime/begemot-fast-build")
            shard_updater = sdk2.parameters.Resource("Shard updater for YT (if empty, use last released)", resource_type=BEGEMOT_SHARD_UPDATER)
            test_shard_updater = sdk2.parameters.Bool("Fail task if shard updater failed", default=False)
            separate_build = sdk2.parameters.Bool("Multithreading fresh build", default=False)

            with separate_build.value[True]:
                threads_count = sdk2.parameters.Integer("Threads count for separate fresh build", default=1)

            cypress_cache = sdk2.parameters.String("Begemot data path in Cypress", default="//home/begemot/begemot-tests/shards_cache")
            yt_proxy = sdk2.parameters.String("YT_PROXY", default='hahn')
            yt_token_testenv = sdk2.parameters.YavSecret("YAV token for testenv", default="sec-01g3e1n4ebff90pnxk8rf492xc")
            is_precommit = sdk2.parameters.Bool("Is precommit check from pr", default=False)
        
        with mode.value["build_executable"]:
            executable_targets = {}
            for target in EXECUTABLE_TARGETS:
                executable_targets.update(
                    {res.name: res for res in BW_RESOURCES[target]}
                )
            with sdk2.parameters.CheckGroup('Begemot executables to build') as targets:
                for resource_name in sorted(executable_targets.keys()):
                    setattr(targets.values, resource_name, targets.Value(resource_name))
        
        with mode.value["collect_files"]:
            static_files = {resource.__name__: resource for resource in BW_RESOURCES['EventsFiles'] + BW_RESOURCES['NannyConfigs']}
            with sdk2.parameters.CheckGroup('Begemot files to collect') as static_targets:
                for resource_name in sorted(static_files.keys()):
                    setattr(static_targets.values, resource_name, static_targets.Value(resource_name))

        parent_task_id = sdk2.parameters.Integer("Parent task id", default=-1)

        with sdk2.parameters.Output():
            output_resources = sdk2.parameters.Dict("Output resources type with id")

    def _shard_size_gb(self, shard_name):
        return Begemots[shard_name].shard_size_gb

    def _create_task(self):
        if self.Parameters.mode == 'build_data':
            self.build_task = BuildBegemotData(
                task=self,
                ctx=self.Context,
                arcadia_url=self.Parameters.checkout_arcadia_from_url,
                shard_name=self.Parameters.shards_to_build,
                build_fresh=self.Parameters.build_fresh,
                all_in_one_config=self.Parameters.all_in_one_config,
                build_test_shard=self.Parameters.build_test_shard,
                use_full_shard_build=self.Parameters.use_full_shard_build,
                use_fast_build=self.Parameters.use_fast_build,
                force_rebuild=self.Parameters.force_rebuild,
                fast_build_cache=self.Parameters.fast_build_cache,
                shard_updater=self.Parameters.shard_updater,
                test_shard_updater=self.Parameters.test_shard_updater,
                separate_build=self.Parameters.separate_build,
                threads_count=self.Parameters.threads_count,
                cypress_cache=self.Parameters.cypress_cache,
                yav_token=self.Parameters.yt_token_testenv.data()['yav_token'],
                yt_proxy=self.Parameters.yt_proxy,
                parent_id=self.Parameters.parent_task_id,
                is_precommit=self.Parameters.is_precommit,
            )
        elif self.Parameters.mode == 'build_fast_data_deployer':
            self.build_task=BuildFastDataDeployer(
                arcadia_url=self.Parameters.checkout_arcadia_from_url,
                task=self,
            )
        elif self.Parameters.mode == 'build_executable':
            self.build_task = BuildExecutable(
                arcadia_url=self.Parameters.checkout_arcadia_from_url,
                task=self,
                targets=self.Parameters.executable_targets,
                selected_targets=self.Parameters.targets,
                def_flags=self.Parameters.definition_flags,
            )
        elif self.Parameters.mode == 'build_bstr':
            self.build_task = BuildBegemotBstr(
                arcadia_url=self.Parameters.checkout_arcadia_from_url,
                task=self,
            )
        elif self.Parameters.mode == 'collect_files':
            self.build_task = CollectStaticFiles(
                task=self,
                static_targets=self.Parameters.static_targets,
                static_resources=self.Parameters.static_files,
            )

    def on_enqueue(self):
        if self.Parameters.mode == 'build_data':
            self.Requirements.cores = 24
            self.Requirements.disk_space = max(self._shard_size_gb(self.Parameters.shards_to_build[0]) * 1024 * 3, 10240) if not self.Parameters.build_fresh else 50 * 1024
            self.Parameters.kill_timeout = (3 * (self._shard_size_gb(self.Parameters.shards_to_build[0]) > 150) + 2) * 60 * 60
            if self.Parameters.shards_to_build[0].startswith('Spellchecker'):
                self.Requirements.disk_space += self._shard_size_gb(self.Parameters.shards_to_build[0]) * 1024
                self.Parameters.kill_timeout += 2.5 * 60 * 60
            self.Parameters.tags = self.Parameters.shards_to_build
            w_utils.setup_hosts(self)
        elif self.Parameters.mode == 'build_executable':
            self.Requirements.cores = 24
            self.Requirements.ram = 50 * 1024
            self.Requirements.disk_space = 150 * 1024
            self.Parameters.kill_timeout = 6 * 3600
        elif self.Parameters.mode == 'build_bstr':
            self.Requirements.disk_space = 60 * 1024

        super(BuildBegemotCommon, self).on_enqueue()

    def on_execute(self):
        self._create_task()
        super(BuildBegemotCommon, self).on_execute()

    def pre_build(self, source_dir):
        self.build_task.pre_build(source_dir)

    def get_resources(self):
        if self.Parameters.mode == 'build_data':
            if not self.Parameters.use_full_shard_build:
                return {}
            shard_name = self.Parameters.shards_to_build[0]
            resource = {
                'description': 'Begemot shard ' + shard_name,
                'resource_path': shard_name,
                'resource_type': AllBegemotServices.Service[shard_name].data_resource_type_test
            }
            return {'projects': resource}
        elif self.Parameters.mode == 'build_fast_data_deployer':
            build_path = 'search/tools/fast_data_deployment/deployer'
            resources = [FastDataDeployer]
            result = {}
            for r in resources:
                result[r.name] = {
                    'description': r.name,
                    'resource_type': r,
                    'resource_path': os.path.basename(build_path)
                }
            return result
        elif self.Parameters.mode == 'build_executable':
            resources = {}
            for resource_name in self.Parameters.targets:
                build_path = os.path.basename(self.Parameters.executable_targets[resource_name].arcadia_build_path)
                resources[resource_name] = {
                    'description': build_path,
                    'resource_path': build_path,
                    'resource_type': resource_name
                }
            return resources
        elif self.Parameters.mode == 'build_bstr':
            resources = {}
            for r in BW_RESOURCES['Bstr']:
                resources[r.name] = {
                    'description': r.name,
                    'resource_type': r,
                    'resource_path': os.path.join(os.path.basename(r.arcadia_build_path), os.path.basename(r.arcadia_build_path)),
                }
            return resources
        elif self.Parameters.mode == 'collect_files':
            return {}

    def get_targets(self):
        return self.build_task.get_targets()

    def get_arts(self):
        return self.build_task.get_arts()

    def get_build_def_flags(self):
        return self.build_task.get_build_def_flags()

    def post_build(self, source_dir, output_dir, pack_dir):
        self.build_task.post_build(source_dir, output_dir, pack_dir)
