import datetime
import logging
from logging.config import valid_ident
import traceback
import os
import subprocess

from .build_task import BuildTask

from sandbox import sdk2
import sandbox.projects.common.wizard.utils as w_utils
from sandbox.common.errors import TaskFailure
from sandbox.common.types.notification import Transport
from sandbox.common.utils import get_task_link
from sandbox.projects.websearch.begemot.common import Begemots
from sandbox.projects.websearch.begemot.common.fast_build import FastBuilder
from ..cypress_shard_updater import CypressShardUpdater
from sandbox.sdk2.vcs.svn import Arcadia


class BuildBegemotData(CypressShardUpdater, BuildTask):
    def __init__(self, **kwargs):
        logging.info("Build begemot data constructor in")

        self._task = kwargs['task']

        self._arcadia_url = kwargs['arcadia_url']

        self._shard_names = kwargs['shard_name']
        self._build_fresh = kwargs['build_fresh']
        self._all_in_one_config = kwargs['all_in_one_config']
        self._build_test_shard = kwargs['build_test_shard']
        self._use_full_shard_build = kwargs['use_full_shard_build']
        self._use_fast_build = kwargs['use_fast_build']
        self._force_rebuild = kwargs['force_rebuild']
        self._fast_build_cache = kwargs['fast_build_cache']
        self._shard_updater = kwargs['shard_updater']
        self._test_shard_updater = kwargs['test_shard_updater']
        self._separate_build = kwargs['separate_build']
        self._threads_count = kwargs['threads_count']
        self._parent_id = kwargs['parent_id']
        self._ctx = kwargs['ctx']
        self._is_precommit = kwargs['is_precommit']

        super(BuildBegemotData, self).__init__(kwargs['cypress_cache'], kwargs['yt_proxy'], kwargs['yav_token'], self._task, self._parent_id)

        if not self._use_full_shard_build and not self._use_fast_build:
            raise TaskFailure("Neither 'Full' nor 'Fast' build is selected")

        if self._build_fresh and self._use_full_shard_build:
            raise TaskFailure("Can't run fill build for fresh. Select fast build or use WIZARD_RUNTIME_BUILD task")

        if not len(self._shard_names):
            raise TaskFailure("Select shard to build")

        if not self._build_fresh and len(self._shard_names) > 1:
            raise TaskFailure("Select exactly one shard for slow data build")
        self._shard_name = self._shard_names[0]

    @property
    def execution_space(self):
        return max(self._shard_size_gb(self._shard_name) * 1024 * 3, 10240) if not self._build_fresh else 50 * 1024

    @property
    def kill_timeout(self):
        return (3 * (self._shard_size_gb(self._shard_name) > 150) + 2) * 60 * 60

    def pre_build(self, source_dir):
        logging.info("Build begemot data pre_build in")
        if not self._use_fast_build:
            return

        do_rebuild = self._force_rebuild
        testenv_db = self._ctx.testenv_database
        if testenv_db is not None and 'ws-begemot' in testenv_db and 'trunk' not in testenv_db:
            if self._build_fresh:
                do_rebuild = True

        try:
            with_cypress = self._cypress_cache is not None and len(self._cypress_cache)
            self.fast_builder = FastBuilder(
                self._fast_build_cache,
                source_dir,
                ' '.join(self._shard_names),
                self._build_test_shard,
                fresh=self._build_fresh,
                one_config=self._all_in_one_config,
                rebuild=do_rebuild,
                additional_flags=self.get_build_def_flags(),
                with_cypress=with_cypress,
                checkout=False,  # TODO arcadia url
                separate_build=self._separate_build,
                threads=self._threads_count,
            )
            self.fast_builder.make_checkout(self._arcadia_url, clear_build=False)  # TODO arcadia params
            rules = self.fast_builder.resolve_rules_to_build()
            self._ctx.rules_list = rules
            if not self._use_full_shard_build:
                self.fast_builder.create_targets()
        except Exception as e:
            logging.error("Exception is FastBuilder: %s", traceback.print_exc())
            raise TaskFailure("Prebuild failed: %s" % e)

    def get_targets(self):
        if not self._use_full_shard_build:
            return self.fast_builder.get_targets()

        build_path = os.path.join(Begemots.shards_path, self._shard_name)
        logging.debug('Build path is : {}'.format(build_path))
        return [build_path]

    def get_arts(self):
        if not self._use_full_shard_build:
            return []

        shard_name = self._shard_name
        return [
            {
                'path': os.path.join(Begemots.shards_path, shard_name, 'search', 'wizard', 'data', 'wizard', '*'),
                'dest': shard_name,
            }
        ]

    def get_build_def_flags(self):
        if self._shard_name == 'Geo':
            return '-DGEO_WIZARD_SHARD'
        return ''

    def _apply_testpatches(self, path):
        os.system('chmod -R a+w ' + path)
        for curr, dirs, files in os.walk(path):
            for file in files:
                origin_file = os.path.join(curr, file)
                patch_file = origin_file + ".testpatch"
                logging.debug("Try patch %s", origin_file)
                if os.path.exists(patch_file):
                    logging.info("Found testpatch for file %s", files)
                    sdk2.helpers.subprocess.run(["patch", origin_file, patch_file])

    def _last_change(self, default):
        url = self._arcadia_url
        if url.startswith(Arcadia.ARCADIA_HG_SCHEME):
            return default
        if '#' in url:
            revision = url[url.index('#') + 2:]
            try:
                return int(revision)
            except ValueError:
                return url[url.index('#') + 1:]
        info = Arcadia.info(url)
        if not info:
            return default
        return info.get('commit_revision', default)

    def _make_version_info(self):
        revision = Arcadia.get_revision(self._arcadia_url)
        return {
            "GenerationTime": '"{}"'.format(datetime.datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ')),
            "Revision": self._last_change(revision),
            "Task": self._task.id,
            "ShardName": '"{}"'.format(self._shard_name),
        }

    def _create_version_file(self, path):
        version_info = self._make_version_info()
        with open(os.path.join(path, 'version.pb.txt'), 'w') as version:
            for key, value in version_info.items():
                version.write('{}: {}\n'.format(key, value))

    def post_build(self, source_dir, output_dir, pack_dir):
        if self._is_precommit:
            return

        pack_path = os.path.join(pack_dir, self._shard_names[0]) if not self._build_fresh else os.path.join(pack_dir, "rules")

        if not self._use_full_shard_build:
            self.fast_builder.pack_rules(output_dir)

        if self._build_test_shard:
            self._apply_testpatches(pack_path)

        cypress_config = self.get_cypress_cache_config()
        existing_cypress_paths = []

        if self._use_fast_build:
            self.fast_builder.get_failed_rules_logs(self._task)
            if self._use_full_shard_build:
                self.fast_builder.make_own_pack()
            self.fast_builder.make_resources(self._task, self._make_version_info())
            self._ctx.begemot_shard_resource_ids = {}
            for shard, res_type in self.fast_builder.config_resource_types.items():
                if self._all_in_one_config:
                    res = sdk2.Resource.find(type='BEGEMOT_FAST_BUILD_FRESH_CONFIG', task_id=self._task.id).first()
                else:
                    res = sdk2.Resource.find(type=res_type, task_id=self._task.id).first()
                self._ctx.begemot_shard_resource_ids[shard] = res.id

            if self._all_in_one_config:
                self._task.Parameters.output_resources = {'BEGEMOT_FAST_BUILD_FRESH_CONFIG': res.id}
            else:
                self._task.Parameters.output_resources = {res_type: res.id}

        if not self._use_full_shard_build:
            try:
                if cypress_config:
                    self.fast_builder.extend_existing_cypress_paths_ttl(cypress_config)
            except Exception as e:
                logging.debug("Warn: failed to update existing resources cypress paths ttl: {}".format(e))

            logging.info('Len on shards {}'.format(len(self._shard_names)))
            for shard in self._shard_names:
                existing_cypress_paths = self.fast_builder.get_existing_cypress_paths() if cypress_config else []
                cypress_shard_file_path = self.update_cypress_shard(
                    pack_path,
                    shard,
                    is_fresh=self._build_fresh,
                    existing_paths=existing_cypress_paths,
                    rules_filter=self.fast_builder.get_rules_for_shard(shard),
                    retries=3,
                    updater_id=self._shard_updater,
                    test_updater=self._test_shard_updater,
                    fake_update=cypress_config is None,
                )
                if cypress_config:
                    self.fast_builder.update_cypress_paths(cypress_shard_file_path)

        if self._use_full_shard_build and cypress_config:
            cypress_shard_file_path = self.update_cypress_shard(
                pack_path,
                self._shard_name,
                existing_paths=existing_cypress_paths,
                updater_id=self._shard_updater,
                test_updater=self._test_shard_updater,
            )

        if self._use_fast_build:
            self.fast_builder.update_cache()

        if self._use_full_shard_build:
            self._create_version_file(pack_path)

        if self._use_full_shard_build:
            self._ctx.rules_size_mb = {
                name: int(subprocess.check_output(['du', '-smL', os.path.join(pack_path, name)]).split('\t')[0])
                for name in os.listdir(pack_path)
            }
        else:
            self._ctx.rules_size_mb = self.fast_builder.get_rules_size_mb()

        if self._use_fast_build:
            # For correct deserialization
            self.fast_builder = None

        # The limits are actually set in sandbox/projects/websearch/begemot/__init__.py
        actual_size_gb = int(sum(x[1] for x in self._ctx.rules_size_mb.items())) >> 10
        if not self._build_fresh and self._shard_size_gb(self._shard_name) + 2 < actual_size_gb:
            self._task.server.notification(
                recipients=['gluk47', 'denis28', 'bgsavko'],
                subject='BUILD_BEGEMOT_DATA: HDD limit for %s exceeded' % self._shard_name,
                body='<br />\n'.join([
                    'Hello, fellow begemot developer.', '',
                    'One of your shards (<a href="https://a.yandex-team.ru/arc/trunk/arcadia/search/begemot/data/{name}">{name}</a>) '
                    'has grown up and now exceeds its HDD limit.'.format(name=self._shard_name),
                    'You should go to <a href="https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/websearch/begemot/common/__init__.py">begemot/common/__init__.py</a>,',
                    'find this shard inside __init__ of "class BegemotAllServices" and set shard_size_gb there to %s' % (actual_size_gb + 5), '',
                    'This email will be generated by all subsequent tasks until the limit is fixed.',
                    'Please, fix the limit as soon as possible.',
                    '--',
                    'This letter is generated from a sandbox task <a href="{url}">{id}</a>'.format(url=get_task_link(self._task.id), id=self.id),
                    'To unsubscribe, please, modify the source of this task.'
                ]),
                transport=Transport.EMAIL
            )

    def _shard_size_gb(self, shard_name):
        return Begemots[shard_name].shard_size_gb
