# -*- coding: utf-8 -*-

import datetime
import logging
import os
import time


from sandbox import sdk2
from sandbox.common.types.resource import State
from sandbox.common.errors import ResourceNotFound, TaskFailure
from sandbox.projects.websearch.begemot.resources import BEGEMOT_CYPRESS_SHARD


class CypressShardUpdater(object):
    def __init__(self, cypress_cache, yt_proxy, yav_token, task, parent_id):
        self._cypress_cache = cypress_cache
        self._yt_proxy = yt_proxy
        self._yav_token = yav_token
        self._task = task
        self._parent_id = parent_id

    __logger = logging.getLogger('SHARD_UPDATER')
    __logger.setLevel(logging.DEBUG)

    def filter_rules(self, filename, rules_filter):
        result = []
        with open(filename, "r") as f:
            for line in f.readlines():
                line = line.strip("\n")
                if os.path.basename(line) in rules_filter:
                    logging.debug("OK for path {}".format(line))
                    result.append(line)
                else:
                    logging.debug("SKIP path {}".format(line))

        with open(filename, "w") as f:
            f.write('\n'.join(result) + '\n')

    # Updates shard in Cypress and creates resource with Cypress paths
    # Also saves the same file into parent tasks Cypress shard resource, if any
    def update_cypress_shard(self, local_shard_path, shard_name, is_fresh=False, existing_paths=(), rules_filter=None, retries=1, updater_id=None, test_updater=False, fake_update=False):
        updater_success = False
        result_file = '{}.txt'.format(shard_name)
        while not updater_success and retries > 0 and not fake_update:
            self.run_shard_updater(updater_id, result_file, local_shard_path, shard_name, is_fresh, ttl_days=30)
            updater_success = os.path.isfile(result_file)
            retries -= 1
            if not updater_success and retries > 0:
                time.sleep(15)

        if not updater_success and test_updater:
            raise TaskFailure("Shard updater failed")

        resource = BEGEMOT_CYPRESS_SHARD(self._task, shard_name, result_file)
        resource.shard_name = shard_name
        resource.is_fresh = is_fresh
        resource.is_broken = not updater_success

        if not updater_success:
            self.__logger.error('Shard updater failed: %s not found, creating empty file', result_file)
            open(result_file, 'a').close()

        with open(result_file, 'a') as file:
            for path in existing_paths:
                file.write(path + '\n')

        if rules_filter is not None:
            logging.debug("Filtering rules for shard {}".format(shard_name))
            self.filter_rules(result_file, rules_filter)

        parent_resource = self.get_parent_resource(shard_name, is_fresh)
        if parent_resource:
            self._task.save_parent_task_resource(result_file, parent_resource)
            resource.is_broken = not updater_success

        return result_file

    def get_parent_resource(self, shard_name, is_fresh):
        if self._parent_id == -1:
            self.__logger.info('Parent task not found')
            return None
        resource = BEGEMOT_CYPRESS_SHARD.find(
            task_id=self._parent_id,
            status=State.NOT_READY,
            attrs={
                'shard_name': shard_name,
                'is_fresh': is_fresh,
            }
        ).first()

        if resource:
            self.__logger.info('Found parent resource, id: %d', resource.id)
            return resource
        else:
            self.__logger.info('Parent resource not found')

    def get_cypress_cache_config(self):
        if not self._cypress_cache:
            self.__logger.info('Updating shard in Cypress is disabled')
            return None

        # try:
        #     tokens = self._yav_token.data()
        #     token = tokens['yav_token']
        # except Exception as e:
        #     raise RuntimeError('Failed to get yt token: {}'.format(e.message))

        return {
            'path': self._cypress_cache,
            'proxy': self._yt_proxy,
            'token': self._yav_token,
        }

    def run_shard_updater(self, updater_id, result_file, local_shard_path, shard_name, is_fresh, ttl_days):
        config = self.get_cypress_cache_config()
        if config is None:
            return
        env = os.environ.copy()
        cypress_cache = config['path']
        env['YT_TOKEN'] = config['token']
        env['YT_PROXY'] = config['proxy']

        self.__logger.info(
            'Updating shard, cypress_cache: %s, shard_name: %s, is_fresh: %s, %ttl_days: %d',
            cypress_cache, shard_name, is_fresh, ttl_days,
        )
        if updater_id is None:
            shard_updater_resource = sdk2.Resource.find(
                type='BEGEMOT_SHARD_UPDATER',
                attrs={
                    'released': 'stable',
                }
            ).first()
            if shard_updater_resource is None:
                raise ResourceNotFound('BEGEMOT_SHARD_UPDATER resource not found')
            updater_id = shard_updater_resource.id
        else:
            shard_updater_resource = sdk2.Resource[updater_id]

        self.__logger.info("Using shard updater #{}".format(updater_id))

        shard_updater = str(sdk2.ResourceData(shard_updater_resource).path)
        expiration_date = datetime.datetime.utcnow() + datetime.timedelta(days=ttl_days)
        args = [
            shard_updater,
            '--data', local_shard_path,
            '--result-paths-file', result_file,
            '--cache-path', cypress_cache,
            '--ttl', expiration_date.isoformat(),
            '--jobs', '20',
            '--debug',
        ]
        try:
            with sdk2.helpers.ProcessLog(self._task, logger='shard_updater') as pl:
                sdk2.helpers.subprocess.run(args, env=env, stdout=pl.stdout, stderr=pl.stderr)
        except BaseException as e:
            self.__logger.error('Shard updater process failed: ' + e.message)

    def on_execute(self):
        raise NotImplementedError('This task is abstract')
