# -*- coding: utf-8 -*-
import datetime
import logging
import os
import shutil
import subprocess
import traceback

import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
import sandbox.projects.common.wizard.utils as w_utils
import sandbox.sandboxsdk.process as sandboxsdk_process

from sandbox.common.errors import TaskFailure
from sandbox.common.rest import Client
from sandbox.common.utils import get_task_link
from sandbox.projects.common import utils
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.websearch.begemot import AllBegemotServices
from sandbox.projects.websearch.begemot.common import Begemots
from sandbox.projects.websearch.begemot.common.fast_build import FastBuilder
from sandbox.projects.websearch.begemot.resources import BEGEMOT_SHARD_UPDATER
from sandbox.projects.websearch.begemot.tasks.BegemotYT.common import CypressShardUpdater, CypressCache
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxBoolGroupParameter, SandboxStringParameter, ResourceSelector, SandboxIntegerParameter
from sandbox.sandboxsdk.svn import Arcadia


class ShardName(SandboxBoolGroupParameter):
    name = "ShardName"
    description = "Shard names to build. If you don't build fresh, select one shard"
    required = True
    choices = sorted([(shard_name, shard_name) for shard_name in Begemots.keys()])


class BuildFresh(SandboxBoolParameter):
    name = "BuildFresh"
    description = "Build fresh instead of slow data"
    default_value = False


class AllInOneConfig(SandboxBoolParameter):
    name = "AllInOneConfig"
    description = "Build only one fast build config with all rules (actual for fresh)"
    default_value = False


class BuildTestShard(SandboxBoolParameter):
    name = "BuildTestShard"
    description = "Apply testpatches to data"
    default_value = False


class UseFullShardBuild(SandboxBoolParameter):
    """
    Use default build (always rebuilds whole shard).
    """
    name = "UseFullShardBuild"
    description = "Full shard build"
    default_value = False


class UseFastBuild(SandboxBoolParameter):
    """
    Uses build cache for each separate rule of given shard,
    if rule with same hash is already built, previous
    result resource will be used, otherwise new resource is
    created.
    Produces JSON config with array of all rule resources
    for given shard.
    """
    name = "UseFastBuild"
    description = "Fast build"
    default_value = True


class ForceRebuild(SandboxBoolParameter):
    name = "ForceRebuild"
    description = "Rebuild all rules in fast build mode"
    default_value = False


class FastBuildCache(SandboxStringParameter):
    name = 'FastBuildCache'
    description = 'Fast build cache path in yt'
    default_value = '//home/search-runtime/begemot-fast-build'
    required = False


class ShardUpdater(ResourceSelector):
    name = 'ShardUpdater'
    description = 'Shard updater for YT (if empty, use last released)'
    resource_type = BEGEMOT_SHARD_UPDATER
    required = False


class TestShardUpdater(SandboxBoolParameter):
    name = "TestShardUpdater"
    description = "Fail task if shard updater failed"
    default_value = False


class SeparateBuild(SandboxBoolParameter):
    name = "SeparateFreshBuild"
    description = "Build every fresh dir"
    default_value = False


class ThreadsCount(SandboxIntegerParameter):
    name = "ThreadsCountForSeparateFreshBuild"
    description = "Thread count"
    default_value = 1


class BuildBegemotData(YaMakeTask, CypressShardUpdater):

    type = 'BUILD_BEGEMOT_DATA'

    input_parameters = [
        ShardName,
        BuildFresh,
        AllInOneConfig,
        BuildTestShard,
        UseFullShardBuild,
        UseFastBuild,
        ForceRebuild,
        FastBuildCache,
        ShardUpdater,
        TestShardUpdater,
        SeparateBuild,
        ThreadsCount,
        build_params.ArcadiaUrl,
        build_params.ArcadiaPatch,
        build_params.BuildSystem,
        build_params.BuildType,
        build_params.UseArcadiaApiFuse,
        build_params.YaTimeout,
    ] + CypressShardUpdater.input_parameters

    environment = (
        environments.PipEnvironment("yandex-yt", version='0.10.8'),
    ) + YaMakeTask.environment + CypressShardUpdater.environment

    cores = 24

    # Here we need to list all possible tags and later refine them in on_enqueue.
    # That's the only possible way in SDK1 (and YaMakeTask is currently on SDK1).
    client_tags = w_utils.ALL_SANDBOX_HOSTS_TAGS

    def _shard_size_gb(self, shard_name):
        return Begemots[shard_name].shard_size_gb

    def on_enqueue(self):
        if not self.ctx.get(UseFullShardBuild.name) and not self.ctx.get(UseFastBuild.name):
            raise TaskFailure("Neither 'Full' nor 'Fast' build is selected.")

        if self.ctx.get(BuildFresh.name) and self.ctx.get(UseFullShardBuild):
            raise TaskFailure("Can't run full build for fresh. Select fast build or use WIZARD_RUNTIME_BUILD task")

        shard_names = self.ctx[ShardName.name].split()
        if not len(shard_names):
            raise TaskFailure("Select shards to build")

        if not self.ctx.get(BuildFresh.name) and len(shard_names) > 1:
            raise TaskFailure("Select exactly one shard for slow data build")
        shard_name = shard_names[0]

        # The limits are actually set in sandbox/projects/websearch/begemot/__init__.py
        self.execution_space = max(self._shard_size_gb(shard_name) * 1024 * 3, 10240) if not self.ctx.get(BuildFresh.name) else 50 * 1024  # ×3 : caches multiply HDD usage
        self.ctx['kill_timeout'] = (3 * (self._shard_size_gb(shard_name) > 150) + 2) * 60 * 60  # 3 hours more for huge shards
        if shard_name.startswith('Spellchecker'):
            self.execution_space += self._shard_size_gb(shard_name) * 1024
            self.ctx['kill_timeout'] += 2.5 * 60 * 60
        self.ctx[consts.CHECKOUT] = False if utils.get_or_default(self.ctx, build_params.ArcadiaPatch) else True
        Client().task[self.id].tags(shard_names)  # BEGEMOT-1367
        w_utils.setup_hosts_sdk1(self)
        YaMakeTask.on_enqueue(self)

    def get_resources(self):
        if not self.ctx.get(UseFullShardBuild.name):
            return {}

        shard_name = self.ctx.get(ShardName.name)
        resource = {
            'description': 'Begemot shard ' + shard_name,
            'resource_path': shard_name,
            'resource_type': AllBegemotServices.Service[shard_name].data_resource_type_test,
        }
        return {'project': resource}

    def pre_build(self, source_dir):
        if not self.ctx.get(UseFastBuild.name):
            return

        do_rebuild = self.ctx.get(ForceRebuild.name)
        testenv_db = self.ctx.get('testenv_database')
        if testenv_db is not None and 'ws-begemot' in testenv_db and 'trunk' not in testenv_db:
            if self.ctx.get(BuildFresh.name):
                do_rebuild = True # Always rebuild fresh in release databases (BEGEMOT-2307)

        try:
            with_cypress = self.ctx.get(CypressCache.name) is not None and len(self.ctx.get(CypressCache.name))
            self.fast_builder = FastBuilder(
                self.ctx.get(FastBuildCache.name),
                source_dir,
                self.ctx.get(ShardName.name),
                self.ctx.get(BuildTestShard.name),
                fresh=self.ctx.get(BuildFresh.name),
                one_config=self.ctx.get(AllInOneConfig.name),
                rebuild=do_rebuild,
                additional_flags=self.get_build_def_flags(),
                with_cypress=with_cypress,
                checkout=(not utils.get_or_default(self.ctx, build_params.UseArcadiaApiFuse)),
                separate_build=self.ctx.get(SeparateBuild.name),
                threads=self.ctx.get(ThreadsCount.name)
            )
            self.fast_builder.make_checkout(self.ctx.get(build_params.ArcadiaUrl.name), clear_build=self.ctx.get(consts.CLEAR_BUILD_KEY))
            rules = self.fast_builder.resolve_rules_to_build()
            self.ctx['rules_list'] = rules
            if not self.ctx.get(UseFullShardBuild.name):
                self.fast_builder.create_targets()
        except Exception as e:
            logging.error("Exception in FastBuilder: %s", traceback.print_exc())
            raise TaskFailure("PreBuild Failed: %s" % e)


    def get_targets(self):
        if not self.ctx.get(UseFullShardBuild.name):
            return self.fast_builder.get_targets()

        build_path = os.path.join(Begemots.shards_path, self.ctx.get(ShardName.name))
        logging.debug('Build path is : %s', build_path)
        return [build_path]

    def get_arts(self):
        if not self.ctx.get(UseFullShardBuild.name):
            return []

        shard_name = self.ctx.get(ShardName.name)
        return [
            {
                'path': os.path.join(
                    Begemots.shards_path, shard_name, 'search', 'wizard', 'data', 'wizard', '*'
                ),
                'dest': shard_name,
            }
        ]

    def get_build_def_flags(self):
        if self.ctx.get(ShardName.name) == "Geo":
            return "-DGEO_WIZARD_SHARD"
        return ""

    def _apply_testpatches(self, path):
        os.system("chmod -R a+w " + path)
        for curr, dirs, files in os.walk(path):
            for f in files:
                origin_file = os.path.join(curr, f)
                patch_file = origin_file + ".testpatch"
                logging.debug("Try patch %s", origin_file)
                if os.path.exists(patch_file):
                    logging.info("Found testpatch for file %s", f)
                    sandboxsdk_process.run_process(["patch", origin_file, patch_file], wait=True, check=True)

    def _last_change(self, default):
        url = self.ctx.get(build_params.ArcadiaUrl.name)
        if url.startswith(Arcadia.ARCADIA_HG_SCHEME):
            return default
        info = Arcadia.info(url)
        if not info:
            return default
        return info.get('commit_revision', default)

    def _make_version_info(self):
        revision = Arcadia.get_revision(self.ctx.get(build_params.ArcadiaUrl.name))
        return {
            "GenerationTime": '"{}"'.format(datetime.datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ')),
            "Revision": self._last_change(revision),
            "Task": self.id,
            "ShardName": '"{}"'.format(self.ctx.get(ShardName.name)),
        }

    def _create_version_file(self, path):
        version_info = self._make_version_info()
        with open(os.path.join(path, 'version.pb.txt'), 'w') as version:
            for key, value in version_info.items():
                version.write('{}: {}\n'.format(key, value))

    def _update_parent_resource(self, resource_type):
        if not self.ctx.get(resource_type):
            return
        self.set_info("Updating parent resource. Why? Idk...")
        w_utils.shrink_resources_ttl(self, resource_type)
        for resource in channel.sandbox.list_resources(resource_type=resource_type, task_id=self.id, limit=1):
            self.save_parent_task_resource(resource.path, self.ctx[resource_type])
            for attr in ['data_size_kb', 'version']:
                try:
                    parent_resource = channel.sandbox.get_resource(self.ctx[resource_type])
                    attr_val = channel.sandbox.get_resource_attribute(resource, attr)
                    channel.sandbox.set_resource_attribute(parent_resource, attr, attr_val)
                except Exception as e:
                    logging.debug("Failed to copy {} attribute. {}".format(attr, e))

    def post_build(self, source_dir, output_dir, pack_dir):
        shard_names = self.ctx.get(ShardName.name).split()
        pack_path = os.path.join(pack_dir, shard_names[0]) if not self.ctx.get(BuildFresh.name) else os.path.join(pack_dir, "rules")

        if not self.ctx.get(UseFullShardBuild.name):
            self.fast_builder.pack_rules(output_dir)

        if self.ctx.get(BuildTestShard.name):
            self._apply_testpatches(pack_path)

        cypress_config = self.get_cypress_cache_config()
        existing_cypress_paths = []

        if self.ctx.get(UseFastBuild.name):
            self.fast_builder.get_failed_rules_logs(self)
            if self.ctx.get(UseFullShardBuild.name):
                self.fast_builder.make_own_pack()
            self.fast_builder.make_resources(self, self._make_version_info())
            self.ctx["begemot_shard_resource_ids"] = {}
            for shard, res_type in self.fast_builder.config_resource_types.items():
                if self.ctx.get(AllInOneConfig.name):
                    res = channel.sandbox.list_resources(resource_type='BEGEMOT_FAST_BUILD_FRESH_CONFIG', task_id=self.id)[0]
                else:
                    res = channel.sandbox.list_resources(resource_type=res_type, task_id=self.id)[0]
                self.ctx["begemot_shard_resource_ids"][shard] = res.id

        if not self.ctx.get(UseFullShardBuild.name):
            try:
                if cypress_config:
                    self.fast_builder.extend_existing_cypress_paths_ttl(cypress_config)
            except Exception as e:
                logging.debug("Warn: failed to update existing resources cypress paths ttl")

            for shard in shard_names:
                existing_cypress_paths = self.fast_builder.get_existing_cypress_paths() if cypress_config else []
                cypress_shard_file_path = self.update_cypress_shard(
                    pack_path,
                    shard,
                    is_fresh=self.ctx.get(BuildFresh.name),
                    existing_paths=existing_cypress_paths,
                    rules_filter=self.fast_builder.get_rules_for_shard(shard),
                    retries=3,
                    updater_id=self.ctx.get(ShardUpdater.name, None),
                    test_updater=self.ctx.get(TestShardUpdater.name),
                    fake_update=cypress_config is None,
                )
                if cypress_config:
                    self.fast_builder.update_cypress_paths(cypress_shard_file_path)

        if self.ctx.get(UseFullShardBuild.name) and cypress_config:
            cypress_shard_file_path = self.update_cypress_shard(
                pack_path,
                self.ctx.get(ShardName.name),
                existing_paths=existing_cypress_paths,
                updater_id=self.ctx.get(ShardUpdater.name, None),
                test_updater=self.ctx.get(TestShardUpdater.name),
            )

        if self.ctx.get(UseFastBuild.name):
            self.fast_builder.update_cache()
            for shard in shard_names:
                self._update_parent_resource(self.fast_builder.config_resource_types[shard])
            if self.ctx.get(AllInOneConfig.name):
                self._update_parent_resource('BEGEMOT_FAST_BUILD_FRESH_CONFIG')

        if self.ctx.get(UseFullShardBuild.name):
            self._create_version_file(pack_path)
            self._update_parent_resource(Begemots[self.ctx[ShardName.name]].shard_resource_name)

        w_utils.delete_raw_build_output(self.id)

        if self.ctx.get(UseFullShardBuild.name):
            self.ctx['rules_size_mb'] = {
                name: int(subprocess.check_output(['du', '-smL', os.path.join(pack_path, name)]).split('\t')[0])
                for name in os.listdir(pack_path)
            }
        else:
            self.ctx['rules_size_mb'] = self.fast_builder.get_rules_size_mb()

        if self.ctx.get(UseFastBuild.name):
            # For correct deserialization
            self.fast_builder = None


        # The limits are actually set in sandbox/projects/websearch/begemot/__init__.py
        actual_size_gb = int(sum(x[1] for x in self.ctx.get('rules_size_mb', {}).items())) >> 10
        if not self.ctx.get(BuildFresh.name) and self._shard_size_gb(self.ctx.get(ShardName.name)) + 2 < actual_size_gb:
            channel.sandbox.send_email(
                mail_to=['gluk47', 'ageraab', 'denis28'],
                mail_cc=[],
                mail_subject='BUILD_BEGEMOT_DATA: HDD limit for %s exceeded' % self.ctx.get(ShardName.name),
                mail_body='<br />\n'.join([
                    'Hello, fellow begemot developer.', '',
                    'One of your shards (<a href="https://a.yandex-team.ru/arc/trunk/arcadia/search/begemot/data/{name}">{name}</a>) '
                    'has grown up and now exceeds its HDD limit.'.format(name=self.ctx.get(ShardName.name)),
                    'You should go to <a href="https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/websearch/begemot/common/__init__.py">begemot/common/__init__.py</a>,',
                    'find this shard inside __init__ of "class BegemotAllServices" and set shard_size_gb there to %s' % (actual_size_gb + 5), '',
                    'This email will be generated by all subsequent tasks until the limit is fixed.',
                    'Please, fix the limit as soon as possible.',
                    '--',
                    'This letter is generated from a sandbox task <a href="{url}">{id}</a>'.format(url=get_task_link(self.id), id=self.id),
                    'To unsubscribe, please, modify the source of this task.'
                ]),
                content_type='text/html'
            )

    @property
    def footer(self):
        sizes = sorted(self.ctx.get('rules_size_mb', {}).items(), key=lambda x: (-x[1], x[0]))
        if not sizes:
            return
        sizes.insert(0, ['_Total', sum(s[1] for s in sizes)])
        return {
            '<h4>Rule sizes in MB</h4>': {
                'header': [{'key': k, 'title': k} for k in ['Rule', 'MB']],
                'body': {
                    'Rule': [x[0] for x in sizes],
                    'MB': [x[1] for x in sizes],
                }
            }
        }


__Task__ = BuildBegemotData
