# -*- coding: utf-8 -*-
import logging
import time

import sandbox.common.types.client as ctc
import sandbox.projects.common.build.parameters as params
import sandbox.projects.common.arcadia.sdk as sdk
import sandbox.projects.common.nanny.nanny as nanny
import sandbox.projects.common.vcs.aapi as aapi
import sandbox.projects.release_machine.mixins.build as rm_build_mixin
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.configs.begemot as begemot_cfg
import sandbox.projects.common.search.gdb as gdb
import sandbox.projects.common.error_handlers as eh
import sandbox.sandboxsdk.paths as paths
import sandbox.sdk2 as sdk2


import os
import sandbox.projects.websearch.begemot.resources as br

STRING = "string"
RESOURCE = "resource"
LOCAL_RELEASE_DIR = "release"

build_fields = {
    "begemot_logger": br.BEGEMOT_LOGGER_EXECUTABLE,
}


class BuildBegemotLogger(rm_build_mixin.ComponentReleaseTemplate, nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Begemot Logger Build
    """

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group(params.BASE_BUILD_GROUP_NAME) as bb_group:
            checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)
            clear_build = sdk2.parameters.Bool('Clear build', default=True)

        with sdk2.parameters.Group("Dependency banner") as db_group:
            banned_dependencies = sdk2.parameters.String("Fail build if it requires specified paths (comma-separated)")
            whitelisted_dependencies = sdk2.parameters.String(
                "Ignore banned_dependencies for specified sub-paths (comma-separated)",
            )
            check_dependencies = sdk2.parameters.Bool("Check build dependencies")
            ban_ut_dependencies = sdk2.parameters.Bool("Ban unit tests as dependencies (SEARCH-2944)")

        with sdk2.parameters.Group(params.BUILD_SYSTEM_GROUP_NAME) as bs_group:
            with sdk2.parameters.String('Build type', required=True) as build_type:
                build_type.values.release = build_type.Value('Release', default=True)
                build_type.values.debug = 'Debug'
                build_type.values.profile = 'Profile'
                build_type.values.coverage = 'Coverage'
                build_type.values.relwithdebinfo = 'Release with debug info'
                build_type.values.valgrind = 'Valgrind (debug)'
                build_type.values.valgrind_release = 'Valgrind (release) (only for build with ya)'

            with sdk2.parameters.String("Build system") as build_system:
                build_system.values.ya = build_system.Value("Ya", default=True)
                build_system.values.ya_force = "Ya force"
                build_system.values.semi_distbuild = "Semi distbuild"
                build_system.values.distbuild = "distbuild"

            definition_flags = sdk2.parameters.String(
                'Definition flags (for example "-Dkey1=val1 ... -DkeyN=valN")', required=False
            )

            target_platform_flags = sdk2.parameters.String(
                'Target platform flags (only for cross-compilation)', required=False
            )
            musl = sdk2.parameters.Bool("Use --musl option", default=True)
            thinlto = sdk2.parameters.Bool("Build with ThinLTO", default=False)
            # Checkout parameter will not be used in case use_fuse=True
            checkout = sdk2.parameters.Bool("Set checkout option for build", default=True)
        with sdk2.parameters.Group("Projects") as p_group:
            with sdk2.parameters.CheckGroup('Build search projects') as search_projects:
                for field in build_fields:
                    search_projects.values[field] = search_projects.Value('Build {}'.format(field), checked=True)

    class Requirements(sdk2.Task.Requirements):
        disk_space = 40 * 1024
        client_tags = ctc.Tag.SSD

    def on_enqueue(self):
        # dependency_banner.create_resources(self)
        setattr(self.Context, rm_const.COMPONENT_CTX_KEY, begemot_cfg.BegemotCfg.name)
        for ft in self.get_targets():
            if getattr(self.Parameters, "{}_res_id".format(ft), None):
                continue
            ft_name = getattr(build_fields[ft], 'arcadia_build_name', build_fields[ft].arcadia_build_path)
            build_fields[ft](self, ft_name, os.path.join(LOCAL_RELEASE_DIR, build_fields[ft].arcadia_build_path))

    def get_targets(self):
        if isinstance(self.Parameters.search_projects, list):
            return self.Parameters.search_projects
        return self.Parameters.search_projects.split(' ')

    def on_execute(self):
        gdb.append_to_stable_only(self)

        arcadia_src = str(self.Parameters.checkout_arcadia_from_url).strip("/")

        target_dirs = [os.path.dirname(build_fields[rk].arcadia_build_path) for rk in self.get_targets()]

        sdk2.path.Path(os.path.abspath(LOCAL_RELEASE_DIR)).mkdir(mode=0o755, exist_ok=True)

        def_flags = sdk.parse_flags(self.Parameters.definition_flags)
        def_flags["SANDBOX_TASK_ID"] = self.id

        self._fuse_build(arcadia_src, def_flags, target_dirs)

    def _time_info(self, time_ctx, time_val):
        setattr(self.Context, time_ctx, time_val)
        self.set_info("{0} is {1} sec.".format(time_ctx, time_val))

    def _do_build(self, arcadia_src, target_dirs, def_flags, checkout):
        build_start = int(time.time())
        sdk.do_build(
            self.Parameters.build_system,
            arcadia_src,
            target_dirs,
            self.Parameters.build_type,
            clear_build=self.Parameters.clear_build,
            results_dir=LOCAL_RELEASE_DIR,
            def_flags=def_flags,
            target_platform_flags=self.Parameters.target_platform_flags,
            checkout=checkout,
            musl=self.Parameters.musl,
            thinlto=self.Parameters.thinlto,
            # patch=self.Parameters.arcadia_patch,
        )
        self._time_info("build_time", int(time.time()) - build_start)

    def _fuse_build(self, arcadia_src, def_flags, target_dirs):
        path, revision = aapi.ArcadiaApi.extract_path_and_revision(arcadia_src)
        eh.ensure(path, "Url {} is not present in arcadia-api".format(arcadia_src))

        if revision is None:
            revision = aapi.ArcadiaApi.svn_head()

        if not aapi.ArcadiaApi.has_svn_revision(revision):
            logging.info("Revision {} is not present in arcadia-api. Waiting 180s.", revision)
            eh.ensure(
                aapi.ArcadiaApi.wait_svn_revision(revision, 180),
                "Revision {} is not present in arcadia-api. Waited 180s.".format(revision)
            )
        logging.info("Release revision is %s.", revision)

        with aapi.ArcadiaApi.mount_svn_path(path, revision=revision) as source_dir_origin:
            logging.info("Arcadia source dir: %s", source_dir_origin)

            arc_dir = paths.make_folder("arc")
            source_dir = os.path.join(arc_dir, 'arcadia')
            self._ensure_link(source_dir_origin, source_dir)
            self._do_build(source_dir, target_dirs, def_flags, False)

    @staticmethod
    def _ensure_link(path, link):
        if os.path.exists(link):
            eh.ensure(os.path.islink(link), "Cannot create symlink, non-link already exists: {}".format(link))
            os.remove(link)
        os.symlink(path, link)
