# -*- coding: utf-8 -*-
import logging
import os
import sys
import re
from datetime import datetime

import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.wizard.parameters as w_params
import sandbox.projects.common.wizard.utils as w_utils

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common import constants
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.websearch.begemot.tasks.BegemotYT.common import CypressShardUpdater


class BuildWizardData(YaMakeTask, CypressShardUpdater):

    type = 'BUILD_WIZARD_DATA'
    execution_space = 160 * 1024  # sometimes Arcadia checkout size is added to the task HDD usage
    # Here we need to list all possible tags and later refine them in on_enqueue.
    # That's the only possible way in SDK1 (and YaMakeTask is currently on SDK1).
    client_tags = w_utils.ALL_SANDBOX_HOSTS_TAGS
    cores = 24

    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.ArcadiaPatch,
        build_params.BuildType,
        w_params.BuildForGeosearch
    ] + CypressShardUpdater.input_parameters

    def pre_execute(self):
        if not self.ctx.get(build_params.ArcadiaPatch.name):
            self.ctx[constants.CHECKOUT] = True
        YaMakeTask.pre_execute(self)

    def get_resources(self):
        resources = {
            'description': 'Build wizard data from arcadia',
            'resource_path': os.path.basename(self.ctx['wizard_shard_name']),
            'resource_type': resource_types.WIZARD_SHARD,
        }
        return {'project': resources}

    def get_targets(self):
        return [self.ctx['wizard_data_path']]

    def get_arts(self):
        return [{
            'path': self.ctx['wizard_data_path'],
            'dest': self.ctx['wizard_shard_name'],
        }]

    def get_build_def_flags(self):
        if self.ctx.get(w_params.BuildForGeosearch.name):
            return "-DGEO_WIZARD_SHARD"
        return ""

    # TODO: убрать дублирование и заспользовать таск сборки конфига.  Сделать методы static
    def build_config(self, data_path):
        config_dir = os.path.join(data_path, 'conf')

        config_generator = os.path.join(config_dir, 'config.py')
        params = ['--all']
        process.run_process(
            [sys.executable, config_generator] + params + [config_dir],
            log_prefix='config_generation',
            wait=True,
            check=True
        )

        for thesaurus_config_dir in [os.path.join(data_path, thes_dir) for thes_dir in ('thesaurus', 'Thesaurus')]:
            generator = os.path.join(thesaurus_config_dir, 'generate_geosearch_config.py')
            if os.path.exists(generator):
                source_config = os.path.join(thesaurus_config_dir, 'thesaurus.cfg')
                dest_config = os.path.join(thesaurus_config_dir, 'thesaurus_geo.cfg')
                if os.path.isfile(dest_config):
                    return
                with open(source_config) as fin, open(dest_config, 'w') as fout:
                    process.run_process(
                        [sys.executable, generator],
                        stdin=fin,
                        stdout=fout,
                        wait=True,
                        shell=True,
                        log_prefix='thesaurus_geo_cfg_gen'
                    )

    def patch_geoaddr_config(self, data_path):
        if self.ctx.get(w_params.BuildForGeosearch.name):
            for dirname in ('geoaddr', 'GeoAddr'):
                for filename in ('geoaddr.cfg', 'config.pb.txt'):
                    cfg_path = os.path.join(data_path, dirname, filename)
                    if not os.path.isfile(cfg_path):
                        continue
                    with open(cfg_path, 'r') as fin:
                        content = re.sub(r'#(ActivateUSA|ActiverFrance|AktivierenDeutschland)', r'\1', fin.read())
                        content = re.sub(r'(ActivateUSA|ActiverFrance|AktivierenDeutschland): *false', r'\1: true', content)
                    with open(cfg_path, 'w') as fout:
                        print >>fout, content
                    return

    def make_version_file(self, result_path):
        # ap_arcadia_revision is posted by BaseArcadiaProjectTask
        with file(os.path.join(result_path, "version.info"), "wb") as versionFile:
            versionFile.write('{"GenerationTime":"%s","Revision":%s,"Task":%s}\n' % (
                datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ'),  self.ctx.get('ap_arcadia_revision'), self.id))

    def post_build(self, source_dir, output_dir, pack_dir):
        resources = channel.sandbox.list_resources(resource_type=resource_types.WIZARD_SHARD, task_id=self.id)
        for resource in resources:
            try:
                # В продакшене конфиг берется из данных
                self.build_config(resource.path)
                # В геопоиске используется больше языков в правиле GeoAddr
                self.patch_geoaddr_config(resource.path)
            except Exception as x:
                logging.error("Failed to generate config: %s" % x)
                pass
            # version.info
            self.make_version_file(resource.path)
            self.update_cypress_shard(resource.path, self.ctx['wizard_shard_name'])
            # Проверям передавал ли родительский таск id ресурса, который нужно заполнить
            if self.ctx.get(resource_types.WIZARD_SHARD.name):
                self.save_parent_task_resource(resource.path, self.ctx[resource_types.WIZARD_SHARD.name])

        w_utils.shrink_resources_ttl(self, resource_types.WIZARD_SHARD)
        w_utils.delete_raw_build_output(self.id)

    def on_enqueue(self):
        # Distbuild is completely useless for downloading data from sandbox
        self.ctx[constants.BUILD_SYSTEM_KEY] = constants.YA_MAKE_FORCE_BUILD_SYSTEM

        w_utils.setup_hosts_sdk1(self)

        # Чтобы проинициализировать шард, нужно ему дать уникальное имя
        self.ctx['wizard_shard_name'] = 'WIZARD_SHARD'
        self.ctx['wizard_data_path'] = resource_types.WIZARD_SHARD.arcadia_build_path
        YaMakeTask.on_enqueue(self)


__Task__ = BuildWizardData
