from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import requests_wrapper
from sandbox.projects.websearch.begemot.common import BegemotAllServices
import logging
import time


class CheckBegemotSignals(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        signals_limit = sdk2.parameters.Integer(
            "Golovan signals limit for begemot worker",
            default=3000,
            description='max_sig_count value from https://bb.yandex-team.ru/projects/SEARCH_INFRA/repos/yasm/browse/CONF/agent.begemot.conf'
        )

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024  # 1 Gb

    class Context(sdk2.Task.Context):
        workers_info = ""

    @property
    def footer(self):
        return "<h3>Begemot signals to Golovan</h3>" + "<ul>" + self.Context.workers_info + "</ul>"

    def get_num_signals(self, golovan_prj):
        api_params = {
            "itype": "begemot",
            "ctype": ["prod", "prestable", "priemka"],
            "hosts": "ASEARCH",
            "signal_pattern": "begemot-WORKER",
            "prj": golovan_prj,
        }
        for attempt in range(16):
            try:
                time.sleep(attempt)
                response = requests_wrapper.get("https://yasm.yandex-team.ru/metainfo/signals/", params=api_params)
                response.raise_for_status()
                return response.json()['response']['total']
            except Exception as x:
                logging.debug(response.text)
                logging.debug(x)

        self.set_info(response.text)
        return response.json()['response']['total']

    def on_execute(self):
        begemot = BegemotAllServices()
        problem_workers = []
        for worker_name in begemot.Services:
            golovan_prj = begemot.Service[worker_name].prj
            if golovan_prj:
                num_signals = self.get_num_signals(golovan_prj)
                if num_signals >= self.Parameters.signals_limit:
                    problem_workers.append(worker_name)
                worker_result = "<li><b>{}</b>:\t{}</li>".format(worker_name, num_signals)
                self.Context.workers_info += worker_result
        if len(problem_workers) > 0:
            error_text = "Signals count is {} or more on {}.".format(self.Parameters.signals_limit, ", ".join(problem_workers))
            raise errors.TaskFailure(error_text)
