import json
import logging
import time

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.betas.yappy_api import YappyApi
from sandbox.sandboxsdk import environments

YAPPY_API_URL = "https://yappy.z.yandex-team.ru/"

def get_hosts_by_beta_name(beta):
    api = YappyApi(YAPPY_API_URL, ssl_verify=False)
    beta_info = api._make_req_post(
        '/api/yappy.services.Model/retrieveBeta',
        data=json.dumps({'name': beta, 'withHistory': False})
    )
    instances = []
    for component in beta_info['components']:
        instances.extend(component['slot']['instances'])
    instances = [inst.split(':')[0] for inst in instances]
    return instances

class CheckBetaCoredumps(sdk2.Task):

    class Requirements(sdk2.Requirements):
        environments = (
            environments.PipEnvironment('yasmapi'),
        )

    class Parameters(sdk2.Parameters):
        name = sdk2.parameters.String(
            'Beta name',
            required=True
        )

        itype = sdk2.parameters.String(
            'Yasm itype',
            required=True
        )

        signals = sdk2.parameters.List(
            'Signals should be equal to zero',
            required=True
        )

        exclude_substrings = sdk2.parameters.List(
            'Exclude hosts with this substrings in name',
            required=False
        )

        time_to_finish = sdk2.parameters.Integer(
            'Time to wait for coredumps (minutes)',
            required=True,
            default=180,
            description='If task founds no coredumps during this time, it will be finished'
        )

        retry_time = sdk2.parameters.Integer(
            'Retry period (minutes)',
            required=True,
            default=30
        )

    class Context(sdk2.Context):
        start_time = 0
        check_start_time = 0
        last_failure = 0
        last_success = 0
        fails = 0

    def get_request_time(self):
        retry_time = self.Parameters.retry_time * 60
        start_time = self.Context.check_start_time - retry_time
        end_time = int(time.time())

        PERIODS = [5, 300, 3600]
        period = PERIODS[0]
        for p in PERIODS:
            if p < retry_time:
                period = p

        return start_time, end_time, period

    def get_nonzero_signals(self, hosts):
        from yasmapi import GolovanRequest

        start_time, end_time, period = self.get_request_time()
        reports = {}
        signals_cnt = 0
        warnings_cnt = 0
        for host in hosts:
            skip = False
            for substr in self.Parameters.exclude_substrings:
                if substr in host:
                    skip = True
            if skip:
                logging.debug("Excluded host {}".format(host))
                continue
            for sig in self.Parameters.signals:
                args = {
                    'host': host,
                    'period': period,
                    'st': start_time,
                    'et': end_time,
                    'fields': ["itype={}:{}".format(self.Parameters.itype, sig)],
                    'max_retry': 5,
                    'retry_delay': 3
                }
                gr = GolovanRequest(**args)
                logging.debug('GolovanRequest: {}'.format(args))
                for ts, values in gr:
                    for sig in values:
                        cnt = values[sig]
                        signals_cnt += 1
                        logging.debug("{}, {}, {}: {}".format(host, sig, ts, cnt))
                        if cnt is None:
                            warnings_cnt += 1
                            logging.debug("WARNING: failed to find signal {} for host {} and timestamp {}".format(sig, host, ts))
                        elif cnt > 0:
                            if (host, sig) not in reports:
                                reports[(host, sig)] = 0
                            reports[(host, sig)] += cnt

        return reports, 100.0 * warnings_cnt / signals_cnt

    def get_chart_url(self, host, signal):
        start_time, end_time, _ = self.get_request_time()
        return "https://yasm.yandex-team.ru/chart/hosts={};{}/?from={}&to={}".format(
            host,
            signal.replace(":", ";signals="),
            start_time * 1000,
            end_time * 1000
        )

    def get_problems_report(self, problems):
        report = ["Task found following problems for last {} minutes:".format(self.Parameters.retry_time)]
        for key in problems:
            host, signal, cnt = key[0], key[1], problems[key]
            chart = self.get_chart_url(host, signal)
            report.append('{}: {} coredumps. <a href="{}">Chart</a>'.format(host, cnt, chart))
        return "\n".join(report)


    def on_execute(self):
        if self.Context.start_time == 0:
            self.Context.start_time = int(time.time())
            raise sdk2.WaitTime(60 * self.Parameters.retry_time)
        self.Context.check_start_time = int(time.time())
        hosts = get_hosts_by_beta_name(self.Parameters.name)
        problems, warnings = self.get_nonzero_signals(hosts)
        if warnings > 0:
            self.set_info("WARNING: task failed to receive {:.2f}% values of signals. Grep 'failed to find signal' in debug.log for more info".format(warnings))
        if not problems:
            self.Context.last_success = self.Context.check_start_time
            no_cores_time = self.Context.last_success - max(self.Context.last_failure, self.Context.start_time)
            logging.debug("no cores time: {}".format(no_cores_time))
            time_to_finish = self.Parameters.time_to_finish
            if no_cores_time > (time_to_finish * 60):
                self.set_info("No coredumps for {} minutes".format(time_to_finish))
                if self.Context.fails > 0:
                    raise TaskFailure("No coredumps last {} munutes, but some coredumps were found earlier. Pay attention to charts.".format(time_to_finish))
                if self.Context.fails > 0:
                    raise TaskFailure("No coredumps last {} munutes, but some coredumps were found earlier. Pay attention to charts.".formst(time_to_finish))
                return
            self.set_info("No coredumps. Waiting for {} minutes".format(self.Parameters.retry_time))
        else:
            self.Context.fails += 1
            self.set_info(self.get_problems_report(problems), do_escape=False)
            self.Context.last_failure = self.Context.check_start_time
            if self.Context.fails >= 10:
                raise TaskFailure("Too many coredumps")
        raise sdk2.WaitTime(60 * self.Parameters.retry_time)
