import datetime
import logging
import operator
import pytz

from sandbox import sdk2
import sandbox.common.types.notification as ctn


class ResourceDescription:
    def __init__(self, t, id, date, shard, age):
        self.type = t
        self.id = id
        self.date = date
        self.shard = shard
        self.age = age

    def html(self):
        return 'Resource {} <a href="https://sandbox.yandex-team.ru/resource/{}">#{}</a> from {} ({} days old)'.format(
                self.type, self.id, self.id, self.date, self.age
            )


class CheckResourcesAge(sdk2.Task):
    class Parameters(sdk2.Parameters):
        res_to_check = sdk2.parameters.List(
            'List of resources numbers to check',
            required=True,
        )
        max_age = sdk2.parameters.Integer(
            'Critical age (days)',
            required=True,
            default=14,
        )
        email_list = sdk2.parameters.List(
            'Staff logins for email notification',
            required=False,
        )
        testenv_db_name = sdk2.parameters.String(
            'Testenv database name',
            required=False,
        )

    def get_resources_info(self, res_id):
        # returns [type, id, creation date, shard, age] for each resource
        resources = []
        for res_id in self.Parameters.res_to_check:
            res = sdk2.Resource.find(id=res_id).first()
            resources.append([res, (datetime.datetime.now(pytz.timezone('Europe/Moscow')) - res.created).days])
        resources = sorted(resources, key=operator.itemgetter(0), reverse=True)
        for i, res in enumerate(resources):
            resources[i] = ResourceDescription(res[0].type.name, res[0].id, res[0].created.strftime('%Y-%m-%d'), res[0].shard, res[1])
        info = '\n'.join([r.html() for r in resources])
        self.set_info(info, do_escape=False)
        return resources

    def send_emails(self, info, email_list):
        body = '<br>'.join([r.html() for r in info])
        task_link = '<a href="https://sandbox.yandex-team.ru/task/{}">#{}</a>'.format(self.id, self.id)
        testenv_link = 'testenv database'
        if self.Parameters.testenv_db_name:
            testenv_link = '<a href="https://testenv.yandex-team.ru/?screen=jobs&database={}">testenv database</a>'.format(self.Parameters.testenv_db_name)
        msg = '<br>'.join([
            'Task {} found some old resources in your {}'.format(task_link, testenv_link),
            '',
            'Here is info about checked resources:',
            body
        ])
        subject = 'Old resources in database {}'.format(self.Parameters.testenv_db_name) if self.Parameters.testenv_db_name else 'Old sandbox resources'
        self.server.notification(
            subject=subject,
            body=msg,
            recipients=email_list,
            transport=ctn.Transport.EMAIL,
            urgent=False,
            type='html'
        )

    def on_execute(self):
        info = self.get_resources_info(self.Parameters.res_to_check)
        max_age = info[0].age
        if max_age > self.Parameters.max_age and self.Parameters.email_list:
            self.send_emails(info, self.Parameters.email_list)
