# -*- coding: utf-8 -*-
import json
import logging
import requests
import socket
import subprocess
import time

import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import network
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common.wizard import utils as wizard_utils
from sandbox.projects.websearch.begemot import resources
from sandbox.projects.websearch.begemot.common import BegemotAllServices
from sandbox.projects.websearch.begemot.common.fast_build import ShardSyncHelper

_WIZARD_SOURCE = 'search.app_host.sources.(_.name eq "INPUT").results.(_.type eq "wizard")'


class CheckWizhosts(sdk2.Task):
    '''
    Validate that srcrwr-s expected by Priemka actually work
    '''
    begemot_port, begemot_grpc_port = (None, None)

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 40 * 1024
        client_tags = wizard_utils.ALL_SANDBOX_HOSTS_TAGS

    class Parameters(sdk2.Task.Parameters):
        host = sdk2.parameters.String('Host to check', default='https://hamster.yandex.ru')
        requests_num = sdk2.parameters.Integer('Number of requests', default=24)
        requests_timeout = sdk2.parameters.Integer('Requests timeout', default=10)
        begemot_binary = sdk2.parameters.Resource(
            'Begemot executable resource id', required=True, resource_type=resources.BEGEMOT_EXECUTABLE
        )
        fast_build_config = sdk2.parameters.Resource(
            'Begemot shard fast build config', required=False, resource_type=resources.BEGEMOT_FAST_BUILD_CONFIG_MERGER
        )

    def on_save(self):
        wizard_utils.setup_hosts(self)

    def on_enqueue(self):
        db = self.Context.testenv_database or ''
        if db == 'ws-begemot-trunk':
            self.Requirements.disk_space = 2 * 1024

    def perform_request(self, more_args={}):
        params = {
            'json_dump': _WIZARD_SOURCE,
            'text': 'test',
            'lr': 213,
            'dbgwzr': 2
        }
        params.update(more_args)
        try:
            url = '{}/search'.format(self.Parameters.host)
            logging.debug('Performing request to %s,\n%s' % (url, json.dumps(params)))
            r = requests_wrapper.get(url, params=params)
        except requests.ConnectionError:
            logging.warning("Connection error: %s", eh.shifted_traceback())
            raise

        try:
            rules = None
            j = r.json()
            rules = j[_WIZARD_SOURCE][0][0]['rules']
            return rules['.version']
        except (KeyError, IndexError):
            logging.warning('No .version in response: %s', rules or j)
            raise

    @staticmethod
    def same_worker(version_value, w):
        return version_value.endswith(' ' + w) or ' %s;' % w in version_value

    def perform_check(self, worker, ah_source, my_host):
        srcrwr = '{ah_source}=[{host}]:{port}:{timeout_ms}'.format(
            host=network.get_my_ipv6(), port=self.begemot_grpc_port, timeout_ms=10000, ah_source=ah_source
        )
        nattempts = self.Parameters.requests_num
        backends = {}
        for i in range(nattempts):
            if i:
                time.sleep(self.Parameters.requests_timeout)

            try:
                backends = self.perform_request({'srcrwr': srcrwr})
            except (requests.ConnectionError, KeyError, IndexError):
                continue

            logging.debug('Received versions: %s', backends)

            if my_host not in backends:
                # Timeout or connection problems. Retry.
                continue

            for host, version in backends.items():
                if host != my_host and CheckWizhosts.same_worker(version, worker):
                    return (
                        False,
                        '{ah_source}: Error, there is another response from {worker} (host {host})'.format(**locals()),
                    )

            return True, '{ah_source}: OK'.format(**locals())

        backends = '\n    ' + '\n    '.join(['%s: %s' % (h, w) for h, w in backends.items()])
        return (
            False,
            '{ah_source}: Failed after {nattempts} attempts, no {my_host} in backends: {backends}'.format(**locals()),
        )

    def on_execute(self):
        executable = str(sdk2.ResourceData(self.Parameters.begemot_binary).path)

        data = ShardSyncHelper(self.Parameters.fast_build_config).sync_shard()

        self.begemot_port, self.begemot_grpc_port = (8892, 8893)
        args = [executable, '--data', data, '--port', str(self.begemot_port), '--grpc', str(self.begemot_grpc_port)]
        logging.debug('Starting begemot: %s' % args)
        begemot = subprocess.Popen(args)
        my_host = '{host}:{port}'.format(host=socket.gethostname(), port=self.begemot_port)
        info = []
        fails = []
        ok = True
        default_backends = None
        for attempt in range(self.Parameters.requests_num):
            try:
                default_backends = self.perform_request()
                # expect no answer from at most one worker
                if len(default_backends) < 6:
                    break
            except Exception as x:
                self.set_info(str(x))
                pass
        # accept here a response from at least one worker
        if default_backends is None:
            raise TaskFailure('Unable to get response from hamster')
        try:
            for service in BegemotAllServices().Service.values():
                ah_source = service.apphost_source_name
                if not ah_source:
                    continue
                for v in default_backends.values():
                    if CheckWizhosts.same_worker(v, service.name):
                        break
                else:
                    # No such worker in response, so we cannot check srcrwr in this test
                    continue
                ok1, report = self.perform_check(service.name, ah_source, my_host)
                ok = ok and ok1
                if ok1:
                    info.append(report)
                else:
                    fails.append(report)
        finally:
            begemot.kill()
        self.set_info('\n'.join(info))
        if not ok:
            eh.check_failed('\n'.join(fails))
