# -*- coding: utf-8 -*-
import re
import os
import uuid
import json
import logging
import requests

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from collections import defaultdict
import sandbox.projects.WizardRuntimeBuild as WizardRuntimeBuild
from sandbox.projects.common import resource_selectors
from sandbox.projects.common import utils
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.constants import (
    ARCADIA_URL_KEY,
    SEMI_DISTBUILD_BUILD_SYSTEM,
)
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.websearch.begemot import AllBegemotServices
from sandbox.projects.websearch.begemot import resources
from sandbox.projects.websearch.begemot import BW_RESOURCES
from sandbox.projects.websearch.begemot.common import Begemots
from sandbox.projects.BuildBegemotExecutable import Targets
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox import sdk2


def _names_to_resources(names):
    return [resources.__dict__[name] for name in names]


NANNY_API_URL = 'http://nanny.yandex-team.ru'
SANDBOX_URL = 'https://proxy.sandbox.yandex-team.ru/'
# These resources from the dict above and from the list of shards
#  are not built when building 'search_begemot'
RELEASED_SEPARATELY = (
    'WizardRuntime',
    'BegemotFresh',
    'BegemotFreshFastBuild',
    'BegemotFreshFastBuildCommon',

    'AdvqExecutable',
    'MegamindExecutable',
    'BegginsExecutable',
    'RequestInitExecutable',
    'CaesarModelsExecutable',
)

ACCEPT_MISSING_IN_RELEASE = {
    'RTC_SIMPLE_LIBCUDA_LAYER'  # Released very manually as of now
}


class RevParameter(parameters.SandboxIntegerParameter):
    name = 'needed_revision'
    description = 'Revision of all resources'
    required = False


class BegemotResourcesParameter(parameters.SandboxBoolGroupParameter):
    name = "begemot_resources"
    description = "Select resources"
    required = True
    choices = [(shard.name, shard.name) for shard in Begemots.values() if shard.release] + [(t, t) for t in BW_RESOURCES]


class IsSearchRelease(parameters.SandboxBoolParameter):
    """
    Search begemot: релиз поискового бегемота через дежсмену
    Не search begemot: авторелиз быстрых данных или отдельных бегемотов
    """
    name = "all_resources"
    default_value = False
    description = "Release search begemot (binaries + slow data)"
    sub_fields = {
        'false': [BegemotResourcesParameter.name]
    }


class BegemotStableServicesDashboard(parameters.SandboxSelectParameter):
    """
    Nanny dashboard to resolve production(stable) begemot services for
    """
    choices = [
        ('begemot', 'begemot'),
        ('begemot_advq', 'begemot_advq'),
        ('begemot_model_services', 'begemot_model_services'),
    ]
    default_value = 'begemot'
    name = 'begemot_stable_services_dashboard'
    description = 'nanny Dashboard to resolve stable begemot services'


class BegemotTestingServicesPattern(parameters.SandboxSelectParameter):
    """
    Nanny path regexp for testing services
    """
    choices = [
        ('^/search/begemot/.*/testing/$', '^/search/begemot/.*/testing/$'),
        ('^/yabs/caesar_models/hamster/.*', '^/yabs/caesar_models/hamster/.*'),
    ]
    default_value = '^/search/begemot/.*/testing/$'
    name = 'begemot_testing_service_pattern'
    description = 'Nanny path regexp for testing services'


class NeedStableReleaseConfig(parameters.SandboxBoolParameter):
    """Whether to make BEGEMOT_STABLE_RELEASE_CONFIG resource"""
    default_value = False
    name = 'need_stable_release_config'
    description = 'Whether to make BEGEMOT_STABLE_RELEASE_CONFIG resource'


class WizardDataDiskSpace(parameters.SandboxIntegerParameter):
    name = 'wizard_data_disk_space'
    description = 'Disk space for BUILD_WIZARD_DATA and wizard shard of BUILD_BEGEMOT_DATA tasks, in MB'
    required = False


class CollectBegemotResources(task.SandboxTask):
    execution_space = 1024  # 1 Gb (last detected usage is 100 Mb)
    input_parameters = (
        RevParameter,
        IsSearchRelease,
        BegemotResourcesParameter,
        build_params.ArcadiaUrl,
        build_params.ArcadiaPatch,
        build_params.BuildType,
        build_params.DefinitionFlags,
        build_params.UseArcadiaApiFuse,
        WizardRuntimeBuild.RobotsRevision,
        WizardDataDiskSpace,
        NeedStableReleaseConfig,
        BegemotStableServicesDashboard,
    )

    def initCtx(self):
        ctx = task.SandboxTask.initCtx(self)
        ctx.update(kill_timeout=21600)
        return ctx

    def get_targets(self):
        if utils.get_or_default(self.ctx, IsSearchRelease):
            return [
                target[0] for target in BegemotResourcesParameter.choices
                if target[0] not in RELEASED_SEPARATELY
            ]
        return self.ctx[BegemotResourcesParameter.name].strip().split(' ')

    def run_fast_fresh_build_task(self, fresh_shards, common_config=False):
        input_params = {
            'ShardName': ' '.join(fresh_shards),
            'BuildFresh': True,
            'UseFastBuild': True,
            'AllInOneConfig': common_config,
            ARCADIA_URL_KEY: self.ctx['svn_url_with_rev']
        }
        if common_config:
            res_name = 'BEGEMOT_FAST_BUILD_FRESH_CONFIG'
            input_params.update({res_name: self.ctx[res_name]})
        else:
            for shard in fresh_shards:
                res_name = AllBegemotServices[shard].fresh_fast_build_config_resource_type.name
                input_params.update({res_name: self.ctx[res_name]})
        build_fresh_task = self.create_subtask(
            'BUILD_BEGEMOT_DATA',
            description=self.description,
            input_parameters=input_params,
            inherit_notifications=True,
        )
        return build_fresh_task.id

    def run_full_fresh_build_task(self, fresh_shards):
        runtime_task = self.create_subtask(
            task_type='WIZARD_RUNTIME_BUILD',
            description=self.description,
            input_parameters={
                WizardRuntimeBuild.RobotsRevision.name: utils.get_or_default(
                    self.ctx, WizardRuntimeBuild.RobotsRevision
                ),
                WizardRuntimeBuild.ParentResources.name: {
                    t: self.ctx[t]
                    for t in self.ctx['target_fresh_types']
                },
                WizardRuntimeBuild.BegemotShards.name: ' '.join(fresh_shards),
                WizardRuntimeBuild.RuntimeDataSource.name: WizardRuntimeBuild.RuntimeDataSource.Arcadia,
                WizardRuntimeBuild.RuntimeLegacy.name: WizardRuntimeBuild.RuntimeDataSource.Robots,
                WizardRuntimeBuild.DoMerge.name: False,
                WizardRuntimeBuild.DoCommit.name: True,
                WizardRuntimeBuild.DoBuild.name: True,
            },
            inherit_notifications=True,
        )
        return runtime_task.id

    def build_fresh(self, fast_build=False, common_fast_build=False):
        if common_fast_build:
            fast_build = True
        fresh_shards = [name for name, s in Begemots if s.release_fresh]
        # Begemot fresh shards in cypress
        for shard_name in fresh_shards:
            # Resource will be found in child task by parent_id and shard_name is_fresh attributes
            self.create_resource(
                description=shard_name, resource_type=resources.BEGEMOT_CYPRESS_SHARD,
                attributes={'shard_name': shard_name, 'is_fresh': True}, resource_path=str(uuid.uuid4()),
            )
        if fast_build:
            task_id = self.run_fast_fresh_build_task(fresh_shards, common_fast_build)
        else:
            task_id = self.run_full_fresh_build_task(fresh_shards)

        self.ctx['build_tasks_id'].append(task_id)
        for t in self.ctx['target_fresh_types']:
            utils.set_resource_attributes(
                self.ctx[t], {'runtime_data': utils.get_or_default(self.ctx, WizardRuntimeBuild.RobotsRevision)}
            )

    def build_binary(
        self, description,
        task_type='BUILD_BEGEMOT_EXECUTABLE',
        resources=(resources.BEGEMOT_EXECUTABLE, ),
        input_parameters=None,
    ):
        merged_input = {
            ARCADIA_URL_KEY: self.ctx['svn_url_with_rev'],
            build_params.ArcadiaPatch.name: utils.get_or_default(self.ctx, build_params.ArcadiaPatch),
            build_params.BuildType.name: utils.get_or_default(self.ctx, build_params.BuildType),
            build_params.DefinitionFlags.name: self.ctx[build_params.DefinitionFlags.name],
            build_params.LTO.name: False,
            build_params.Musl.name: True,
            build_params.SeparateDebug.name: True,
            build_params.UseArcadiaApiFuse.name: self.ctx[build_params.UseArcadiaApiFuse.name],

            Targets.name: [r.name for r in resources],
        }
        for r in resources:
            merged_input[r.name] = self.ctx[r.name]
        if input_parameters is not None:
            merged_input.update(input_parameters)

        build_binary_task = self.create_subtask(
            task_type=task_type,
            description=description + self.description,
            input_parameters=merged_input,
            arch='linux',
        ).id
        self.ctx['build_tasks_id'].append(build_binary_task)
        return build_binary_task

    def collect_static_files(self, target):
        parsed_url = Arcadia.parse_url(self.ctx[build_params.ArcadiaUrl.name])
        for resource in BW_RESOURCES[target]:
            modified_path = os.path.join(parsed_url.path, resource.arcadia_build_path)
            modified_url = Arcadia.replace(self.ctx[build_params.ArcadiaUrl.name], path=modified_path)
            Arcadia.export(modified_url, resource.name)
            self.mark_resource_ready(self.ctx[resource.name])

    def create_subtasks(self):
        target_resources = self.get_targets()
        target_shards = [
            target for target in target_resources
            if target not in BW_RESOURCES and Begemots[target].release
        ]
        self.ctx['svn_url_with_rev'] = self.ctx[ARCADIA_URL_KEY]
        if self.ctx[RevParameter.name]:
            self.ctx['svn_url_with_rev'] = self.ctx['svn_url_with_rev'] + "@" + str(self.ctx[RevParameter.name])
        self.ctx['build_tasks_id'] = []

        if "BegemotExecutable" in target_resources:
            self.build_binary(
                resources=[
                    resources.BEGEMOT_EXECUTABLE,
                    resources.BEGEMOT_EVLOGDUMP,
                ],
                description='Main (search)',
            )

            self.build_binary(
                resources=[
                    resources.BEGEMOT_BERT_EXECUTABLE,
                ],
                description='Bert',
                input_parameters={
                    build_params.Musl.name: False,
                },
            )

            self.build_binary(
                resources=[
                    resources.BEGEMOT_EVLOG_UPLOADER,
                    resources.BEGEMOT_YT_EVENTLOG_MAPPER,
                    resources.BEGEMOT_YT_MAPPER,
                    resources.BEGEMOT_FAST_BUILD_DOWNLOADER,
                    resources.BEGEMOT_ARGUMENTS_PARSER,
                    resources.BEGEMOT_SHARD_UPDATER,
                ],
                description='Utilities',
            )

            self.build_binary(
                resources={
                    s.binary_resource for s in AllBegemotServices.values()
                    if s.release and s.released_by == 'search' and not s.beta and s.binary_resource not in {
                        resources.BEGEMOT_EXECUTABLE,  # already built above anyway
                        resources.BEGEMOT_BERT_EXECUTABLE,  # requires no-musl
                    }
                },
                description='Not beta-tested',
            )

        if "AdvqExecutable" in target_resources:
            self.build_binary(
                resources=BW_RESOURCES['AdvqExecutable'],
                description='Advq',
            )

        if "MegamindExecutable" in target_resources:
            self.build_binary(
                resources=BW_RESOURCES['MegamindExecutable'],
                description='Megamind',
            )

        if "BegginsExecutable" in target_resources:
            self.build_binary(
                resources=BW_RESOURCES['BegginsExecutable'],
                description='Beggins',
                input_parameters={
                    build_params.BuildSystem.name: SEMI_DISTBUILD_BUILD_SYSTEM,
                    build_params.DefinitionFlags.name: '-DCUDA_VERSION=11.4 -DCUDNN_VERSION=8.0.5',
                },
            )

        if "BegginsExecutableAddons" in target_resources:
            self.build_binary(
                resources=BW_RESOURCES['BegginsExecutableAddons'],
                description='Beggins executable addons without exectuable',
            )

        if "RequestInitExecutable" in target_resources:
            self.build_binary(
                resources=BW_RESOURCES['RequestInitExecutable'],
                description='RequestInit',
            )

        if "CaesarModelsExecutable" in target_resources:
            self.build_binary(
                resources=BW_RESOURCES['CaesarModelsExecutable'],
                description='CaesarModels',
                input_parameters={
                    build_params.Musl.name: False,
                },
            )

        if "BegemotConfig" in target_resources:
            build_config_task = self.create_subtask(
                task_type='BUILD_BEGEMOT_CONFIG',
                description=self.description,
                input_parameters={
                    ARCADIA_URL_KEY: self.ctx['svn_url_with_rev'],
                    build_params.ArcadiaPatch.name: utils.get_or_default(self.ctx, build_params.ArcadiaPatch),
                    resources.BEGEMOT_CONFIG.name: self.ctx[resources.BEGEMOT_CONFIG.name],
                },
                arch='linux',
            )
            self.ctx['build_tasks_id'].append(build_config_task.id)

        description = self.ctx.get('testenv_database', '')
        if description:
            description = ' for ' + description

        for shard_name in target_shards:
            if shard_name == "Geo":  # We use full build for shard Geo
                continue

            # Resource will be found in child task by parent_id and shard_name is_fresh attributes
            self.create_resource(
                description=shard_name, resource_type=resources.BEGEMOT_CYPRESS_SHARD,
                attributes={'shard_name': shard_name, 'is_fresh': False}, resource_path=str(uuid.uuid4()),
            )

            fb_res_type = AllBegemotServices[shard_name].fast_build_config_resource_type
            build_shard_task = self.create_subtask(
                'BUILD_BEGEMOT_DATA',
                '{shard}. Created by RELEASE_BEGEMOT_DATA{desc}'.format(shard=shard_name, desc=description),
                input_parameters={
                    'ShardName': shard_name,
                    ARCADIA_URL_KEY: self.ctx['svn_url_with_rev'],
                    build_params.ArcadiaPatch.name: utils.get_or_default(self.ctx, build_params.ArcadiaPatch),
                    build_params.BuildType.name: utils.get_or_default(self.ctx, build_params.BuildType),
                    'UseFastBuild': True,
                    'CypressCache': '//home/begemot/begemot-tests/shards_cache' if 'Spellchecker' not in shard_name else '',
                    fb_res_type.name: self.ctx[fb_res_type.name]
                },
                execution_space=self.ctx.get(WizardDataDiskSpace.name) if shard_name == 'Wizard' else None,
            )
            self.ctx['build_tasks_id'].append(build_shard_task.id)

        if "Geo" in target_shards:
            geoshard_resource_type = AllBegemotServices["Geo"].data_resource_type
            build_geoshard_task = self.create_subtask(
                'BUILD_BEGEMOT_DATA',
                '{shard}. Created by RELEASE_BEGEMOT_DATA{desc}'.format(shard="Geo", desc=description),
                input_parameters={
                    'ShardName': "Geo",
                    ARCADIA_URL_KEY: self.ctx['svn_url_with_rev'],
                    build_params.ArcadiaPatch.name: utils.get_or_default(self.ctx, build_params.ArcadiaPatch),
                    build_params.BuildType.name: utils.get_or_default(self.ctx, build_params.BuildType),
                    'UseFastBuild': False,
                    'UseFullShardBuild': True,
                    geoshard_resource_type.name: self.ctx[geoshard_resource_type.name]
                },
            )
            self.ctx['build_tasks_id'].append(build_geoshard_task.id)

        if "WizardExecutable" in target_resources:
            task_id = self.build_binary(
                task_type='BUILD_WIZARD_EXECUTABLE',
                resources=BW_RESOURCES["WizardExecutable"],
                description='Legacy wizard',
            )
            self.ctx['wizard_binary_task_id'] = task_id

        self.ctx['target_fresh_types'] = [
            t.name for t in
            (
                BW_RESOURCES['WizardRuntime'] * ('WizardRuntime' in target_resources) +
                BW_RESOURCES['BegemotFresh'] * ('BegemotFresh' in target_resources) +
                BW_RESOURCES['BegemotFreshFastBuild'] * ('BegemotFreshFastBuild' in target_resources) +
                BW_RESOURCES['BegemotFreshFastBuildCommon'] * ('BegemotFreshFastBuildCommon' in target_resources)
            )
        ]
        if self.ctx['target_fresh_types']:
            self.build_fresh('BegemotFreshFastBuild' in target_resources, 'BegemotFreshFastBuildCommon' in target_resources)

        if 'Bstr' in target_resources:
            input_params = {
                build_params.ArcadiaUrl.name: self.ctx[build_params.ArcadiaUrl.name],
                build_params.UseArcadiaApiFuse.name: self.ctx[build_params.UseArcadiaApiFuse.name],
            }
            input_params.update((r.name, self.ctx[r.name]) for r in BW_RESOURCES['Bstr'])
            build_bstr_task = self.create_subtask(
                task_type='BUILD_BEGEMOT_BSTR',
                description=self.description,
                input_parameters=input_params,
                inherit_notifications=True,
            )
            self.ctx['build_tasks_id'].append(build_bstr_task.id)

        if 'NannyConfigs' in target_resources:
            self.collect_static_files('NannyConfigs')

        if 'EventsFiles' in target_resources:
            self.collect_static_files('EventsFiles')
            self.build_binary(
                resources=[
                    resources.BEGEMOT_EVLOGDUMP_FOR_UNIFIED_AGENT,
                ],
                description='evlogdump for unified agent',
            )

    def create_target_resource(self, target_name):
        if target_name not in BW_RESOURCES:
            # If a target is not in BW_RESOURCES - this is a begemot shard
            svc = AllBegemotServices[target_name]
            if not svc.release:
                return
            requested_resources = [
                (svc.fast_build_config_resource_type, target_name + "_fast_build") if target_name != "Geo" else
                (svc.data_resource_type, target_name),
            ]
        elif 'BegemotFresh' in target_name:
            requested_resources = []
            for res in BW_RESOURCES[target_name]:
                name = 'fast_build_config.json' if 'FastBuild' in target_name else 'fresh'
                path = os.path.join(res.dir_name, name)
                if res.name.endswith('_PACKED'):
                    path += '.tar'
                requested_resources.append((res, path))
        else:
            requested_resources = [(res, res.name) for res in BW_RESOURCES[target_name]]

        for res_type, res_path in requested_resources:
            if res_type and res_type.name not in self.ctx:
                self.ctx[res_type.name] = self.create_resource(
                    description=res_type.name, resource_type=res_type, resource_path=res_path, arch='linux'
                ).id

    def on_enqueue(self):
        target_resources = self.get_targets()
        self.ctx['arcadia_revision'] = self.ctx.get('needed_revision')

        svn_branch = self.ctx.get('testenv_database', '').split('-')[-1]
        self.svn_branch = None
        # allow running on trunk te base
        try:
            self.svn_branch = int(self.svn_branch)
        except TypeError, ValueError:
            pass

        for target in target_resources:
            self.create_target_resource(target)
        task.SandboxTask.on_enqueue(self)

    def on_execute(self):
        token, session = None, None

        def get_token_and_start_client(token=None, session=None):
            if session is None:
                try:
                    token = sdk2.Vault.data('BEGEMOT', 'Begemot Nanny token')
                    session = NannyClient(api_url=NANNY_API_URL, oauth_token=token)
                    return token, session
                except Exception:
                    logging.debug("We failed during taking token from Vault and starting Nanny client")
                    return None, None
            return token, session

        def get_stable_services_list(token=None, session=None):
            dashboard_id = utils.get_or_default(self.ctx, BegemotStableServicesDashboard)
            token, session = get_token_and_start_client(token, session)
            return session.get_dashboard_services(dashboard_id)

        def get_testing_services_list(token=None, session=None):
            re_service_category = utils.get_or_default(self.ctx, BegemotTestingServicesPattern)
            token, session = get_token_and_start_client(token, session)
            url = NANNY_API_URL + '/api/repo/ListSummaries/'
            all_services = session._get_url(url)['value']
            begemot_pattern = re.compile(re_service_category)
            return [s['serviceId'] for s in all_services if re.match(begemot_pattern, s['category'])]

        def get_files_from_service(service_id, token=None, session=None):
            token, session = get_token_and_start_client(token, session)
            response = session.get_service_runtime_attrs(service_id=service_id)
            return response['content']['resources']['sandbox_files']

        def get_url(session, url):
            try:
                r = session.get(url)
                r.raise_for_status()
            except requests.HTTPError as e:
                raise Exception("Failed to get FastBuildConfig data from url `{}`".format(url))
            except Exception as e:
                raise Exception('Failed to get url "{}"\nError: {}'.format(url, e))
            return r.json()

        def check_child_resource(res_type):
            res = sdk2.Resource[res_type].find(task_id=self.id).first()
            return res is not None

        def check_if_need_release_rule(res_id):
            res = sdk2.Resource.find(id=res_id, attrs={"released": "stable"}).first()
            return res is None

        def resource_description_dict(res_id, local_path, use_ssd, sandbox_release):
            result_dict = {
                'resource_id': res_id,
                'local_path': local_path,
                'sandbox_release': sandbox_release,
            }
            if use_ssd:
                result_dict['storage'] = '/ssd'
            return result_dict

        def get_resources_for_service(fast_build_config_name, use_ssd):
            session = requests.Session()
            retries = Retry(total=3, backoff_factor=1.0, status_forcelist=[429, 500, 502, 503, 504])
            session.mount(SANDBOX_URL, HTTPAdapter(max_retries=retries))
            res_id = self.ctx.get(fast_build_config_name, '')
            if res_id:
                url = SANDBOX_URL + str(res_id)
                resources = get_url(session, url)['resources']
                fb_config = [resource_description_dict(res_id, fast_build_config_name, use_ssd, sandbox_release=True)]
                files = [resource_description_dict(i["resource_id"], 'data/' + i["name"], use_ssd, sandbox_release=check_if_need_release_rule(i["resource_id"])) for i in sorted(resources)]
                return fb_config, files
            self.set_info("Found no FastBuildConfig resource in task, file name is `{}`".format(fast_build_config_name))
            return [], []

        # add up-to-date executable: BEGEMOT-2685
        def add_unified_agent_executable(file, use_ssd):
            resource_type = file['resource_type']
            last_resource_id, _ = resource_selectors.by_last_released_task(resource_type=sdk2.Resource[resource_type])
            return resource_description_dict(
                last_resource_id, file['local_path'], use_ssd, sandbox_release=check_child_resource(resource_type)
            )

        def create_release_json(services_list):
            release_json = defaultdict(list)
            for service in services_list:
                logging.debug("Starting build json description of resources of service {}".format(service))
                files = get_files_from_service(service, token=token, session=session)
                use_ssd = any([file.get('storage') == '/ssd' for file in files])
                if use_ssd:
                    logging.debug('Will use ssd for putting resources')
                else:
                    logging.debug('There are no resources on ssd now, so will not use it')
                for file in files:
                    if file['resource_type'].startswith('BEGEMOT_FAST_BUILD_CONFIG_'):
                        shard_name = file['resource_type']
                    elif file['resource_type'] == 'BEGEMOT_FAST_BUILD_RULE_DATA':
                        continue  # this is rules data
                    elif file['resource_type'] == 'BEGEMOT_FAST_DATA_CALLBACK':
                        continue  # we add it after cycle, do not add it twice
                    elif file['resource_type'] == 'BEGEMOT_BSTR_CALLBACK':
                        continue  # do not deploy old callback
                    elif file['resource_type'] == 'UNIFIED_AGENT_BIN' and file['local_path'] == 'unified_agent_setrace':
                        release_json[service].append(add_unified_agent_executable(file, use_ssd))
                    else:
                        file_name = file['resource_type']
                        file_id = self.ctx.get(file_name, '')
                        do_release = check_child_resource(file['resource_type'])
                        if not file_id:
                            file_id = file['resource_id']
                            if file_name not in ACCEPT_MISSING_IN_RELEASE:
                                self.set_info("Found no `{}` resource in task for service `{}`".format(file_name, service))
                        release_json[service].append(resource_description_dict(file_id, file['local_path'], use_ssd, sandbox_release=do_release))
                release_json[service].append(resource_description_dict(self.ctx.get('BEGEMOT_FAST_DATA_CALLBACK', ''), 'fast_data_callback', use_ssd, sandbox_release=True))

                fast_build_config_file, new_files = get_resources_for_service(shard_name, use_ssd)
                logging.debug("fast_build_config_file {}".format(type(fast_build_config_file)))
                logging.debug("release_json[service] {}".format(type(release_json[service])))
                logging.debug("new_files {}".format(type(new_files)))
                release_json[service] = fast_build_config_file + release_json[service] + new_files
            return json.dumps(release_json, encoding='utf-8', indent=2)

        token, session = get_token_and_start_client()
        if session is None:
            self.set_info("Sadly, we failed starting Nanny client.")
            if token is None:
                logging.debug("Also we failed taking token from Vault.")
            nanny_imported = False
        else:
            self.set_info("Started Nanny client successfully!")
            nanny_imported = True

        self.ctx['begemot_check_task_id'] = []
        self.description = ' for ' + self.type
        db = self.ctx.get('testenv_database', '')
        if db:
            self.description += ', ' + db

        if 'build_tasks_id' not in self.ctx:
            self.create_subtasks()
            utils.wait_all_subtasks_stop()
        elif not utils.check_all_subtasks_done():
            utils.restart_broken_subtasks()
        else:
            utils.check_subtasks_fails(stop_on_broken_children=True, fail_on_first_failure=True)
            for task_id in self.ctx['begemot_check_task_id']:
                cur_task = channel.sandbox.get_task(task_id)
                if cur_task.is_failure():
                    raise SandboxTaskFailureError('task type={} ; id={} FAILED'.format(cur_task.type, task_id))

        if utils.get_or_default(self.ctx, NeedStableReleaseConfig) or (
            nanny_imported and utils.get_or_default(self.ctx, IsSearchRelease)
        ):
            stable_release_json = create_release_json(get_stable_services_list(token=token, session=session))
            testing_release_json = create_release_json(get_testing_services_list(token=token, session=session))

            with open("release_stable.json", "w") as f:
                f.write(stable_release_json)
            stable_release_res = self.create_resource(
                description=self.description,
                resource_path="release_stable.json",
                resource_type=resources.BEGEMOT_STABLE_RELEASE_CONFIG
            ).id
            channel.sandbox.set_resource_attribute(resource_id=stable_release_res, attribute_name='ttl', attribute_value=60)
            self.ctx['out_resource_id'] = stable_release_res

            with open("release_testing.json", "w") as f:
                f.write(testing_release_json)
            testing_release_res = self.create_resource(
                description=self.description,
                resource_path="release_testing.json",
                resource_type=resources.BEGEMOT_TESTING_RELEASE_CONFIG
            ).id
            channel.sandbox.set_resource_attribute(resource_id=testing_release_res, attribute_name='ttl', attribute_value=30)
