import os
import shutil

from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.websearch.begemot import parameters as bp
from sandbox.projects.websearch.begemot.common.fast_build import ShardSyncHelper
from sandbox.projects.websearch.begemot.common.fast_build.hasher import ArcadiaDependencyHasher


class CompareBegemotFastAndFullShardBuild(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 200 << 10

    class Parameters(sdk2.Task.Parameters):
        full_shard = sdk2.parameters.Resource(
            'Begemot full shard',
            required=True
        )
        fast_build_config = bp.FastBuildConfigResource(required=True)
        fail_on_any_error = True

    def check_hashes_equality(self, first_path, second_path):
        first_hasher = ArcadiaDependencyHasher(first_path, {})
        second_hasher = ArcadiaDependencyHasher(second_path, {})
        first_hash = first_hasher.count_physical_hash("")
        second_hash = second_hasher.count_physical_hash("")
        return first_hash == second_hash

    def on_enqueue(self):
        self.Requirements.disk_space = self.Requirements.ram = min(500 * 1024, 3 * (ShardSyncHelper(self.Parameters.fast_build_config).get_shard_size() >> 20))

    def _prepare_full_shard(self, path="full_build_shard"):
        return str(sdk2.ResourceData(self.Parameters.full_shard).path)

    def _prepare_fast_build_shard(self, path="fast_build_shard"):
        shard_helper = ShardSyncHelper(self.Parameters.fast_build_config)
        data_path = str(self.path(path))
        shard = shard_helper.sync_shard(data_path)
        return shard

    def on_execute(self):
        full_shard = self._prepare_full_shard()
        fast_build_shard = self._prepare_fast_build_shard()

        # Full shard is readonly. We cant delete version.pb.txt so let us just copy it
        # to fast build shard.
        full_shard_version_pbtxt = os.path.join(full_shard, 'version.pb.txt')
        fast_build_shard_version_pbtxt = os.path.join(fast_build_shard, 'version.pb.txt')
        if os.path.isfile(full_shard_version_pbtxt):
            shutil.copyfile(full_shard_version_pbtxt, fast_build_shard_version_pbtxt)

        if not self.check_hashes_equality(full_shard, fast_build_shard):
            self.set_info("Hashes are not equal. Running diff.")
            with sdk2.helpers.ProcessLog(self, logger='diff') as pl:
                pl.logger.propagate = 1
                subprocess.check_call(['diff', '-r', full_shard, fast_build_shard], stderr=subprocess.STDOUT, stdout=pl.stdout)
