import json

from sandbox import sdk2
from sandbox.projects.websearch.begemot import resources as br


class CompareBegemotShardSize(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        first_rules_size = sdk2.parameters.Dict(
            'First rules size',
            value_type=sdk2.parameters.Integer,
        )

        second_rules_size = sdk2.parameters.Dict(
            'Second rules size',
            value_type=sdk2.parameters.Integer,
        )

        acceptable_diff_percent = sdk2.parameters.Float(
            'What percentage of the change in the size of the shard '
            '(the sum of the size of the rules) considered a diff'
        )

        number_mb_for_diff = sdk2.parameters.Integer(
            'What size change in MB is considered a diff (put -1 to not use). Used when percent is not reached',
            default=-1,
        )

        acceptable_diff_mb = sdk2.parameters.Integer(
            'What size change in MB does not count as a diff (put -1 to not use). Used when the percentage is reached',
            default=-1,
        )

        with sdk2.parameters.Output:
            diff_mb = sdk2.parameters.Integer('The differences of sum rules in mb')
            diff_percent = sdk2.parameters.Float('The differences of sum rules in percent')
            diffs = sdk2.parameters.Resource('Diffs resource')

    def on_execute(self):
        sum_first = sum(self.Parameters.first_rules_size.values())
        sum_second = sum(self.Parameters.second_rules_size.values())

        diff = sum_second - sum_first
        if sum_first > 0:
            percent = abs(diff) / float(sum_first) * 100
        else:
            raise RuntimeError('First shard is empty')

        self.Parameters.diff_mb = diff
        self.Parameters.diff_percent = percent

        if (percent > self.Parameters.acceptable_diff_percent or
                (abs(diff) >= self.Parameters.number_mb_for_diff and self.Parameters.number_mb_for_diff != -1)):
            self.Context.has_diff = (abs(diff) > self.Parameters.acceptable_diff_mb or self.Parameters.acceptable_diff_mb == -1)
        else:
            self.Context.has_diff = False

        if self.Context.has_diff:
            by_rule = {}
            for k in set(self.Parameters.first_rules_size.keys() + self.Parameters.second_rules_size.keys()):
                value = self.Parameters.second_rules_size.get(k, 0) - self.Parameters.first_rules_size.get(k, 0)
                if value:
                    by_rule[k] = value
            by_rule['TOTAL_MB'] = diff
            with open('output.txt', 'w') as out:
                json.dump(by_rule, out, indent=2)

            self.Parameters.diffs = br.BEGEMOT_SHARD_SIZE_DIFF(
                self,
                '',
                'output.txt'
            ).id
