import json
import tree_config
import os
import sys
import sandbox.projects.websearch.begemot.tasks.GetBegemotBetasQueries.answer_normalizer as answer_normalizer
from sandbox.projects.websearch.begemot.tasks.BegemotCreateResponsesDiff import jsondiff
from sandbox.projects import resource_types
from sandbox import sdk2


class DiffMetaResource(sdk2.Resource):
    """ Diff meta resource """
    queries_count = sdk2.parameters.Integer("All queries count")
    relev_changed_count = sdk2.parameters.Integer("Queries with changed relev")
    rearr_changed_count = sdk2.parameters.Integer("Queries with changed rearr")
    diff_count = sdk2.parameters.Integer("Queries with diff")


class ComputeBegemotDiffMetrics(sdk2.Task):
    """ Download queries from target beta """

    class Parameters(sdk2.Task.Parameters):
        # custom parameters
        server1 = sdk2.parameters.String("Beta #1 (old) (example: begemot-200-1)", required=False)
        server2 = sdk2.parameters.String("Beta #2 (new) (example: begemot-201-1)", required=False)
        responses1 = sdk2.parameters.Resource(
            "Beta #1 (old) responses",
            required=True,
            # type=resource_types.PLAIN_TEXT_QUERIES  # FIXME: invalid argument (SANDBOX-6404)
        )
        responses2 = sdk2.parameters.Resource(
            "Beta #2 (new) responses",
            required=True,
            # type=resource_types.PLAIN_TEXT_QUERIES  # FIXME: invalid argument (SANDBOX-6404)
        )

    @property
    def footer(self):

        if self.Context.diff_meta_id:
            betasCgi = ""
            for i, beta in enumerate([self.Parameters.server1, self.Parameters.server2], 0):
                if beta: betasCgi += "&beta{}={}".format(i, beta)

            return [{
                "helperName": "",
                "content": "<a href=\"https://proxy.sandbox.yandex-team.ru/471410542/view.html?resourceId={}{}\">Diff</a>".format(
                    self.Context.diff_meta_id, betasCgi
                )
            }]

    def on_execute(self):
        diff_html_output = sdk2.ResourceData(resource_types.OTHER_RESOURCE(self, "Html diff", "diff.html"))
        diff_meta_output = DiffMetaResource(self, "Json meta diff", "meta-diff.json")

        with open(str(sdk2.ResourceData(self.Parameters.responses1).path)) as f:
            old = json.load(f)
        with open(str(sdk2.ResourceData(self.Parameters.responses2).path)) as f:
            new = json.load(f)

        MOD_KEY = "$modification"
        EXTRA_KEY = "$extra"

        def fill_changes(x, output):
            cur_mod = output.get(MOD_KEY, "")
            next_mod = ""
            if isinstance(x, jsondiff.Add):
                next_mod = "add"
            elif isinstance(x, jsondiff.Del):
                next_mod = "del"
            elif isinstance(x, jsondiff.Mod):
                next_mod = "mod"
            elif isinstance(x, dict):
                for key in x:
                    temp = output.get(key, {})
                    if fill_changes(x[key], temp):
                        next_mod = "mod"
                    if len(temp) > 0:
                        output[key] = temp
            elif isinstance(x, list):
                for item in x:
                    if fill_changes(item, output):
                        next_mod = "mod"

            if cur_mod and next_mod and cur_mod != next_mod:
                next_mod = "mod"

            if next_mod: output[MOD_KEY] = next_mod
            return next_mod

        def fill_conf(x, path, config):
            weight = 0
            if x.has_key(MOD_KEY):
                weight = config.find_attribute(path, "weight-{}".format(x[MOD_KEY]), None) or \
                         config.find_attribute(path, "weight", None)
                if weight is None: weight = 0
            if weight > 0:
                extra = x.get(EXTRA_KEY, {})
                extra["weight"] = weight
                x[EXTRA_KEY] = extra

            for key in x:
                if key.startswith("$"): continue
                path.append(key)
                fill_conf(x[key], path, config)
                path.pop(len(path) - 1)

        conf = tree_config.TreeConfig()
        with open(os.path.join(os.path.dirname(__file__), 'conf.json')) as conf_file:
            tree_config.load_config(conf, json.load(conf_file))

        for x in (old, new):
            for k in x:
                answer_normalizer.normalize_all(x[k])

        diff = jsondiff.diff(old, new)
        groups = jsondiff.group([(_, diff[_]) for _ in diff])
        if not groups: groups = []
        small = [(group[0], [group_item[0] for group_item in group]) for group in groups]

        result = {}
        for group in small:
            group_title = group[0][0]
            group_changes = group[0][1]
            group_items = group[1]

            result[group_title] = {
                EXTRA_KEY: {
                    "count": len(group_items),
                    "links": group_items
                }
            }
            fill_changes(group_changes, result[group_title])
            fill_conf(result[group_title], [], conf)

        diff_meta_output.queries_count = len(diff)
        diff_meta_output.relev_changed_count = 0
        diff_meta_output.rearr_changed_count = 0
        diff_meta_output.diff_count = 0

        for q in result:
            queries_count = result[q][EXTRA_KEY]["count"]
            diff_meta_output.diff_count += queries_count
            if result[q].has_key("relev"):
                diff_meta_output.relev_changed_count += queries_count
            if result[q].has_key("rearr"):
                diff_meta_output.rearr_changed_count += queries_count

        sdk2.ResourceData(diff_meta_output).path.write_bytes(json.dumps(result))
        self.Context.diff_meta_id = diff_meta_output.id
        self.Context.cache_hit = 1.0 - ((diff_meta_output.relev_changed_count * 1.0) / diff_meta_output.queries_count)

        html_lines = []
        for tag in jsondiff.render_html([("{} ({} queries)".format(group[0][0], len(group[1])), group[0][1]) for group in small]):
            if sys.hexversion < 0x03000000:
                tag = tag.encode('utf-8')
            html_lines.append(tag)
        diff_html_output.path.write_bytes("\n".join(html_lines))
