import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

from sandbox import sdk2
from sandbox import sandboxsdk
from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import NotExists
from sandbox.common.types.task import Status
from sandbox.projects.websearch.begemot.tasks.DeployBegemotFreshDataWithBstr import DeployBegemotFreshDataWithBstr
from sandbox.projects.websearch.begemot.tasks.ReleaseBegemotFresh import build_fast_data_config
from sandbox.projects.websearch.upper.fast_data.DeployFastData import DeployFastData
from sandbox.projects.websearch.upper import resources as upper_resources


class DeployBegemotFresh(sdk2.Task):
    """
    Deploys begemot fresh/runtime data.
    """

    class Parameters(sdk2.Parameters):
        services = sdk2.parameters.List('Nanny services', default=[])
        shard = sdk2.parameters.String('Begemot shard name')
        resource_type = sdk2.parameters.String('Fresh package resource type', default='BEGEMOT_REALTIME_PACKAGE')
        build_task_type = sdk2.parameters.String('Build task type')
        check_child_resource = sdk2.parameters.Bool('Check last released fresh is produced by child task (if Build task type is set)', default=True)
        deploy_parallel = sdk2.parameters.Bool('Deploy all locations in parallel', default=False)
        check_attrs = sdk2.parameters.Bool('Check only released resources', default=True)
        deploy_timeout = sdk2.parameters.Integer('Kill timeout for single task (in seconds)', default=2*60*60)
        geo = sdk2.parameters.List('Locations to deploy', default=['sas', 'man', 'vla'])

        with sdk2.parameters.RadioGroup("Deployer") as deployer:
            deployer.values['Samogon'] = deployer.Value('Samogon', default=True)
            deployer.values['Fast data deployer'] = deployer.Value('Fast data deployer')
        with deployer.value['Fast data deployer']:
            nanny_token = sdk2.parameters.String('Nanny token', default='Begemot Nanny token', required=True)
            yt_token = sdk2.parameters.String('YT token', default='yt_token_for_testenv', required=True)
            deployer_resource = sdk2.parameters.Resource(
                'Deployer resource',
                resource_type=upper_resources.FastDataDeployer,
                required=False
            )
            deployer_service = sdk2.parameters.String('Nanny deployer service', required=False)

        with sdk2.parameters.RadioGroup("Begemot data type") as data_type:
            data_type.values.fresh = data_type.Value(value='fresh', default=True)
            data_type.values.runtime = data_type.Value(value='runtime')

    class Requirements(sdk2.Requirements):
        environments = [sandboxsdk.environments.PipEnvironment('yandex-yt', version='0.10.8')]
        client_tags = ctc.Tag.GENERIC
        cores = 1
        ram = 4 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        deployed = []
        deployed_geo = []
        deploy_tasks_ids = []
        deployed_all = False

    def check_subtasks(self):
        for task in sdk2.Task.find(parent=self, children=True).limit(50):
            if task.status != Status.SUCCESS and task.status != Status.DRAFT:
                if task.Context.no_rtc_component is NotExists or not task.Context.no_rtc_component:
                    err_msg = 'task {} finished with status {}'.format(task.Parameters.description, task.status)
                    self.set_info(err_msg)
                    raise Exception(err_msg)

                self.set_info("Child task {} failed to connect to data center".format(task.Parameters.description))

    def check_last_released_fresh(self, fresh_res):
        if self.Parameters.build_task_type:
            if fresh_res.task_id != self.Context.build_task:
                msg = "Last released fresh resource is #{}. It is not produced by child task #{}. Check that your task produces and releases a fresh resource with type {}".format(
                    fresh_res.id,
                    self.Context.build_task,
                    self.Parameters.resource_type
                )
                raise TaskFailure(msg)

    def need_deploy_in_geo(self, service, geo):
        if not service[-4:] in ['_sas', '_man', '_vla'] and not service[:4] in ['sas_', 'man_', 'vla_']:
            return True
        if service[-4:] == '_' + geo or service[:4] == geo + '_':
            return True
        return False

    def on_execute(self):
        if self.Parameters.build_task_type:
            with self.memoize_stage.build:
                kwargs = {}
                build_task_class = sdk2.Task[self.Parameters.build_task_type]
                if hasattr(build_task_class.Parameters, "build_resource_type"):
                    kwargs["build_resource_type"] = self.Parameters.resource_type

                self.Context.build_task = build_task_class(self, **kwargs).enqueue().id
                raise sdk2.WaitTask(self.Context.build_task, statuses=Status.Group.FINISH | Status.Group.BREAK)

            self.check_subtasks()

        with self.memoize_stage.package:
            attrs = {}
            if self.Parameters.check_attrs:
                attrs = {
                    'released': 'stable',
                    'shard': self.Parameters.shard
                }
            last_released_fresh = sdk2.Resource[self.Parameters.resource_type].find(attrs=attrs, state=ctr.State.READY).order(-sdk2.Resource.id).first()
            self.set_info('Last released fresh: {}'.format(last_released_fresh.id))
            if self.Parameters.check_child_resource:
                self.check_last_released_fresh(last_released_fresh)
            self.Context.skynet_id = last_released_fresh.skynet_id

        self.check_subtasks()
        if self.Parameters.deployer == 'Samogon':
            prefix = '//home/search-runtime/' + ('begemot-fresh' if self.Parameters.data_type == 'fresh' else 'realtime-data')

            import yt.wrapper as yt
            for geo in self.Parameters.geo:
                if geo in self.Context.deployed_geo:
                    continue
                for service in self.Parameters.services:
                    if service not in self.Context.deployed and self.need_deploy_in_geo(service, geo):
                        task = DeployBegemotFreshDataWithBstr(
                            self,
                            description='Begemot fresh data for {}'.format(service),
                            rbtorrent=self.Context.skynet_id,
                            yt_prefix=yt.ypath_join(prefix, service, geo),
                            geo=geo,
                            nanny_service=service,
                            kill_timeout=self.Parameters.deploy_timeout
                        ).enqueue().id
                        self.Context.deployed.append([service, geo])
                        self.Context.deploy_tasks_ids.append(task)
                self.Context.deployed_geo.append(geo)
                if not self.Parameters.deploy_parallel:
                    raise sdk2.WaitTask(self.Context.deploy_tasks_ids, Status.Group.FINISH | Status.Group.BREAK)

            if not self.Context.deployed_all and self.Parameters.deploy_parallel:
                self.Context.deployed_all = True
                raise sdk2.WaitTask(self.Context.deploy_tasks_ids, Status.Group.FINISH | Status.Group.BREAK)
        else:
            if not self.Context.deployed_all:
                prefix = '//home/search-runtime/fast-data/begemot{}'.format('_realtime' if self.Parameters.data_type == 'runtime' else '')
                config = build_fast_data_config(self.Parameters.services, prefix, prepare_dl=1.0, parallel=self.Parameters.deploy_parallel)
                self.Context.fast_data_config = config
                if not self.Parameters.deployer_resource:
                    deployer = upper_resources.FastDataDeployer.find(state='READY', attrs={'released': 'stable'}).first().id
                else:
                    deployer = self.Parameters.deployer_resource

                task_id = DeployFastData(
                    self,
                    fast_data_bundle=last_released_fresh,
                    deployer_mode='nanny_service' if self.Parameters.deployer_service else 'standalone',
                    nanny_service=self.Parameters.deployer_service,
                    yt_token_name=self.Parameters.yt_token,
                    nanny_token_name=self.Parameters.nanny_token,
                    deployer=deployer,
                    deploy_config=config,
                    kill_timeout=self.Parameters.deploy_timeout,
                ).enqueue().id
                self.Context.deployed_all = True
                raise sdk2.WaitTask(task_id, Status.Group.FINISH | Status.Group.BREAK)
