import os
import subprocess

from sandbox import common

import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.common.types import notification
from sandbox.projects.resource_types.releasers import wizard_releasers
from sandbox.projects.websearch.begemot.resources import BEGEMOT_BSTR, BEGEMOT_BSTR_PUSHER


NO_RTC_COMPONENT = "One of RTC components currently is not available."

class BegemotBstrForFreshDataDeployment(sdk2.Resource):
    releasable = True
    executable = True
    releasers = wizard_releasers


class BegemotPusherForFreshDataDeployment(sdk2.Resource):
    releasable = True
    executable = True
    releasers = wizard_releasers


class SearchReleasersLastReleasedResource(sdk2.parameters.Resource):
    @common.utils.classproperty
    def default_value(cls):
        resources = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            owner='SEARCH-RELEASERS',
            attrs={'released': 'stable'},
            limit=1
        )['items']
        if not resources:
            return None
        return resources[0]['id']


class BegemotLastReleasedResource(sdk2.parameters.Resource):
    @common.utils.classproperty
    def default_value(cls):
        resources = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            owner='BEGEMOT',
            attrs={'released': 'stable'},
            limit=1
        )['items']
        if not resources:
            return None
        return resources[0]['id']


class DeployBegemotFreshDataWithBstr(sdk2.Task):
    """bstr push"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 300
        ram = 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3 * 60 * 60

        rbtorrent = sdk2.parameters.String("rbtorrent with a shard", required=True)
        yt_prefix = sdk2.parameters.String("yt prefix", required=True)
        geo = sdk2.parameters.String("geo", required=True)
        nanny_service = sdk2.parameters.String("nanny service for pusher to watch", required=True)

        bstr_resource = BegemotLastReleasedResource("bstr", resource_type=BEGEMOT_BSTR)
        pusher_resource = BegemotLastReleasedResource("pusher", resource_type=BEGEMOT_BSTR_PUSHER)

    def on_execute(self):
        self.Context.no_rtc_component = False
        bstr_path, pusher_path = [
            str(sdk2.ResourceData(r).path)
            for r in [self.Parameters.bstr_resource, self.Parameters.pusher_resource]
        ]
        pusher_cmd = [
            pusher_path,
            '--file', self.Parameters.rbtorrent,
            '--geo', self.Parameters.geo,
            '--yt-prefix', self.Parameters.yt_prefix,
            '--bstr-binary', bstr_path,
            '--nanny-service', self.Parameters.nanny_service,
            '--sandbox-task', str(self.id),
            '--log', str(self.log_path('pusher.log')),
        ]

        pusher_env = os.environ.copy()
        pusher_env['YT_TOKEN'] = sdk2.Vault.data('yt_token_for_robot-wizard-bstr')
        pusher_env['OAUTH_NANNY'] = sdk2.Vault.data('nanny_token_for_robot-wizard-bstr')

        self.set_info('Starting pusher: ' + ' '.join(pusher_cmd))

        with sdk2.helpers.ProcessLog(self, logger="pusher") as pl:
            try:
                subprocess.check_call(
                    pusher_cmd,
                    env=pusher_env,
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                )
            except subprocess.CalledProcessError as e:
                with self.log_path('pusher.err.log').open() as f:
                    pusher_logs = f.read()
                    self.set_info(pusher_logs)
                    if NO_RTC_COMPONENT in pusher_logs:
                        self.Context.no_rtc_component = True
                raise

        self.set_info('Done running pusher')
