import datetime
import tarfile
import os
from sandbox import sdk2
from sandbox import sandboxsdk
from sandbox.common.types.misc import NotExists
from sandbox.common.types.task import Status
from sandbox.projects.mt.spellchecker import SpellcheckerUpdate
from sandbox.projects.websearch.begemot.tasks.DeployBegemotFreshDataWithBstr import DeployBegemotFreshDataWithBstr
from sandbox.projects.websearch.begemot.resources import BEGEMOT_FRESH_DATA_FOR_SPELLCHECKER_PACKED
from sandbox.projects.websearch.begemot.tasks.ReleaseBegemotFresh import build_fast_data_config
from sandbox.projects.websearch.upper.fast_data.DeployFastData import DeployFastData


def set_permissions(tarinfo):
    tarinfo.mode = 0o777
    return tarinfo


class DeployBegemotSpellcheckerFresh(sdk2.Task):
    """
    Builds begemot Spellchecker shard fresh package from SpellcheckerUpdate
    """

    class Shards(object):
        FULL = 'full'
        RUS = 'rus'
        ENG = 'eng'
        MISC1 = 'misc1'

    class Parameters(sdk2.Parameters):
        services = sdk2.parameters.List('Nanny services', default=[])
        spellchecker_update = sdk2.parameters.Resource('Old Spellchecker package', resource_type=SpellcheckerUpdate)

        with sdk2.parameters.RadioGroup("Deployer") as deployer:
            deployer.values['Samogon'] = deployer.Value('Samogon', default=True)
            deployer.values['Fast data deployer'] = deployer.Value('Fast data deployer')
        with deployer.value['Fast data deployer']:
            nanny_token = sdk2.parameters.String('Nanny token', default='Begemot Nanny token', required=True)
            yt_token = sdk2.parameters.String('YT token', default='yt_token_for_testenv', required=True)


    class Requirements(sdk2.Requirements):
        environments = [sandboxsdk.environments.PipEnvironment('yandex-yt', version='0.10.8')]

    class Context(sdk2.Context):
        deploy_tasks_ids = []
        deployed = []
        deployed_geo = []
        resource_by_shard = {}
        deployed_all = False

    def check_subtasks(self):
        for task in sdk2.Task.find(parent=self, children=True).limit(50):
            if task.status != Status.SUCCESS:
                if task.Context.no_rtc_component is NotExists or not task.Context.no_rtc_component:
                    err_msg = 'task {} finished with status {}'.format(task.Parameters.description, task.status)
                    self.set_info(err_msg)
                    raise Exception(err_msg)

                self.set_info("Child task {} failed to connect to data center".format(task.Parameters.description))

    def package_for_service(self, service):
        if self.Shards.RUS in service:
            return self.Context.resource_by_shard[self.Shards.RUS]
        elif self.Shards.ENG in service:
            return self.Context.resource_by_shard[self.Shards.ENG]
        elif self.Shards.MISC1 in service:
            return self.Context.resource_by_shard[self.Shards.MISC1]
        else:
            return self.Context.resource_by_shard[self.Shards.FULL]  # main spellchecker shard

    def create_archive(self, shard, data):
        rule_name = {
            self.Shards.FULL: 'Spellchecker',  # main spellchecker shard
            self.Shards.ENG: 'SpellcheckerENG',
            self.Shards.RUS: 'SpellcheckerRUS',
            self.Shards.MISC1: 'SpellcheckerMisc1',
        }[shard]
        # Path should be unique for each resource, so we need to put it in a directory: "shard/fresh.tar".
        os.mkdir(shard)
        archive_path = '{}/fresh.tar'.format(shard)
        archive = tarfile.TarFile(archive_path, 'w')
        archive.add(data, '{}/update'.format(rule_name), filter=set_permissions)
        version_file = 'version.pb.txt'
        with open(version_file, 'w') as fp:
            fp.write(
                'GenerationTime: "{}"\nRevision: {}\nTask: {}\nShardName: "Spellchecker"\n'.format(
                    datetime.datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ'), self.id, self.id,
                )
            )
        archive.add(version_file)
        archive.close()
        package = BEGEMOT_FRESH_DATA_FOR_SPELLCHECKER_PACKED(self, 'Begemot fresh package for Spellchecker', archive_path, version=int(self.id))
        sdk2.ResourceData(package).ready()
        if self.Parameters.deployer == 'Samogon':
            return package.skynet_id
        else:
            return package.id

    def on_execute(self):
        with self.memoize_stage.package:
            last_released_fresh = SpellcheckerUpdate.find(released='stable').order(-sdk2.Resource.id).first()
            self.set_info('SpellcheckerUpdate resource: {}'.format(last_released_fresh.id))
            data = str(sdk2.ResourceData(self.Parameters.spellchecker_update or last_released_fresh).path)
            for shard in [self.Shards.FULL, self.Shards.RUS, self.Shards.ENG, self.Shards.MISC1]:
                self.Context.resource_by_shard[shard] = self.create_archive(shard, data)

        self.check_subtasks()

        if self.Parameters.deployer == 'Samogon':
            prefix = '//home/search-runtime/begemot-fresh'
            import yt.wrapper as yt
            for geo in ['sas', 'man', 'vla']:
                if geo in self.Context.deployed_geo:
                    continue
                for service in self.Parameters.services:
                    if service not in self.Context.deployed and (service[-3:] == geo or not service[-4:] in ['_sas', '_man', '_vla']):
                        skynet_id = self.package_for_service(service)
                        task = DeployBegemotFreshDataWithBstr(
                            self,
                            description='Begemot fresh data for {}'.format(service),
                            rbtorrent=skynet_id,
                            yt_prefix=yt.ypath_join(prefix, service, geo),
                            geo=geo,
                            nanny_service=service,
                        ).enqueue().id
                        self.Context.deployed.append(service)
                        self.Context.deploy_tasks_ids.append(task)
                self.Context.deployed_geo.append(geo)
                raise sdk2.WaitTask(self.Context.deploy_tasks_ids, Status.Group.FINISH | Status.Group.BREAK)
        else:
            if self.Context.deployed_all:
                return
            prefix = '//home/search-runtime/fast-data/begemot'
            services_by_resource = {}
            for service in self.Parameters.services:
                resource = self.package_for_service(service)
                if resource in services_by_resource:
                    services_by_resource[resource].append(service)
                else:
                    services_by_resource[resource] = [service]
            self.Context.services_by_resource = services_by_resource

            for resource in services_by_resource:
                config = build_fast_data_config(services_by_resource[resource], prefix)
                self.Context.fast_data_config = config
                task_id = DeployFastData(
                    self,
                    fast_data_bundle=resource,
                    deployer_mode='standalone',
                    deployer=1969548435,
                    yt_token_name=self.Parameters.yt_token,
                    nanny_token_name=self.Parameters.nanny_token,
                    deploy_config=config,
                ).enqueue().id
                self.Context.deploy_tasks_ids.append(task_id)

            self.Context.deployed_all = True
            raise sdk2.WaitTask(self.Context.deploy_tasks_ids, Status.Group.FINISH | Status.Group.BREAK)
