from json import dumps
from os.path import isdir, join
from requests import Session
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from time import time
import re

from sandbox import common, sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia


class FindBegemotObsoleteRules(sdk2.Task):

    class Requirements(sdk2.Requirements):
        environments = [environments.PipEnvironment('yasmapi')]

    def _get_session(self, token=None):
        session = Session()
        session.headers['Content-Type'] = 'application/json'
        if token is not None:
            session.headers['Authorization'] = 'OAuth {}'.format(token)
        session.mount('https://', HTTPAdapter(max_retries=Retry(total=10, backoff_factor=0.1, status_forcelist=[202])))
        return session

    def _get_all_rules(self):
        url_prefix = 'https://yasm.yandex-team.ru/metainfo/signals/'
        rules = set()
        session = self._get_session()
        for itype, prefix in self._service_parameters:
            request_url = url_prefix + '?itype={itype}&signal_pattern={prefix}-.*-TIME_dhhh'.format(itype=itype, prefix=prefix)
            signals = session.get(request_url).json()['response']['result']
            current_rules = [re.findall(r'\w+-\w+-([A-Z][a-zA-Z0-9]*[a-z][a-zA-Z0-9]*)-TIME_dhhh', s) for s in signals]
                # e. g., "begemot-WORKER-ThesaurusDataParams-TIME_dhhh" -> "ThesaurusDataParams"
            rules |= set([s[0] for s in current_rules if s])
        return rules

    def _get_signals(self, rule):
        return ['itype={itype};ctype=prod,prestable:{prefix}-{rule}-REQUESTS_dmmm'.format(itype=itype, prefix=prefix, rule=rule)
            for itype, prefix in self._service_parameters]

    def _get_useless_rules(self, rules):
        from yasmapi import GolovanRequest
        signals = {}
        for rule_name in rules:
            signals.update({s: rule_name for s in self._get_signals(rule_name)})
        packs_of_signals = [signals.keys()[i: i + 100] for i in range(0, len(signals), 100)]

        hosts='ASEARCH'
        period = 24 * 60 * 60
        current_time = int(time())
        half_year_ago = current_time - period * 30 * 6
        golovan_dots = []
        for current_signals in packs_of_signals:
            for _, signal_dict in GolovanRequest(hosts, period, half_year_ago, current_time, current_signals):
                golovan_dots += list(signal_dict.items())
        useful_rules = set([signals[s] for s, value in golovan_dots if value != 0])
        return rules - useful_rules

    def _get_deleted_rules(self, rules):
        Arcadia.checkout('arcadia:/arc/trunk/arcadia/search/wizard/data/wizard', 'ruledirs', depth='immediates')
        existing_rules = {rule_name for rule_name in rules if isdir(join('ruledirs', rule_name))}
        return rules - existing_rules

    def _create_startrek_ticket(self, session, rule):
        response = session.post(
            'https://st-api.yandex-team.ru/v2/issues',
            data={
                'queue': 'BEGEMOT',
                'summary': 'Rule {rule} is not used anymore. Please delete it.'.format(rule=rule),
                'tags': ['the-rule-is-no-more'],
            },
        )
        if response.ok:
            try:
                answer = '<a href="{ref}">{rule}</a>'.format(ref=response.json()['self'], rule=rule)
                return True, answer
            except:
                return True, rule
        else:
            try:
                answer = dumps(response.json(), indent=4)
                return False, answer
            except:
                return False, rule + ':\n' + response.text

    def on_execute(self):
        self._service_parameters = [
            ('wizard', 'unistat-WIZARD'),
            ('begemot', 'begemot-WORKER'),
        ]
        all_rules = self._get_all_rules()
        obsolete_rules = self._get_useless_rules(all_rules)
        deleted = self._get_deleted_rules(obsolete_rules)
        obsolete_rules -= deleted
        if not obsolete_rules:
            self.set_info('No useless rules found')
            return

        api = common.rest.Client()
        oauth_token = api.vault.read(name='Begemot startrek token', owner='BEGEMOT', limit=1)["items"][0]["id"]
        session = self._get_session(token=oauth_token)

        successes = []
        failures = []
        for rule in obsolete_rules:
            ok, answer = self._create_startrek_ticket(session, rule)
            if ok:
                successes.append(answer)
            else:
                failures.append(answer)

        if successes:
            self.set_info('Successfully created tasks for obsolete rules:\n' + '\n'.join(successes))
        if failures:
            self.set_info('Failed to create tasks for obsolete rules:\n' + '\n'.join(failures))
