import json
import logging
import answer_normalizer
from sandbox.projects.common.fasthttp import queriesloader
from sandbox.projects import resource_types
from sandbox import sdk2

KnownWorkers = [
    "Merger",
    "MisspellFeatures",
    "MisspellFeatures2",
    "Ethos",
    "Bravo",
    "Wizard",
]


class GetBegemotBetasQueries(sdk2.Task):
    """ Download queries from target beta """

    class Parameters(sdk2.Task.Parameters):
        # custom parameters
        server = sdk2.parameters.String("Beta id (example: begemot-200-1)")
        base_cgi = sdk2.parameters.String("Base cgi", default="&waitall=da&srcskip=APP_HOST_ADV_MACHINE&srcskip=APP_HOST_WEB&srcskip=APP_HOST_WEB_SETUP&srcskip=METASEARCH&srcskip=QUICK&srcskip=REGIONUS&srcskip=TEMPLATES&srcskip=UPPER&srcskip=WEB&srcskip=WEB_MISSPELL&srcskip=ADV_MACHINE&srcskip=WORD_STAT_PROXY&srcskip=IMAGES_SUGGEST_TAGS_PROXY&srcskip=REQUEST_EXTENSIONS_PROXY")
        additional_cgi = sdk2.parameters.String("Additional cgi", default="&json_dump_responses=WIZARDRY")
        retries = sdk2.parameters.Integer("Retries count", default=5, required=True)
        repeat_validation = sdk2.parameters.Integer("Repeat validation", default=6, required=True)
        threads = sdk2.parameters.Integer("Threads", default=6, required=True)
        batch_resource = sdk2.parameters.Resource("Batch resource", required=True)
        version_validate_list = sdk2.parameters.String(
            "Check version contains (split by comma)",
            default=", ".join(KnownWorkers))

    class Context(sdk2.Task.Context):
        output_queries_id = None

    def on_enqueue(self):
        if self.Context.output_queries_id is None:
            self.Context.output_queries_id = resource_types.PLAIN_TEXT_QUERIES(self, "Beta queries output", "queries.json").id

    def on_execute(self):
        output_dict_format = sdk2.ResourceData(sdk2.Resource[self.Context.output_queries_id])
        server = str(self.Parameters.server)
        retries = int(self.Parameters.retries)
        async_size = int(self.Parameters.threads)
        base_cgi = str(self.Parameters.base_cgi)
        additional_cgi = str(self.Parameters.additional_cgi)
        repeat_validation = int(self.Parameters.repeat_validation)
        version_validate_list = [x.strip() for x in str(self.Parameters.version_validate_list).split(",")]

        def make_request(request):
            request = queriesloader.GetRequest(
                "https://{beta}.hamster.yandex.{request}&sandbox_task-{task_id}".format(
                    beta=server, request=request, task_id=self.id
                )
            )
            return request

        def begemot_post_processor(data):
            try:
                data = json.loads(data)
                responses = data["WEB_SEARCH"]["WIZARDRY"]["response"]
            except:
                return None, "Invalid json"

            begemot_outputs = []
            for result_item in responses:
                if result_item["type"] == "wizard":
                    begemot_outputs.append(result_item["binary"])

            if not begemot_outputs:
                return None, "Empty wizard output"
            else:
                for x in begemot_outputs:
                    if "rules" not in x:
                        return None, "Rules not found"
                    if ".version" in x["rules"] and x["rules"][".version"]:
                        version = x["rules"][".version"]
                        for check_version in version_validate_list:
                            ok = False
                            for key in version:
                                if check_version in version[key]:
                                    ok = True
                                    break
                            if not ok:
                                return None, check_version + " worker unanswer"

                    answer_normalizer.drop_host_info(x)
                return begemot_outputs, ""

        with open(str(sdk2.ResourceData(self.Parameters.batch_resource).path)) as f:
            requests = [x.strip() for x in f.readlines()]

        result = {}

        for req_id, response in queriesloader.download_queries(
                [make_request(req) for req in requests],
                add_cgi=additional_cgi,
                async_size=async_size,
                retries=retries,
                hash_validation_retries=repeat_validation,
                post_processor=begemot_post_processor):

            if response and len(response) > 0:
                result[requests[req_id]] = response[0]
            else: result[requests[req_id]] = None

        output_dict_format.path.write_bytes(json.dumps(result))
