# -*- coding: utf-8 -*-
import os
import json
import logging
from six.moves import urllib

from sandbox.common import errors
from collections import defaultdict, deque
from contextlib import contextmanager
from sandbox.common.types.task import Status
from sandbox.sandboxsdk import process
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.constants import constants
from sandbox.projects.websearch.begemot import parameters as bp
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.WizardRuntimeBuild.ya_make import YaMake
from sandbox.projects.websearch.begemot.common import Begemots

ACTIONS_FILE = "actions.json"
YaMake = YaMake.YaMake


class GetBegemotResponsesMulti(sdk2.Task):
    """ Get Begemot responses from shards that are affected by a patch """
    fail_on_any_error = True

    class Parameters(sdk2.Task.Parameters):
        begemot_binary_list = bp.BegemotExecutableResource(multiple=True)
        begemot_binary_names = sdk2.parameters.List("Names of Begemot binary types",
            description="First name in the \'begemot_binary_names\' list corresponds to the first queries resource in the \'begemot_binary_list\' list, etc.",
            default=['BEGEMOT'],
            required=False
        )
        shards_list = sdk2.parameters.List(
            "List of shards corresponding to queries resources",
            description="First shard in the \'shards_list\' list corresponds to the first queries resource in the \'requst_plan\' list, etc.",
            default=[],
            required=True
        )
        requests_plan = bp.BegemotQueriesResource(multiple=True)
        arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)
        testenv_database_debug = sdk2.parameters.String("Value of testenv_database for debugging", default="")
        postcommit_test = sdk2.parameters.Bool('True if testing trunk now', default=False)
        verbose = sdk2.parameters.Bool('Check the box only if you want to debug', default=False)

    @contextmanager
    def get_arcadia(self, arcadia_url):
        try:
            with sdk.mount_arc_path(arcadia_url) as arcadia:
                yield arcadia
        except errors.TaskFailure as e:
            logging.exception(e)
            yield svn.Arcadia.get_arcadia_src_dir(arcadia_url)

    def verbose(self, *args, **kwargs):
        if self.Parameters.verbose:
            logging.debug(*args, **kwargs)
        else:
            pass

    def cut_basename(self, path):
        # it is safer to use os.path.isdir(path), but it is 10 times slower
        if '.' not in os.path.basename(path):
            return path
        else:
            return os.path.dirname(path)

    def build_graph(self):
        graph = defaultdict(list)
        nodes = set()
        with self.get_arcadia(self.arcadia_url) as arcadia:
            graph_dir = str(self.path('graph'))
            with open(graph_dir, 'w') as file:
                dump_graph_rule = process.run_process(
                    [os.path.join(arcadia, 'ya'),
                    'dump',
                    'dir-graph',
                    os.path.join(arcadia, 'search/begemot/rules/')],
                    stdout=file
                ).communicate()[0]
            node = str()
            for line in open(graph_dir, 'r'):
                if '"' not in line:
                    continue
                path = line.split('"')[1]
                path = self.cut_basename(path)
                nodes.add(path)
                if '[' in line:
                    node = path
                    continue
                graph[path].append(node)
        return (graph, nodes)

    def prepare_patch(self, nodes):
        def strip(path):
            if 'trunk/arcadia/' in path:
                return self.cut_basename(path.split()[1].replace('trunk/arcadia/', ''))
            else:
                return self.cut_basename(path.split()[1])

        if self.Parameters.verbose and self.Parameters.testenv_database_debug != "":
            db_name = self.Parameters.testenv_database_debug
        else:
            db_name = self.Context.testenv_database
        db_url = urllib.request.urlopen("https://ci.yandex-team.ru/api/v1.0/checks/" + db_name)
        db_json = json.loads(db_url.read().decode())
        patch_dir = self.path('patch')
        patch_dir.mkdir()
        path, zipatch = Arcadia.fetch_patch(db_json['patch_url'], str(patch_dir))

        file = open(path, 'r')
        mod_rules = [strip(ln) for ln in file.readlines() if ln.startswith(('--- trunk/arcadia/', '--- search/'))]
        self.verbose("rules that can be changed by patch:\n%r", mod_rules)
        q = deque(mod_rules)
        return q

    def bfs(self, q, graph, nodes, rules_files_dict, rules_set):
        self.verbose("BFS STARTED")
        self.verbose("deque: \n%r\n", q)
        rules = set()
        while q:
            node = q.popleft()
            self.verbose("BFS ONGOING, node: %r", node)
            if node.startswith('search/begemot/rules') and node in rules_files_dict:
                self.verbose("node in 'search/begemot/rules'")
                rule = rules_files_dict[node]
                rules.add(rule)
            elif node.startswith('search/wizard/data/wizard'):
                self.verbose("node in 'search/wizard/data/wizard'")
                guess_rule = node.split('search/wizard/data/wizard', 2)[1]
                for rule in guess_rule.split('/'):
                    if rule in rules_set:
                        self.verbose("rule found! %r", rule)
                        rules.add(rule)
                        break
            else:
                self.verbose("node is useless")
            self.verbose("children: %r", graph[node])
            for child in graph[node]:
                if child in nodes:
                    nodes.remove(child)
                    q.append(child)
                    self.verbose("adding child: %r", child)
        self.verbose("Rules:\n%r\n", rules)
        self.verbose("BFS ENDED")
        return rules

    def find_shards(self, rules):
        shards = []
        with self.get_arcadia(self.arcadia_url) as arcadia:
            sources = os.path.join(arcadia, 'search/begemot/data')
            for d in os.listdir(sources):
                path = os.path.join(sources, d)
                if os.path.isdir(path):
                    path = os.path.join(path, 'ya.make')
                    yamake = YaMake(path)
                    for r in yamake.peerdir:
                        if os.path.basename(r) in rules:
                            shards.append(d)
                            break
        return shards

    def on_execute(self):
        self.verbose("EXECUTING STARTED")
        with self.memoize_stage.find_affected_shards(commit_on_entrance=False):
            self.arcadia_url = self.Parameters.arcadia_from_url
            if self.Parameters.postcommit_test:
                self.Context.affected_shards = sorted(set(self.Parameters.shards_list))
            else:
                if self.Context.testenv_database == 'ws-begemot-trunk':
                    self.set_info('Precommit check does nothing in testenv begemot trunk database')
                    self.Context.get_begemot_responses_results = []
                    return
                graph, nodes = self.build_graph()
                q = self.prepare_patch(nodes)

                self.verbose("Graph:\n%r\n", graph)
                self.verbose("Nodes:\n%r\n", nodes)
                self.verbose('queue in "prepare_patch" = \n%r\n', q)

                rules_files = dict()
                for binary in self.Parameters.begemot_binary_list:
                    begemot_executable = str(sdk2.ResourceData(binary).path)
                    try:
                        rules_files.update(json.loads(subprocess.check_output([begemot_executable, '--print-rules-files'])))
                        self.verbose('rules_files:\n%r\n', rules_files)
                    except subprocess.CalledProcessError as exc:
                        logging.error("Generating list of (rules, files) failed, exit code {code}.\n{msg}\n========\n".format(
                            code=exc.returncode,
                            msg=exc.output
                        ))
                        raise

                rules_files_dict = dict()  # dict: directory in rules -> name of data
                rules_set = set()
                for name, file in rules_files.iteritems():
                    if not file:
                        continue
                    cut_file = self.cut_basename(file[file.find('search/begemot/rules'):])
                    rules_files_dict[cut_file] = name
                    rules_set.add(name)

                rules = self.bfs(q, graph, nodes, rules_files_dict, rules_set)
                self.Context.affected_shards = sorted(self.find_shards(rules))
                self.verbose('rules_files_dict: \n%r\n', rules_files_dict)
                self.verbose('rules_set: \n%r\n', rules_set)
                self.verbose("rules:\n%r\n", rules)

            self.Context.callable_shards = [shard for shard in self.Context.affected_shards if shard in Begemots.keys()]
            self.set_info("Shards that would be tested: {}".format(self.Context.callable_shards))
            self.set_info("All shards that could be changed by commit: {}".format(self.Context.affected_shards))


        import sandbox.projects.websearch.begemot.tasks.BuildBegemotData as BuildBegemotData
        begemot_data_class = sdk2.Task[BuildBegemotData.BuildBegemotData.type]

        with self.memoize_stage.build_begemot_data(commit_on_entrance=False):
            self.Context.build_begemot_data_tasks = [
                begemot_data_class(
                    self,
                    build_system=constants.YA_MAKE_FORCE_BUILD_SYSTEM,
                    build_type=constants.RELEASE_BUILD_TYPE,
                    ShardName=shard_name,
                    BuildTestShard=True,
                    checkout_arcadia_from_url=self.arcadia_url,
                    use_obj_cache=True,
                    CypressCache='//home/begemot/begemot-tests/shards_cache',
                    YtProxy='hahn',
                    UseFullShardBuild=False,
                    UseFastBuild=True,
                ).enqueue().id
                for shard_name in self.Context.callable_shards
            ]

            raise sdk2.WaitTask(self.Context.build_begemot_data_tasks, Status.Group.FINISH | Status.Group.BREAK, wait_all=True)


        bad_statuses = [Status.FAILURE, Status.EXCEPTION, Status.TIMEOUT]
        flag_failed_tasks = False
        for sub_task_id in self.Context.build_begemot_data_tasks:
            sub_task = sdk2.Task[sub_task_id]
            if sub_task.status in bad_statuses:
                flag_failed_tasks = True
                self.set_info("BuildBegemotData task {} for shard {} have failed.".format(sub_task_id, sub_task.Context.ShardName))
        if flag_failed_tasks:
            raise errors.TaskFailure('Error on building some data.')

        import sandbox.projects.websearch.begemot.tasks.GetBegemotResponses as GetBegemotResponses
        begemot_responses_class = sdk2.Task[GetBegemotResponses.GetBegemotResponses.type]
        get_begemot_responses_tasks = []

        with self.memoize_stage.get_begemot_responses(commit_on_entrance=False):
            shards = self.Context.callable_shards
            shards_list = self.Parameters.shards_list
            requests_plan = self.Parameters.requests_plan
            binaries = self.Parameters.begemot_binary_list
            binaries_names = self.Parameters.begemot_binary_names
            queries = defaultdict(list)
            for i in range(len(shards_list)):
                queries[shards_list[i]].append(requests_plan[i])
            self.verbose("queries:\n%r", queries)
            for i in range(len(shards)):
                list_resource = sdk2.Resource.find(task_id=self.Context.build_begemot_data_tasks[i]).limit(100)

                for resource in list_resource:
                    name_resource = str(resource.type)

                    if not name_resource.startswith('BEGEMOT_FAST_BUILD_CONFIG'):
                        continue
                    for query in queries[shards[i]]:
                        # because of a problem in TestEnv we should take binary this way:
                        bin_index = Begemots[shards[i]].binary if shards[i] != 'Megamind' else 'BEGEMOT'
                        sub_task = begemot_responses_class(
                            self,
                            begemot_binary=binaries[binaries_names.index(bin_index)],
                            fast_build_config=resource,
                            requests_plan=query,
                            ShardName = shards[i]
                        )
                        get_begemot_responses_tasks.append(sub_task.save().enqueue().id)

            self.verbose("get_begemot_responses_tasks:\n%r", get_begemot_responses_tasks)
            self.Context.get_begemot_responses_tasks = get_begemot_responses_tasks

            raise sdk2.WaitTask(self.Context.get_begemot_responses_tasks, Status.Group.FINISH | Status.Group.BREAK, wait_all=True)

        sub_tasks = self.find(begemot_responses_class)
        self.Context.get_begemot_responses_results = [task.Context.out_resource_id for task in sub_tasks]
        self.Context.save()
        flag_failed_tasks = False
        for sub_task_id in self.Context.get_begemot_responses_tasks:
            sub_task = sdk2.Task[sub_task_id]
            if sub_task.status in bad_statuses:
                flag_failed_tasks = True
                self.set_info("GetBegemotResponses task {} for shard {} have failed.".format(sub_task_id, sub_task.Context.ShardName))
        if flag_failed_tasks:
            raise errors.TaskFailure('Error on getting some shard responses.')
