import logging
import requests
import time

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.projects.release_machine.tasks.LaunchMetrics import LaunchMetrics
from sandbox.projects.websearch.begemot.common import Begemots


FILTER_NAME = 'Wizard off'

WARDEN_API_URL = 'https://warden.z.yandex-team.ru/api/warden.Warden/'
BEGEMOT_ALERTS_LINK = 'https://warden.z.yandex-team.ru/components/begemot/alerts'
FUNCTIONALITY_LINK = 'https://warden.z.yandex-team.ru/functionality'

JUGGLER_CHECK_PREFIX = 'https://juggler.yandex-team.ru/check_details/'
YASM_ALERTS_API = 'https://yasm.yandex-team.ru/srvambry/alerts/'
YASM_DESCRIPTION_PREFIX = '[BEGEMOT-2563]'


class WardenApi:

    def __init__(self, oauth, url=WARDEN_API_URL):
        self.url = url
        self.oauth = oauth
        self.session = requests.Session()

    def _request(self, handler, data={}, retries=3, retry_timeout=60):
        for i in range(retries):
            try:
                response = self.session.request(
                    'POST',
                    self.url + handler,
                    headers={
                        'Authorization': 'OAuth {}'.format(self.oauth),
                    },
                    json=data
                )
                return response.json()
            except:
                if i + 1 == retries:
                    response.raise_for_status()
                else:
                    time.sleep(retry_timeout)
        return None

    def _functionality_map(self, name, description, instructions, weight, queue):
        res = {
            'name': name,
            'description': description,
            'weight': weight,
            'instructions': instructions,
        }
        if queue is not None:
            res['targetQueue'] = queue
        return res

    def _alert_info(self, alert_name, alert_url, enable_spi_tools):
        return {
            'name': alert_name,
            'url': alert_url,
            'beholderSettings': {
                'createSpi': enable_spi_tools,
                'startFlow': enable_spi_tools,
                'calculateMetric': enable_spi_tools,
                'calculateBackgroundMetric': enable_spi_tools,
            },
            'category': 6, # Value from https://a.yandex-team.ru/arc/trunk/arcadia/search/mon/warden/proto/structures/alert.proto?rev=r8272693#L40
            'targetValue': 0,
        }

    def getFunctionalityList(self):
        return self._request(
            'getFunctionalityList',
            {'functionality_name': 'Begemot'}
        )['objects']

    def addFunctionality(self, name, description, instructions, weight, component, parent=None, queue=None):
        data = {
            'functionality': self._functionality_map(name, description, instructions, weight, queue),
            'componentName': component
        }
        if parent is not None:
            data['parentComponentName'] = parent

        self._request('addFunctionality', data)

    def updateFunctionality(self, id, name, description, instructions, weight, queue=None):
        func_info = self._functionality_map(name, description, instructions, weight, queue)
        func_info['id'] = id
        data = {
            'functionality': func_info
        }
        self._request('updateFunctionality', data)

    def getComponentAlerts(self, component):
        data = {'componentName': component}
        return self._request('getComponentAlerts', data)['objects']

    def addAlert(self, functionality_id, name, url, enable_spi_tools=False):
        data = {
            'functionalityId': functionality_id,
            'alert': self._alert_info(name, url, enable_spi_tools),
        }
        self._request('addAlert', data)

    def updateAlert(self, alert_id, name, url, comment, enable_spi_tools=False):
        data = {
            'comment': comment,
            'alert': self._alert_info(name, url, enable_spi_tools),
        }
        data['alert']['id'] = alert_id
        self._request('updateAlert', data)


class AlertType:

    '''
    Used to classify existing YASM alert comparing to required alert
    Alerts with 'BEGEMOT-2563' in description can be autoupdated
    The other alerts should be marked as not updatable: task must not spoil Marty alerts
    '''
    SAME_ALERT = 0
    UPDATABLE_ALERT = 1
    NOT_UPDATABLE_ALERT = 2


class YASMAlertsUpdater:

    def __init__(self, itype, abc, url=YASM_ALERTS_API):
        self.itype = itype
        self.abc = abc
        self.url = url
        self.session = requests.Session()
        self.alerts_map = None
        self.errors = {}

    def _request(self, method, handler, params={}, data={}, retries=5, retry_timeout=10):
        for i in range(retries):
            try:
                response = self.session.request(
                    method,
                    self.url + handler,
                    params=params,
                    json=data
                )
                return response.json()
            except:
                if i + 1 == retries:
                    response.raise_for_status()
                else:
                    time.sleep(retry_timeout)
        return None

    # Check if two alerts for same signal are same
    def _compare_alerts(self, alert, existing_alert):
        logging.debug('Comparing yasm alerts:\n{}\n{}'.format(alert, existing_alert))
        has_diff = False

        if len(alert['tags']) != len(existing_alert['tags']):
            has_diff = True
        else:
            for tag in alert['tags']:
                left_set = set(alert['tags'][tag])
                right_tag = existing_alert['tags'].get(tag, [])
                if not isinstance(right_tag, list):
                    right_tag = [right_tag]
                right_set = set(right_tag)

                if len(left_set.symmetric_difference(right_set)) > 0:
                    has_diff = True

        if alert['warn'] != existing_alert['warn'] or alert['crit'] != existing_alert['crit']:
            has_diff = True

        if alert['mgroups'] != existing_alert['mgroups']:
            has_diff = True

        if len(existing_alert.get('juggler_check', {}).get('tags', [])) != len(alert['juggler_check']['tags']):
            has_diff = True

        if not has_diff:
            return AlertType.SAME_ALERT

        if has_diff and YASM_DESCRIPTION_PREFIX in existing_alert.get('description', ''):
            return AlertType.UPDATABLE_ALERT

        return AlertType.NOT_UPDATABLE_ALERT

    def _create_new_alert(self, alert):
        logging.debug('Creating new alert: {}'.format(alert))
        response = self._request(
            'POST',
            'create',
            params={'name': alert['name']},
            data=alert,
        )
        logging.debug('Yasm response: {}'.format(response))
        if response.get('status') != 'ok':
            self.errors[alert] = response

    def _update_alert(self, existing_alert, new_alert):
        logging.debug('Updating alert {}'.format(existing_alert['name']))
        response = self._request(
            'POST',
            'update',
            params={'name': existing_alert['name']},
            data=new_alert,
        )
        logging.debug('Yasm response: {}'.format(response))
        if response.get('status') != 'ok':
            self.errors[alert] = response

    def get_existing_alerts(self, force=False):
        if self.alerts_map is None or force:
            self.alerts_map = {}
            data = {
                'itype': self.itype,
                'abc': self.abc,
            }
            alerts_list = self._request('GET', 'list', params=data)['response']['result']
            for alert in alerts_list:
                geo = alert.get('tags', {}).get('geo', [])
                if isinstance(geo, list):
                    geo = ' '.join(geo)
                signal = alert.get('signal', '')
                key = '{}_{}'.format(geo, signal)
                if key in self.alerts_map:
                    self.alerts_map[key].append(alert)
                else:
                    self.alerts_map[key] = [alert]

        return self.alerts_map

    def add_alert(self, alert):
        geo = alert['tags']['geo']
        if isinstance(geo, list):
            geo = ' '.join(geo)
        key = '{}_{}'.format(geo, alert['signal'])

        most_similar_alert = {}
        most_similar_alert_type = AlertType.NOT_UPDATABLE_ALERT
        if key in self.alerts_map:
            for existing_alert in self.alerts_map[key]:
                alert_type = self._compare_alerts(alert, existing_alert)
                if alert_type <= most_similar_alert_type:
                    most_similar_alert = existing_alert
                    most_similar_alert_type = alert_type

        if most_similar_alert_type == AlertType.NOT_UPDATABLE_ALERT:
            self._create_new_alert(alert)
            return alert['name']
        elif most_similar_alert_type == AlertType.UPDATABLE_ALERT:
            alert['name'] = most_similar_alert['name']
            self._update_alert(most_similar_alert, alert)

        return most_similar_alert['name']

    def get_errors(self):
        return self.errors


class GetBegemotShardsQuality(sdk2.Task):

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.CheckGroup('Begemot shards to test') as shards_to_test:
            for shard_name in sorted(Begemots.Services):
                shard = Begemots[shard_name]
                setattr(
                    shards_to_test.values, shard_name,
                    shards_to_test.Value(shard_name, checked=shard.test_with_merger),
                )

        left_params = sdk2.parameters.String(
            'Left serpset cgi params',
            required=False,
        )

        right_params = sdk2.parameters.String(
            'Right serpset cgi params',
            required=False,
        )

        search_subtype = sdk2.parameters.String(
            'Search subtype',
            required=True,
            default='web'
        )

        template_name = sdk2.parameters.String(
            'Template name',
            required=True,
            default='failcache.json'
        )

        scraper_pool = sdk2.parameters.String(
            'Scraper Over YT pool',
            required=True,
            default='begemot_web_priemka'
        )

        time_to_wait = sdk2.parameters.String(
            'Time interval between starting tasks (seconds)',
            required=True,
            default=3600
        )

        report_to_warden = sdk2.parameters.Bool(
            'Report to warden ({})'.format(FUNCTIONALITY_LINK),
            default=False
        )

        update_alerts = sdk2.parameters.Bool(
            'Update yasm and warden alerts ({})'.format(BEGEMOT_ALERTS_LINK),
            default=False
        )

        with report_to_warden.value[True] or update_alerts.value[True]:
            warden_token = sdk2.parameters.String(
                'Warden token from vault',
                required=True,
                default='Begemot Warden token'
            )

        with update_alerts.value[True]:
            enable_spi_tools = sdk2.parameters.Bool(
                'Create SPI, start flow, calculate metric for all alerts',
                required=True,
                default=False,
            )

            crit_threshold = sdk2.parameters.Float(
                'Crit threshold for all alerts',
                required=True,
                default=2.0,
            )

        email_list = sdk2.parameters.List(
            'Staff logins or mailbox names for email notification',
            required=False,
        )

    def start_child_task(self, shard):
        source = Begemots[shard].apphost_source_name
        failcache_source = Begemots['FailCache'].apphost_source_name
        description = 'Begemot quality test (shard {})'.format(shard)
        task = LaunchMetrics(
                self, description=description,
            binary_executor_release_type='stable',
            metrics_mode_type='custom',
            search_subtype=self.Parameters.search_subtype,
            custom_template_name=self.Parameters.template_name,
            custom_template_descr=description,
            sample_beta='hamster',
            sample_extra_params=self.Parameters.left_params.format(source=source, failcache=failcache_source),
            checked_beta='hamster',
            checked_extra_params=self.Parameters.right_params.format(source=source, failcache=failcache_source),
            scraper_over_yt_pool=self.Parameters.scraper_pool,
        )

        self.Context.child_tasks.append(task.enqueue().id)
        self.Context.shard_by_task[task.id] = shard
        self.set_info("Started child task for shard {}".format(shard))

    def check_child_tasks(self):
        has_fails = False
        for task_id in self.Context.child_tasks:
            shard = self.Context.shard_by_task[str(task_id)]
            task = sdk2.Task.find(id=task_id, children=True).first()
            info = task.Context.launch_info
            if info["status"] != "COMPLETED":
                self.set_info("Shard {}: task finished with status {}".format(shard, info["status"]))
            else:
                diff = None
                try:
                    for serp in info["launches"][0]["serps"]:
                        if FILTER_NAME in serp["name"] and "touch" not in serp["name"] and "tail" not in serp["name"]:
                            for item in serp["metrics"]:
                                if item["metricName"] == 'proxima-v9':
                                    diff = item["diffPercent"]
                except Exception as e:
                    logging.debug('Shard {} error: {}'.format(shard, e))
                    has_fails = True
                if diff is None:
                    self.set_info("Shard {}: not found proxima-v9 value".format(shard))
                    has_fails = True
                else:
                    self.set_info("Shard {}: proxima-v9 diff is {:.2f}%".format(shard, diff))
                    self.Context.quality_loss[shard] = max(0, -diff)

        return not has_fails

    def get_begemot_functionalities(self, warden):
        id_map = {}
        func_list = warden.getFunctionalityList()
        for obj in func_list:
            if 'Begemot' in obj.get('name', ''):
                id_map[obj['name']] = obj['id']
        logging.debug('Found existing warden records: {}'.format(id_map))
        return id_map

    def report_to_warden(self, warden, losses):
        id_map = self.get_begemot_functionalities(warden)

        for shard in losses:
            for location in ['', ' sas', ' man', ' vla']:
                denominator = 100 if location == '' else 300
                weight = round(float(losses[shard]) / denominator, 4)
                descr = 'Begemot {} {} failures'.format(shard, location)
                name = 'Begemot {}{}'.format(shard, location)
                geo = location[1:] if location else 'sas,man,vla'
                instructions = 'Shard YASM panel: https://yasm.yandex-team.ru/template/panel/begemot-source/prj={};geo={};quant=10,99/'.format(Begemots[shard].prj, geo)
                if name in id_map:
                    warden.updateFunctionality(id_map[name], name, descr, instructions, weight)
                    logging.info('Updated warden record for shard {}{}'.format(shard, location))
                else:
                    warden.addFunctionality(name, descr, instructions, weight, 'begemot')
                    logging.info('Created warden record for shard {}{}'.format(shard, location))

    def _get_alert(self, source, shard, geo):
        if shard == 'Merger':
            graph = 'web5b'
        else:
            graph = 'begemot-workers'

        signal = 'perc(unistat-SOURCES-{graph}-{source}-Failures_dmmm,max(sum(unistat-SOURCES-{graph}-{source}-Successes_dmmm,unistat-SOURCES-{graph}-{source}-Failures_dmmm),200))'.format(graph=graph, source=source)
        tags = {
            'geo': [geo],
            'itype': ['apphost'],
            'ctype': ['prestable', 'prod'],
            'prj': ['imgs-main', 'shared', 'video-main-rkub', 'web-main'],
        }

        name = 'Failures-Perc-{}-{}'.format(source, geo)
        result = {
            'name': name,
            'signal': signal,
            'tags': tags,
            'warn': [0.2, self.Parameters.crit_threshold],
            'crit': [self.Parameters.crit_threshold, None],
            'mgroups': ['ASEARCH'],
            'description': YASM_DESCRIPTION_PREFIX + ' Updated by sandbox task #{}'.format(self.id),
            'abc': 'search-wizard',
            'juggler_check': {
                'host': 'yasm_{}'.format(name),
                'service': name,
                'namespace': 'begemot',
                'flaps': {
                    "boost": 0,
                    "critical": 150,
                    "stable": 30
                },
                'tags': [
                    'warden_alert_create_spi',
                    'warden_alert_start_flow',
                    'warden_alert_account_metric',
                    'warden_alert_account_background_metric',
                    'uchenki_service_web',
                    'uchenki_training_stepback',
                ]
            }
        }
        return result

    def get_alert_url(self, yasm, source, shard, location):
        alert_info = self._get_alert(source, shard, location)
        alert_name = yasm.add_alert(alert_info)
        return JUGGLER_CHECK_PREFIX + '?host={}&service={}'.format(
            alert_info['juggler_check']['host'],
            alert_info['juggler_check']['service']
        )

    def update_alerts(self, warden):
        all_sources = {}
        for shard_name in Begemots.Services:
            shard = Begemots[shard_name]
            if shard_name == 'Merger' or shard.test_with_merger:
                for source in shard.all_apphost_sources:
                    all_sources[source] = shard_name

        begemot_alerts = warden.getComponentAlerts('begemot')
        warden_existing_alerts = {}
        for alert in begemot_alerts:
            key = '{}__{}'.format(alert['functionalityName'], alert['alert']['name'])
            warden_existing_alerts[key] = alert['alert']['id']

        yasm = YASMAlertsUpdater('apphost', 'search-wizard')
        yasm_existing_alerts = yasm.get_existing_alerts()

        all_functionalities = self.get_begemot_functionalities(warden)

        for source in all_sources:
            shard = all_sources[source]
            for loc in ['sas', 'man', 'vla']:
                functionality_name = 'Begemot {} {}'.format(shard, loc)
                if functionality_name in all_functionalities:
                    alert_url = self.get_alert_url(yasm, source, shard, loc)
                    alert_name = '[{}] {} failures'.format(loc, source)
                    alert_key = '{}__{}'.format(functionality_name, alert_name)

                    if alert_key in warden_existing_alerts:
                        try:
                            comment = '[BEGEMOT-2563] Updated by task https://sandbox.yandex-team.ru/task/{}'.format(self.id)
                            warden.updateAlert(warden_existing_alerts[alert_key], alert_name, alert_url, comment, self.Parameters.enable_spi_tools)
                            logging.info('Task updated alert {}'.format(alert_key))
                        except:
                            logging.info('Task failed to update alert {}'.format(alert_key))
                    else:
                        try:
                            warden.addAlert(all_functionalities[functionality_name], alert_name, alert_url, self.Parameters.enable_spi_tools)
                            logging.info('Task added alert {}'.format(alert_key))
                        except:
                            logging.info('Task failed to add alert {}'.format(alert_key))
                else:
                    self.set_info('Task failed to update alert for {} in {}. Functionality {} not found'.format(source, loc, functionality_name))

        yasm_errors = yasm.get_errors()
        for alert in yasm_errors:
            self.set_info('Failed to update yasm alert "{}". Yasm response: {}'.format(alert, yasm_errors[alert]))
        if len(yasm_errors):
            raise "Failed to update some yasm alerts"

    def send_email(self, check_shards_ok, warden_report_ok, alerts_update_ok):
        if not self.Parameters.email_list:
            return
        if check_shards_ok and warden_report_ok and alerts_update_ok:
            return

        task_link = '<a href="https://sandbox.yandex-team.ru/task/{}">#{}</a>'.format(self.id, self.id)
        warden_link = '<a href="{}">warden</a>'.format(FUNCTIONALITY_LINK)
        alerts_link = '<a href="{}">warden</a>'.format(BEGEMOT_ALERTS_LINK)

        subject = 'GET_BEGEMOT_SHARDS_QUALITY: problems with task'
        msg = 'Message from task {}'.format(task_link)
        if not check_shards_ok:
            msg = '<br>'.join([
                msg,
                'Task failed to read results from Metrics. Check if main metric name changed'
            ])
        if not warden_report_ok:
            msg = '<br>'.join([
                msg,
                'Task failed to update warden values. You can do it manually ({}) or wait for next GET_BEGEMOT_SHARDS_QUALITY task'.format(warden_link)
            ])
        if not alerts_update_ok:
            msg = '<br>'.join([
                msg,
                'Task failed to update alerts. You can do it manually: {}'.format(alerts_link)
            ])

        self.server.notification(
            subject=subject,
            body=msg,
            recipients=self.Parameters.email_list,
            transport=ctn.Transport.EMAIL,
            urgent=False,
            type='html'
        )

    def on_execute(self):
        with self.memoize_stage["init"](commit_on_entrance=False):
            self.Context.shards_left = self.Parameters.shards_to_test
            self.Context.child_tasks = []
            self.Context.shard_by_task = {}
            self.Context.quality_loss = {}
            self.Context.wait_started = False

        if len(self.Context.shards_left) == 0:
            if not self.Context.wait_started:
                self.Context.wait_started = True
                raise sdk2.WaitTask(self.Context.child_tasks, Status.Group.FINISH | Status.Group.BREAK)
            else:
                check_shards_ok = self.check_child_tasks()
                warden_report_failed, alerts_update_failed = False, False
                if self.Parameters.report_to_warden or self.Parameters.update_alerts:
                    warden = WardenApi(sdk2.Vault.data(self.Parameters.warden_token))
                if self.Parameters.report_to_warden:
                    try:
                        self.report_to_warden(warden, self.Context.quality_loss)
                    except Exception as e:
                        self.set_info('Warden weights update failed with exception: {}'.format(e))
                        warden_report_failed = True

                if self.Parameters.update_alerts:
                    try:
                        self.update_alerts(warden)
                    except Exception as e:
                        self.set_info('Alerts update failed with exception: {}'.format(e))
                        alerts_update_failed = True

                self.send_email(check_shards_ok, not warden_report_failed, not alerts_update_failed)
        else:
            self.start_child_task(self.Context.shards_left[0])
            self.Context.shards_left = self.Context.shards_left[1:]
            raise sdk2.WaitTime(self.Parameters.time_to_wait)
