import logging
import requests
import socket

from urllib3 import Retry

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.websearch.begemot import parameters as bp
from sandbox.projects.websearch.begemot import resources as br
from sandbox.projects.websearch.begemot.common.fast_build import ShardSyncHelper


def reserve_port():
    """
    Find and reserve a free port
    """
    sock = socket.socket(socket.AF_INET6)
    sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    sock.bind(("::", 0))
    port = sock.getsockname()[1]
    return port


class BegemotInterface(object):
    def __init__(self, port):
        self.url = "http://localhost:%d/admin" % port
        retries = {
            "get_maxrss": Retry(
                total=None,
                connect=15,
                read=0,
                backoff_factor=1
            ),
            "shutdown": Retry(
                total=None,
                connect=15,
                read=0,
                backoff_factor=1
            ),
        }
        self.session = {}
        for method, retry_config in retries.items():
            session = self.session[method] = requests.Session()
            session.mount('http://', requests.adapters.HTTPAdapter(max_retries=retry_config))

    def get_maxrss(self):
        session = self.session["get_maxrss"]
        stat = (session.get(self.url, params={"action": "stat"})).json()
        for item in stat:
            if item[0].endswith("-memory-maxrss_axxx"):
                return item[1]
        else:
            raise TaskFailure("Signal memory-maxrss_axxx not found in begemot stat.")

    def shutdown(self):
        session = self.session["shutdown"]
        session.get(self.url, params={"action": "shutdown"})


class GetWorkerMaxrssDifference(sdk2.Task):
    __logger = logging.getLogger('TASK_LOGGER')
    __logger.setLevel(logging.DEBUG)

    class Parameters(sdk2.Parameters):
        fast_build_config = bp.FastBuildConfigResource()
        fresh_old = bp.FreshResource()
        fresh_new = bp.FreshResource()
        begemot_binary = bp.BegemotExecutableResource()
        bstr_callback_binary = sdk2.parameters.Resource(
            "Bstr callback binary",
            resource_type=br.BEGEMOT_BSTR_CALLBACK,
            required=True,
        )
        with sdk2.parameters.Output:
            maxrss_diff = sdk2.parameters.Integer("Difference in usage of memory between new fresh and old fresh (in bytes)", default=0)

    class Requirements(sdk2.Requirements):
        privileged = True

    def get_fresh_size(self, res):
        if "FAST_BUILD" in res.type.name:
            return ShardSyncHelper(res).get_shard_size()
        return res.size

    def get_fresh_path(self, res, suffix=""):
        if "FAST_BUILD" not in res.type.name:
            return str(sdk2.ResourceData(res).path)
        shard_helper = ShardSyncHelper(res)
        data_path = str(self.path("fresh{}".format(suffix)))
        return shard_helper.sync_shard(data_path)

    def on_enqueue(self):
        try:
            shard_id = int(self.Parameters.fast_build_config)
            shard_size = ShardSyncHelper(sdk2.Resource.find(self.Parameters.fast_build_config).first()).get_shard_size()
        except:
            shard_size = ShardSyncHelper(self.Parameters.fast_build_config).get_shard_size()
        data_and_binary_size_mb = (self.Parameters.begemot_binary.size + shard_size +
                                   self.get_fresh_size(self.Parameters.fresh_old) + self.get_fresh_size(self.Parameters.fresh_new) +
                                   self.Parameters.bstr_callback_binary.size) >> 20
        self.Requirements.disk_space = self.Requirements.ram = 5 * 1024 + 2 * data_and_binary_size_mb  # 5 GB + size of input resources

    def on_execute(self):
        begemot_executable = str(sdk2.ResourceData(self.Parameters.begemot_binary).path)
        bstr_callback_executable = str(sdk2.ResourceData(self.Parameters.bstr_callback_binary).path)
        fast_build_config = str(sdk2.ResourceData(self.Parameters.fast_build_config).path)
        shard_helper = ShardSyncHelper(self.Parameters.fast_build_config)
        data_path = str(self.path('data'))
        shard = shard_helper.sync_shard(data_path)
        port = reserve_port()
        begemot_args = [
            begemot_executable,
            "--port", str(port),
            "--grpc", str(reserve_port()),
            "--data", str(shard),
            "--fresh", self.get_fresh_path(self.Parameters.fresh_old, "_old"),
            "--mlock", 'yes'
        ]
        self.__logger.info("Starting begemot daemon")
        with sdk2.helpers.ProcessLog(self, logger="begemot") as begemot_pl:
            try:
                self.__logger.info("Setting memlock unlimited")
                subprocess.check_call("ulimit -l unlimited", shell=True)
                subprocess.Popen(begemot_args, stdout=begemot_pl.stdout, stderr=subprocess.STDOUT)
                begemot = BegemotInterface(port)
                maxrss_old = begemot.get_maxrss()
                self.__logger.info("Memory usage with old fresh = {} bytes".format(maxrss_old))
                self.__logger.info("Reloading fresh")
                bstr_callback_args = [
                    bstr_callback_executable,
                    self.get_fresh_path(self.Parameters.fresh_new, "_new"),
                    "--slow-data-fastbuild", fast_build_config,
                    "reload",
                    "--port", str(port),
                    "--reload-daemon", "begemot",
                ]
                with sdk2.helpers.ProcessLog(self, logger="bstr_callback") as bstr_callback_pl:
                    subprocess.check_call(bstr_callback_args, stdout=bstr_callback_pl.stdout, stderr=subprocess.STDOUT)
                maxrss_new = begemot.get_maxrss()
                self.__logger.info("Memory usage with new fresh = {} bytes".format(maxrss_new))
                self.Parameters.maxrss_diff = maxrss_new - maxrss_old
            finally:
                self.__logger.info("Shutdown begemot daemon")
                begemot.shutdown()
