import json
import logging
import os

from sandbox.projects.websearch.begemot import AllBegemotServices
import sandbox.projects.websearch.begemot.resources as br

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess


def get_rulename(path):
    return os.path.basename(path).strip("\n")

class MergeBegemotFreshRevisions(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        config = sdk2.parameters.JSON(
            "Config",
            description="New and old resources for unpacked fresh and cypress shards",
            required=True
        )

    def merge_cypress_shards(self):
        for pair in self.Context.cypress_shards:
            old = sdk2.Resource.find(id=pair['old']).first()
            new = sdk2.Resource.find(id=pair['new']).first()
            merge = br.BEGEMOT_CYPRESS_SHARD(
                self,
                new.description,
                new.shard_name + "_merged.txt",
                shard_name=new.shard_name,
                is_fresh=True,
                is_broken=False
            )
            old_paths = {}
            with open(str(sdk2.ResourceData(old).path), "r") as old_input:
                for line in old_input.readlines():
                    rule = get_rulename(line)
                    if rule in self.Context.rules_to_exclude:
                        old_paths[rule] = line
            with open(str(sdk2.ResourceData(new).path), "r") as new_input:
                with open(str(sdk2.ResourceData(merge).path), "w") as output:
                    for line in new_input.readlines():
                        rule = get_rulename(line)
                        if rule in old_paths:
                            logging.debug("Get rule {} from old revision of shard {}".format(rule, merge.shard_name))
                            output.write(old_paths[rule])
                        else:
                            output.write(line)
            sdk2.ResourceData(merge).ready()

    def merge_fast_build_configs(self):
        out_resources = {}
        revisions = {}
        out_dir = os.path.join(str(self.path()), "fast_build")
        if not os.path.exists(out_dir):
            os.mkdir(out_dir)

        for pair in self.Context.fast_build_configs:
            old = sdk2.Resource.find(id=pair['old']).first()
            new = sdk2.Resource.find(id=pair['new']).first()
            with open(str(sdk2.ResourceData(old).path), "r") as f:
                old_json = json.load(f)
            with open(str(sdk2.ResourceData(new).path), "r") as f:
                new_json = json.load(f)

            shard = new_json['shard_name']
            if not os.path.exists(os.path.join(out_dir, shard)):
                os.mkdir(os.path.join(out_dir, shard))
            merge_path = os.path.join(out_dir, shard, 'fast_build_config.json')
            merged_json ={
                'shard_name': new_json['shard_name'],
                'resources': [],
                'version_info': new_json['version_info']
            }

            old_res = {}
            for res in old_json['resources']:
                if res['name'] in self.Context.rules_to_exclude:
                    old_res[res['name']] = res

            data_size = 0
            for res in new_json['resources']:
                if res['name'] in self.Context.rules_to_exclude:
                    if res['name'] in old_res:
                        merged_json['resources'].append(old_res[res['name']])
                        data_size += old_res[res['name']]['resource_size_kb']
                        revisions[res['name']] = old_res[res['name']].get('arcadia_revision', 'No data')
                else:
                    merged_json['resources'].append(res)
                    data_size += res['resource_size_kb']
                    revisions[res['name']] = res.get('arcadia_revision', 'No data')

            with open(merge_path, "w") as f:
                new_json = json.dump(merged_json, f)

            resource = AllBegemotServices[shard].fresh_fast_build_config_resource_type(self, new.description, merge_path, data_size_kb=data_size)
            out_resources[resource.type.name] = resource.id
        self.Context.out_resources = out_resources

        latest_revision = max(revisions.values())
        for rule in revisions:
            if rule not in self.Context.rules_to_exclude:
                revisions[rule] = latest_revision
        self.Context.revisions_info = revisions

    def merge_fresh_resources(self):
        out_resources = {}
        unpacked_dir = os.path.join(str(self.path()), "unpacked")
        packed_dir = os.path.join(str(self.path()), "packed")
        if not os.path.exists(unpacked_dir):
            os.mkdir(unpacked_dir)
        if not os.path.exists(packed_dir):
            os.mkdir(packed_dir)

        for pair in self.Context.unpacked_fresh:
            old = sdk2.Resource.find(id=pair['old']).first()
            new = sdk2.Resource.find(id=pair['new']).first()
            old_dir = os.listdir(str(sdk2.ResourceData(old).path))
            new_dir = os.listdir(str(sdk2.ResourceData(new).path))

            # Input resources may have different folder structures, but they always have a shard name in path
            for d in str(old.path).split('/'):
                if d in [s.name for s in AllBegemotServices.Service.values()]:
                    shard = d
                    break

            shard_dir = os.path.join(unpacked_dir, shard)
            shard_packed_dir = os.path.join(packed_dir, shard)
            packed_path = os.path.join(shard_packed_dir, "fresh.tar")
            if not os.path.exists(shard_dir):
                os.mkdir(shard_dir)
            if not os.path.exists(shard_packed_dir):
                os.mkdir(shard_packed_dir)
            for d in new_dir:
                if d in self.Context.rules_to_exclude:
                    if d in old_dir:
                        subprocess.check_call([
                            "cp", "-r", "-p",
                            os.path.join(str(sdk2.ResourceData(old).path), d),
                            os.path.join(shard_dir, d)
                        ])
                else:
                    subprocess.check_call([
                        "cp", "-r", "-p",
                        os.path.join(str(sdk2.ResourceData(new).path), d),
                        os.path.join(shard_dir, d)
                    ])
            subprocess.check_call(["chmod", "ug+rw", "-R", shard_dir])
            subprocess.check_call(['tar', '-cf', packed_path, '.'], cwd=shard_dir)

            res_unpacked = AllBegemotServices[shard].fresh_data_resource_type(self, shard, shard_dir)
            res_packed = AllBegemotServices[shard].fresh_data_resource_packed_type(self, shard, packed_path)
            sdk2.ResourceData(res_unpacked).ready()
            sdk2.ResourceData(res_packed).ready()
            out_resources[res_unpacked.type.name] = res_unpacked.id
            out_resources[res_packed.type.name] = res_packed.id
        self.Context.out_resources = out_resources

    def on_enqueue(self):
        self.Context.cypress_shards = self.Parameters.config["cypress_shards"]
        self.Context.unpacked_fresh = self.Parameters.config.get("unpacked_fresh", None)
        self.Context.fast_build_configs = self.Parameters.config.get("fast_build_configs", None)
        self.Context.rules_to_exclude = self.Parameters.config["rules_to_exclude"]

    def on_execute(self):
        self.merge_cypress_shards()
        if self.Context.fast_build_configs is not None:
            self.merge_fast_build_configs()
        else:
            self.merge_fresh_resources()
