# -*- coding: utf-8 -*-
import datetime as dt
import json
import logging
import os
import time
import traceback

import sandbox.common.types.notification as ctn
import sandbox.projects.common.wizard.wizard_builder as wb
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import NotExists
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.websearch.begemot import AllBegemotServices, resources
from sandbox.projects.websearch.begemot.common import Begemots
from sandbox.projects.websearch.begemot.tasks.BuildBegemotCommon import BuildBegemotCommon
from sandbox.projects.websearch.begemot.tasks.TestBegemotFresh import TestBegemotFresh, TestBegemotFreshParams, FRESH_SVN_URL
from sandbox.projects.websearch.upper.fast_data.DeployFastData import DeployFastData
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sdk2.helpers import subprocess


NANNY_API_URL = 'http://nanny.yandex-team.ru'
SAMOGON_API_URL = 'http://clusterapi-fresh.n.yandex-team.ru'
YT_PREFIX = '//home/search-runtime/begemot-fresh'
YT_FAST_DATA_PREFIX = '//home/search-runtime/fast-data/begemot'
_RUNTIME_URL = 'arcadia:/robots/trunk/wizard-data/'

cache_affecting_rrr = {
    'Video': ['SerialStruct', 'lastEpisode', 'lastAirdate', 'lastAirdateProb', 'lastSeason',
              'recentlyReleasedEpisodes'],
    'EntityFinder': ['MatchesExport'],
}

WARNING_EMAIL_RECIPIENTS = ['ageraab', 'gluk47']


def id_list(resources_list):
    return [r.id if r is not None else None for r in resources_list]

def build_fast_data_config(services, yt_path, prepare_dl=0.3, parallel=False, special_configs={}, infra_config={}, infra_description=''):
    services_config = {}
    dependencies_config = {}
    DEFAULT_OPERATING_DL = 0.3
    DEFAULT_STOP_DL = 0.1
    RETRY_DELAY = 300
    for name in services:
        geo = name[-3:]
        if geo not in ['sas', 'man', 'vla']:
            geo = None

        service_config = special_configs.get(name, {})
        activate = {
            "retry_delay": service_config.get("retry_delay", RETRY_DELAY),
            "degrade_level": service_config.get("activate_degrade_level", DEFAULT_OPERATING_DL),
            "stop_degrade_level": service_config.get("stop_degrade_level", DEFAULT_STOP_DL),
            "failed_retry_delay": 600,
        }
        prepare = {
            "retry_delay": service_config.get("retry_delay", RETRY_DELAY),
            "degrade_level": service_config.get("prepare_degrade_level", prepare_dl),
            "stop_degrade_level": service_config.get("stop_degrade_level", DEFAULT_STOP_DL),
            "failed_retry_delay": 600,
        }
        services_config[name] = {
            'activate': activate,
            'prepare': prepare,
        }

        if infra_config.get(geo, None) is not None:
            services_config[name]['infra'] = {
                'service_id': infra_config[geo]['service'],
                'environment_id': infra_config[geo]['environment'],
            }

        prev_service = None
        if geo == 'man':
            prev_service = name[:-4] + '_sas'
        if geo == 'vla':
            prev_service = name[:-4] + '_sas'
        if not parallel and prev_service is not None and prev_service in services:
            dependencies_config[name] = {
                'prepare': [{
                    'action': 'prepare',
                    'name': prev_service,
                }],
                'activate': [{
                    'action': 'activate',
                    'name': prev_service,
                }],
            }

    communication_config = {
        'cluster': 'locke',
        'cypress_dir': yt_path,
    }
    deploy_config = {
        'only_prepare': False,
        'dependencies': dependencies_config,
    }
    if len(infra_config):
        deploy_config['infra'] = {
            'title': 'Begemot fresh',
            'description': infra_description,
        }

    config = {
        'services': services_config,
        'communication': communication_config,
        'deploy': deploy_config,
    }
    return config

class ReleaseBegemotFresh(TestBegemotFresh):
    __logger = logging.getLogger('TASK_LOGGER')
    __logger.setLevel(logging.DEBUG)

    class Parameters(TestBegemotFreshParams):
        acceptable_responses_change = sdk2.parameters.Float(
            'Begemot responses difference percent share acceptable for auto-release, e.g. 7 (means 7%), max 20%',
            description='Old and new begemot responses of each worker separately will be compared. '
                        'If any worker changes more than threshold% responses, the release will fail. '
                        'This value will be reduced to 20% if it is more than 20%, see SPI-6220.',
            default=7,
        )
        second_attempt_threshold = sdk2.parameters.Float(
            'Second attempt min threshold, %',
            description='For all rules with diff lower that this value, a second attempt of release will be made. The rules with greater diff will be added if sum of all diffs does not exceed the overall threshold.',
            default=1
        )
        do_release = sdk2.parameters.Bool('Release this task to stable in the end', default=True)
        nanny_token = sdk2.parameters.String('Nanny token', default='Begemot Nanny token', required=True)
        deployer_service = sdk2.parameters.String('Nanny deployer service', required=True)
        services = sdk2.parameters.List('Nanny services', description='Begemot services where to deploy fresh', required=True)
        fail_on_any_error = False
        deploy_configs = sdk2.parameters.JSON('Rewrite deploy configs for services', required=False)
        deployer_ncpu = sdk2.parameters.Integer('CPU count for child deployer task', required=True, default=2)
        infra_config = sdk2.parameters.JSON('Infra service and environments', required=False)

    class Context(TestBegemotFresh.Context):
        pkg_id = None
        deploy_start_ts = None
        fresh_tested = False
        deploy_attempts_left = 3
        fast_data_deploy_task = 0

    @classmethod
    def get_nanny_token(cls):
        return sdk2.Vault.data('BEGEMOT', 'Begemot Nanny token')

    def validate_ulimit(self):
        tidiness = str(sdk2.ResourceData(sdk2.Resource.find(id=585550851)).path)  # web/daemons/scripts/nanny_tidiness
        env = os.environ.copy()
        env['OAUTH_NANNY'] = self.get_nanny_token()
        output = subprocess.check_output([tidiness, 'ulimit'], env=env)
        if output:
            self.set_info(output)
            eh.check_failed('Memlock unlimited check failed')

    def deploy_with_fast_data(self, services, deployer):
        description = '\n'.join([
            'Fresh release task: https://sandbox.yandex-team.ru/task/{}/view'.format(self.id),
            'Scheduler: https://sandbox.yandex-team.ru/scheduler/{}'.format(self.scheduler) if self.scheduler else 'Not from scheduler, manual run',
            'Rollback here: https://nanny.yandex-team.ru/ui/#/services/catalog/{}/'.format(deployer) if deployer else '',
            'Responsibles: ageraab@, gluk47@',
        ])
        config = build_fast_data_config(services, YT_FAST_DATA_PREFIX, special_configs=self.Parameters.deploy_configs, infra_config=self.Parameters.infra_config, infra_description=description)
        self.Context.fast_data_config = config
        task = DeployFastData(
            self,
            fast_data_bundle=sdk2.Resource.find(id=self.Context.common_config).first(),
            deployer_mode='nanny_service',
            nanny_service=deployer,
            yt_token_name=self.Parameters.yt_token_vault_name,
            nanny_token_name=self.Parameters.nanny_token,
            infra_token_owner='SEARCH_RELEASERS',
            infra_token_name='Begemot Infra token',
            deploy_config=config,
            kill_timeout=60 * 50  # 50 min
        )
        sdk2.Task.server.task[task.id].update(
            requirements={'ncpu': self.Parameters.deployer_ncpu}
        )
        task.enqueue()
        self.Context.pending_tasks.append(task.id)
        return task.id

    def check_fast_data_deploy_task(self):
        fast_data_task = self.Context.fast_data_deploy_task
        if fast_data_task == 0:
            return False
        task = sdk2.Task.find(id=fast_data_task).first()
        if task.status == Status.SUCCESS:
            return True
        if task.status == Status.EXCEPTION:
            return False
        raise TaskFailure('Fast data deploy task finished with status {}'.format(task.status))

    def get_min_free_memory(self, shard_name):
        from yasmapi import GolovanRequest
        period = 300
        geo = ['sas', 'man', 'vla']
        service = AllBegemotServices.Service[shard_name]
        res = 1 << 50
        self.__logger.info('Getting free memory for shard %s:' % shard_name)
        for g in geo:
            golovan_request_prefix = 'itype=begemot;ctype=prestable,prod;geo=%s;prj=%s:' % (g, service.prj)
            limit_sig = golovan_request_prefix + 'min(portoinst-memory_guarantee_slot_hgram)'
            usage_sig = golovan_request_prefix + 'begemot-WORKER-memory-current-rss_axxx'
            signals = [limit_sig, usage_sig]
            et = time.time() - period * 5
            st = et - period * 5
            memory_limit = 0
            memory_usages = []
            for timestamp, values in GolovanRequest('ASEARCH', period, st, et, signals):
                memory_limit = max(memory_limit, int(values[limit_sig]))
                memory_usages += [values[usage_sig]]
            memory_usage = sorted(memory_usages)[2]
            self.__logger.info('Geo = %s, Memory limit = %d, memory usage = %d' % (g, memory_limit, memory_usage))
            res = min(res, memory_limit - memory_usage)
        return res

    def get_rules_to_exclude(self, stats, answers_count):
        diff_sum = 0
        rule_threshold_diff = answers_count * self.Parameters.second_attempt_threshold / 100.0
        sum_threshold = answers_count * self.Parameters.acceptable_responses_change / 100.0
        rules_to_exclude = []
        rules_to_include = []
        non_excludable_rules = ['QueryFactors', 'qtree', 'pron', '<unknown rule>']
        for record in stats:
            if record['count'] < rule_threshold_diff or record['rule'] in non_excludable_rules:
                rules_to_include.append(record['rule'])
                diff_sum += record['count']
        for record in reversed(stats):
            if record['rule'] not in rules_to_include:
                if diff_sum + record['count'] < sum_threshold:
                    rules_to_include.append(record['rule'])
                    diff_sum += record['count']
                else:
                    rules_to_exclude.append(record['rule'])
        return rules_to_exclude

    def check_tests_result(self, after_merge=False):
        if self.Context.problem_rules is not NotExists and len(self.Context.problem_rules):
            if after_merge:
                eh.check_failed("Begemot reducer failed twice. Rules failed: {}".format(self.Context.problem_rules))
            else:
                self.Context.rules_to_exclude = self.Context.problem_rules
                self.Context.try_merge = True
                self.Context.problem_rules = []
                self.set_info("Trying to release fresh without following rules: {}".format(self.Context.rules_to_exclude))
                return
        if self.Parameters.acceptable_responses_change == 50000:
            acceptable_responses_change = 50  # force release, caution
        else:
            acceptable_responses_change = min(self.Parameters.acceptable_responses_change, 20)  # SPI-6220
        extra_memory = 1 << 30  # 1 GB
        for shard_name, task_id in self.Context.maxrss_tasks.items():
            message = 'Task: https://sandbox.yandex-team.ru/task/%d/view\n' % self.id
            prj = AllBegemotServices.Service[shard_name].prj
            to_time = int(time.time() * 1000)
            from_time = to_time - 24 * 60 * 60 * 1000
            message += ('Link to Golovan: https://yasm.yandex-team.ru/chart/signals=%%7B'
                        'div(begemot-WORKER-memory-maxrss_axxx,1073741824),'
                        'div(portoinst-memory_limit_gb_tmmv,counter-instance_tmmv)%%7D;'
                        'hosts=ASEARCH;itype=begemot;ctype=prestable,prod;'
                        'prj=%s/?by=geo&from=%d&to=%d' % (prj, from_time, to_time))
            try:
                free_memory = self.get_min_free_memory(shard_name)
                if self.Context.maxrss_diff[shard_name] + extra_memory > free_memory:
                    message += 'Free memory = %d MB; ' % (free_memory >> 20)
                    message += 'Needed memory = %d MB; ' % ((self.Context.maxrss_diff[shard_name] + extra_memory) >> 20)
                    message += 'Lack of memory = %d MB.\n' \
                        % ((self.Context.maxrss_diff[shard_name] + extra_memory - free_memory) >> 20)
                    self.server.notification(
                        subject='ReleaseBegemotFresh: not enough memory on hosts for worker %s' % shard_name,
                        body=message,
                        recipients=WARNING_EMAIL_RECIPIENTS,
                        transport=ctn.Transport.EMAIL,
                    )
                    eh.check_failed('Not enough memory on hosts to reload fresh for worker {}. Will not deploy fresh.'.format(shard_name))
            except Exception as e:
                self.set_info(e)
                self.server.notification(
                    subject='ReleaseBegemotFresh: unable to get memory statistics for worker %s' % shard_name,
                    body=message,
                    recipients=WARNING_EMAIL_RECIPIENTS,
                    transport=ctn.Transport.EMAIL,
                )

        diff = self.Context.cache_guess_stats['answers_diff'] * 100
        cache_guess_old = self.Context.cache_guess_stats['cache_guess_old']
        cache_guess_new = self.Context.cache_guess_stats['cache_guess_new']
        cache_guess_change = abs(cache_guess_new - cache_guess_old)
        eh.verify(
            cache_guess_change <= 0.02,
            'Middle search cache hit guess changed by %.2f > 2%%' % cache_guess_change * 100,
        )
        self.__logger.info(
            'Answers diff: %f%%, cache_guess_old: %.2f, cache_guess_new: %.2f',
            diff, cache_guess_old, cache_guess_new,
        )
        if after_merge:
            eh.verify(
                diff <= acceptable_responses_change,
                'Begemot answers diff is %.2f > %.2f. Failed to release fresh without rules %s'
                % (diff, acceptable_responses_change, self.Context.rules_to_exclude)
            )
        elif diff > acceptable_responses_change:
            stats = self.Context.cache_guess_stats
            if not stats["diff_parsed"]:
                eh.check_failed("Begemot answers diff is %.2f > %.2f. Cache hit guess task could not parse diff" % (diff, acceptable_responses_change))
            self.Context.rules_to_exclude = self.get_rules_to_exclude(stats["diff_rules"], stats["answers_count"])
            for rule in self.Context.rules_to_exclude:
                self.Context.problem_tasks[rule] = self.Context.cache_guess_2
            self.set_info("Releasing fresh without rules: [{}]".format(', '.join(self.Context.rules_to_exclude)))
            self.Context.try_merge = True
        else:
            self.Context.try_merge = False

    def get_rules_versions(self):
        task = sdk2.Task.find(id=self.Context.new_runtime_task, children=True).first()
        rev = task.Context.ap_arcadia_revision
        rules = task.Context.rules_list
        return {rule: rev for rule in rules}

    def on_execute(self):
        self.check_pending_tasks()
        try:
            self.validate_ulimit()
        except Exception as e:
            self.__logger.info(e)

        released_revision = int(resources.BEGEMOT_FRESH_DATA_FOR_WIZARD.find(attrs=dict(released='stable')).order(-sdk2.Resource.id).first().runtime_data)
        with self.memoize_stage.run_build_task(commit_on_entrance=False):
            self.Context.new_runtime_task = self.Parameters.prebuilt_fresh
            if self.Context.new_runtime_task is None:
                self.Context.wizard_task = wb.WizardBuilder.get_production_wizard_task().id
                self.Context.revision = int(Arcadia.info(_RUNTIME_URL)['commit_revision'])
                if released_revision > self.Context.revision:
                    self.set_info('New runtime data has been already released')
                    return
                commits = len(Arcadia.log(
                    url=_RUNTIME_URL, revision_from=released_revision + 1, revision_to=self.Context.revision,
                ))
                self.__logger.info('Number of commits = %s', commits)
                arcadia_revision = Arcadia.info(FRESH_SVN_URL)['commit_revision']
                arcadia_url = 'arcadia:/arc/trunk/arcadia'
                if self.Parameters.fast_build:
                    arcadia_url += '@{}'.format(arcadia_revision)

                fresh_shards = [name for name, s in Begemots if s.release_fresh]
                description_for_child = 'Created by RELEASE_BEGEMOT_FRESH task #{}'.format(self.id)

                try:
                    arc_arcadia_url = 'arcadia-arc:/#r{}'.format(arcadia_revision)
                    task = BuildBegemotCommon(
                        self,
                        description=description_for_child,
                        owner=self.owner,
                        binary_executor_release_type='stable',
                        checkout_arcadia_from_url=arc_arcadia_url,
                        shards_to_build=fresh_shards,
                        build_fresh=True,
                        all_in_one_config=True,
                        use_full_shard_build=not self.Parameters.fast_build,
                        use_fast_build=self.Parameters.fast_build,
                        separate_build=True,
                        parent_task_id=self.id,
                        use_arc_instead_of_aapi=True
                    )
                    self.Context.new_runtime_task = task.enqueue().id
                except Exception as e:
                    logging.error('Cannot start new binary build task: {}'.format(e))
                    task = sdk2.Task['BUILD_BEGEMOT_DATA']
                    self.Context.new_runtime_task = task(
                        self,
                        ShardName=' '.join(fresh_shards),
                        BuildFresh=True,
                        AllInOneConfig=self.Parameters.common_config,
                        UseFullShardBuild=not self.Parameters.fast_build,
                        UseFastBuild=self.Parameters.fast_build,
                        SeparateFreshBuild=True,
                        checkout_arcadia_from_url=arcadia_url,
                        description=description_for_child,
                    ).enqueue().id

                self.Context.pending_tasks.append(self.Context.new_runtime_task)
                raise sdk2.WaitTask(self.Context.pending_tasks, Status.Group.FINISH | Status.Group.BREAK)

        if not self.Context.try_merge and not self.Context.fresh_tested:
            self.run_all_tests("first_attempt")
            self.check_tests_result()
            self.Context.fresh_tested = True

        with self.memoize_stage.merge_fresh_revisions(commit_on_entrance=False):
            if self.Context.try_merge:
                self.Context.fresh_tested = False
                self.Context.before_merge = {
                    "fresh_resources": self.Context.fresh_resources,
                    "fresh_torrents": self.Context.fresh_torrents,
                    "new_runtime_task" : self.Context.new_runtime_task
                }
                fresh_merge_config = self.Context.resources_pairs
                fresh_merge_config["rules_to_exclude"] = self.Context.rules_to_exclude
                self.Context.new_runtime_task = sdk2.Task['MERGE_BEGEMOT_FRESH_REVISIONS'](
                    self,
                    description="Trying to build fresh without rules %s" % self.Context.rules_to_exclude,
                    config=fresh_merge_config,
                    type='html',
                    inherit_notifications=True
                ).enqueue().id
                self.Context.pending_tasks.append(self.Context.new_runtime_task)
                raise sdk2.WaitTask(self.Context.pending_tasks, Status.Group.FINISH | Status.Group.BREAK)

        if self.Context.try_merge and not self.Context.fresh_tested:
            self.run_all_tests("second_attempt")
            self.check_tests_result(after_merge=True)
            self.Context.fresh_tested = True

        with self.memoize_stage.release(commit_on_entrance=False):
            if self.Context.try_merge:
                msg = 'The task <a href="{url}">{id}</a> is releasing fresh without rules [{rules}]'.format(
                    url=get_task_link(self.id),
                    id=self.id,
                    rules=', '.join(self.Context.rules_to_exclude)
                )
                self.server.notification(
                    subject='ReleaseBegemotFresh: the task releases fresh without some rules',
                    body=msg,
                    recipients=WARNING_EMAIL_RECIPIENTS,
                    transport=ctn.Transport.EMAIL,
                    urgent=False,
                    type='html'
                )
            if self.Parameters.do_release:
                self.server.release(
                    task_id=self.Context.new_runtime_task,
                    type=ReleaseStatus.STABLE,
                    subject='Wizard.runtime data',
                    comments=self.Context.cache_guess_info
                )

        with self.memoize_stage.deploy(commit_on_entrance=False, commit_on_wait=False):
            deployed = self.check_fast_data_deploy_task()
            if not deployed and self.Context.deploy_attempts_left > 0:
                self.Context.deploy_attempts_left -= 1
                self.Context.fast_data_deploy_task = self.deploy_with_fast_data(self.Parameters.services, self.Parameters.deployer_service)
                raise sdk2.WaitTask(self.Context.pending_tasks, Status.Group.FINISH | Status.Group.BREAK)
            elif not deployed:
                raise TaskFailure('Deploy failed after 3 attempts')

        with self.memoize_stage.write_to_wiki(commit_on_entrance=False):
            WIKI_API = 'https://wiki-api.yandex-team.ru/_api/frontend/'
            WIKI_PAGE = 'begemot/fresh/monitoring/'
            try:
                rules_versions = self.get_rules_versions()
                revision = max(rules_versions.values())
                frozen_rules = dict([pair for pair in rules_versions.items() if pair[1] != revision])

                def get_table_header(print_reasons):
                    return "||**Rule**|**Revision**|**Time (UTC+3)**{}||".format("|**Freeze reason**" if print_reasons else "")

                def get_msk_datetime(utc_str):
                    old_dt = dt.datetime.strptime(utc_str, '%Y-%m-%dT%H:%M:%S.%fZ')
                    new_dt = old_dt + dt.timedelta(hours=3)
                    return new_dt.strftime('%Y-%m-%d %H:%M.%S')

                def get_wiki_version_table(title, versions, print_reasons=False):
                    return [title, "#|", get_table_header(print_reasons)] + ["||{}|r{}|{}{}||".format(
                        k,
                        v,
                        get_msk_datetime(Arcadia.info('arcadia:/arc/trunk/arcadia@{}'.format(v))['date']),
                        "" if not print_reasons else "|See task ((https://sandbox.yandex-team.ru/task/{id}/view {id}))".format(id=self.Context.problem_tasks.get(k, self.id))
                    ) for k, v in sorted(versions.items())] + ["|#"]

                new_body = [
                    "Page updated by task ((https://sandbox.yandex-team.ru/task/%s/view %s))" % (self.id, self.id),
                    "Current fresh revision: r%s" % revision
                ]
                if frozen_rules:
                    new_body.extend(get_wiki_version_table("Frozen_rules:", frozen_rules, print_reasons=True))
                new_body.extend(get_wiki_version_table("Versions of rules:", rules_versions))
                new_body = '\n'.join(new_body)

                new_page = {'title': 'Begemot fresh rules revisions', 'body': new_body}
                oauth = sdk2.Vault.data('BEGEMOT', 'Begemot wiki token')
                headers = {
                    'Authorization': 'OAuth %s' % oauth,
                    'Content-Type': 'application/json'
                }
                response = requests_wrapper.post(WIKI_API + WIKI_PAGE, headers=headers, data=json.dumps(new_page))
                logging.debug("Response from Wiki API: {}".format(response.text))
            except Exception as e:
                self.set_info(traceback.format_exc())
                self.set_info("Task failed to update wiki page. Exception text: {}".format(e))
