# -*- coding: utf-8 -*-
import re
import os
import uuid
import json
import logging
import requests
import shutil

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from collections import defaultdict
from sandbox.projects.WizardRuntimeBuild.ya_make import YaMake
from sandbox.projects.WizardRuntimeBuild.params import WizardRuntimeBuildParams
from sandbox.projects.common import resource_selectors
from sandbox.projects.common import utils
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.constants import ARCADIA_URL_KEY
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.websearch.begemot import AllBegemotServices
from sandbox.projects.websearch.begemot import resources
from sandbox.projects.websearch.upper.resources import FastDataDeployer
from sandbox.projects.websearch.begemot import BW_RESOURCES
from sandbox.projects.websearch.begemot.common import Begemots
from sandbox.projects.websearch.begemot.tasks.BuildBegemotCommon import BuildBegemotCommon
from sandbox.sandboxsdk.svn import Arcadia
from sandbox import sdk2


NANNY_API_URL = 'http://nanny.yandex-team.ru'
SANDBOX_URL = 'https://proxy.sandbox.yandex-team.ru/'
# These resources from the dict above and from the list of shards
#  are not built when building 'search_begemot'
RELEASED_SEPARATELY = (
    'WizardRuntime',
    'BegemotFresh',
    'BegemotFreshFastBuild',
    'BegemotFreshFastBuildCommon',

    'AdvqExecutable',
    'MegamindExecutable',
    'BegginsExecutable',
    'RequestInitExecutable',
    'CaesarModelsExecutable',
)

ACCEPT_MISSING_IN_RELEASE = {
    'RTC_SIMPLE_LIBCUDA_LAYER'  # Released very manually as of now
}

RESOURCE_TO_CREATE = {
    'BegemotFresh',
    'BegemotFreshFastBuild',
    'BegemotFreshFastBuildCommon',
}


class CollectBegemotResources2(sdk2.Task):

    class Requirements(sdk2.Requirements):
        disk_space = 1024

    class Parameters(sdk2.Task.Parameters):
        needed_revision = sdk2.parameters.Integer("Revision of all resources", required=True)
        all_resources = sdk2.parameters.Bool("Release search begemot (binaries + slow data)", default=False)
        with all_resources.value[False]:
            with sdk2.parameters.CheckGroup('Select resources', required=True) as begemot_resources:
                for resource in list(Begemots.keys()) + BW_RESOURCES.keys():
                    setattr(begemot_resources.values, resource, begemot_resources.Value(resource, checked=False))
        checkout_arcadia_from_url = build_params.ArcadiaUrl()
        arcadia_patch = build_params.ArcadiaPatch()
        build_type = build_params.BuildType()
        definition_flags = build_params.DefinitionFlags()
        use_aapi_fuse = build_params.UseArcadiaApiFuse()
        use_arc_instead_of_aapi = build_params.UseArcInsteadOfArcadiaApi()
        runtime_data = WizardRuntimeBuildParams.RobotsRevision()
        wizard_data_disk_space = sdk2.parameters.Integer("Disk space for BUILD_WIZARD_DATA and wizard shard of BUILD_BEGEMOT_DATA tasks, in MB")
        need_stable_release_config = sdk2.parameters.Bool("Whether to make BEGEMOT_STABLE_RELEASE_CONFIG resource", default=False)
        with sdk2.parameters.RadioGroup("nanny Dashboard to resolve stable begemot services") as begemot_stable_services_dashboard:
            begemot_stable_services_dashboard.values["begemot"] = begemot_stable_services_dashboard.Value("begemot", default=True)
            begemot_stable_services_dashboard.values["begemot_advq"] = begemot_stable_services_dashboard.Value("begemot_advq")
            begemot_stable_services_dashboard.values["begemot_model_services"] = begemot_stable_services_dashboard.Value("begemot_model_services")

        with sdk2.parameters.Output():
            output_resources = sdk2.parameters.Dict("Output resources type with id")

    class Context(sdk2.Task.Context):
        build_tasks_id = []
        target_resources = {}
        targets = []


    def get_targets(self):
        if self.Parameters.all_resources:
            return [
                target for target in list(Begemots.keys()) + BW_RESOURCES.keys() if target not in RELEASED_SEPARATELY
            ]
        return self.Parameters.begemot_resources

    def run_fast_fresh_build_task(self, fresh_shards, common_config=False):
        input_params = {
            'ShardName': ' '.join(fresh_shards),
            'BuildFresh': True,
            'UseFastBuild': True,
            'AllInOneConfig': common_config,
            ARCADIA_URL_KEY: self.ctx['svn_url_with_rev']
        }
        if common_config:
            res_name = 'BEGEMOT_FAST_BUILD_FRESH_CONFIG'
            input_params.update({res_name: self.ctx[res_name]})
        else:
            for shard in fresh_shards:
                res_name = AllBegemotServices[shard].fresh_fast_build_config_resource_type.name
                input_params.update({res_name: self.ctx[res_name]})
        build_fresh_task = self.create_subtask(
            'BUILD_BEGEMOT_DATA',
            description=self.description,
            input_parameters=input_params,
            inherit_notifications=True,
        )
        return build_fresh_task.id

    def run_full_fresh_build_task(self, fresh_shards):
        runtime_task = self.create_subtask(
            task_type='WIZARD_RUNTIME_BUILD',
            description=self.description,
            input_parameters={
                WizardRuntimeBuildParams.RobotsRevision.name: utils.get_or_default(
                    self.ctx, WizardRuntimeBuildParams.RobotsRevision
                ),
                WizardRuntimeBuildParams.ParentResources.name: {
                    t: self.ctx[t]
                    for t in self.ctx['target_fresh_types']
                },
                WizardRuntimeBuildParams.BegemotShards.name: ' '.join(fresh_shards),
                WizardRuntimeBuildParams.RuntimeDataSource.name: WizardRuntimeBuildParams.RuntimeDataSource.Arcadia,
                WizardRuntimeBuildParams.RuntimeLegacy.name: WizardRuntimeBuildParams.RuntimeDataSource.Robots,
                WizardRuntimeBuildParams.DoMerge.name: False,
                WizardRuntimeBuildParams.DoCommit.name: True,
                WizardRuntimeBuildParams.DoBuild.name: True,
            },
            inherit_notifications=True,
        )
        return runtime_task.id

    def build_fresh(self, fast_build=False, common_fast_build=False):
        if common_fast_build:
            fast_build = True
        fresh_shards = [name for name, s in Begemots if s.release_fresh]
        # Begemot fresh shards in cypress
        for shard_name in fresh_shards:
            # Resource will be found in child task by parent_id and shard_name is_fresh attributes
            cypress_shard = resources.BEGEMOT_CYPRESS_SHARD(self, shard_name, str(uuid.uuid4()))
            cypress_shard.shard_name = shard_name
            cypress_shard.is_fresh = True
        if fast_build:
            task_id = self.run_fast_fresh_build_task(fresh_shards, common_fast_build)
        else:
            task_id = self.run_full_fresh_build_task(fresh_shards)

        self.Context.build_tasks_id.append(task_id)
        for t in self.target_fresh_types:
            utils.set_resource_attributes(
                self.Context.target_resources[t], {'runtime_data': utils.get_or_default(self.ctx, WizardRuntimeBuildParams.RobotsRevision)}
            )

    def collect_static_files(self, targets):
        collect_task = BuildBegemotCommon(
            self,
            checkout_arcadia_from_url=self.Context.svn_url_with_rev,
            description="Collect static files. Created by RELEASE_BEGEMOT_DATA",
            mode="collect_files",
            static_targets=targets,
            use_aapi_fuse=True,
            use_arc_instead_of_aapi=True
        ).enqueue()
        self.Context.build_tasks_id.append(collect_task.id)

    def build_binary(self, description, resources, **kwargs):
        logging.info(resources)
        resources = [r.name for r in resources]

        build_binary_task = BuildBegemotCommon(
            self,
            description=description + self.description,
            mode='build_executable',
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            arcadia_patch=self.Parameters.arcadia_patch,
            build_type=self.Parameters.build_type,
            definition_flags=self.Parameters.definition_flags,
            lto=False,
            separate_debug=True,
            use_aapi_fuse=True,
            use_arc_instead_of_aapi=True,
            targets=resources,
            parent_task_id=self.id,
            **kwargs
        ).enqueue()

        build_binary_task.Requirements.client_tags = ctc.Tag.Group.LINUX
        self.Context.build_tasks_id.append(build_binary_task.id)
        return build_binary_task

    def create_subtasks(self):
        target_shards = [
            target for target in self.Context.targets
            if target not in BW_RESOURCES and Begemots[target].release
        ]
        self.Context.svn_url_with_rev = self.Parameters.checkout_arcadia_from_url
        if self.Parameters.needed_revision:
            self.Context.svn_url_with_rev = self.Context.svn_url_with_rev + "@" + str(self.Parameters.needed_revision)
        utilities_resource = []
        static_files = []

        if "BegemotExecutable" in self.Context.targets:
            self.build_binary(
                resources=[
                    resources.BEGEMOT_EXECUTABLE,
                    resources.BEGEMOT_EVLOGDUMP,
                ],
                description='Main (search)',
            )

            self.build_binary(
                resources=[
                    resources.BEGEMOT_BERT_EXECUTABLE,
                ],
                description='Bert',
                musl=False,
            )

            utilities_resource.extend([
                resources.BEGEMOT_EVLOG_UPLOADER,
                resources.BEGEMOT_YT_EVENTLOG_MAPPER,
                resources.BEGEMOT_YT_MAPPER,
                resources.BEGEMOT_FAST_BUILD_DOWNLOADER,
                resources.BEGEMOT_ARGUMENTS_PARSER,
                resources.BEGEMOT_SHARD_UPDATER,
            ])

            self.build_binary(
                resources={
                    s.binary_resource for s in AllBegemotServices.values()
                    if s.release and s.released_by == 'search' and not s.beta and s.binary_resource not in {
                        resources.BEGEMOT_EXECUTABLE,  # already built above anyway
                        resources.BEGEMOT_BERT_EXECUTABLE,  # requires no-musl
                    }
                },
                description='Not beta-tested',
            )
        '''
        build_deployer_task = BuildBegemotCommon(
            self,
            description='Fast data deployer',
            mode='build_fast_data_deployer',
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            use_aapi_fuse=True,
            use_arc_instead_of_aapi=True,
            parent_task_id=self.id
        ).enqueue()
        self.Context.build_tasks_id.append(build_deployer_task.id)
        '''

        if "AdvqExecutable" in self.Context.targets:
            self.build_binary(
                resources=BW_RESOURCES['AdvqExecutable'],
                description='Advq',
            )

        if "MegamindExecutable" in self.Context.targets:
            self.build_binary(
                resources=BW_RESOURCES['MegamindExecutable'],
                description='Megamind',
            )

        if "BegginsExecutable" in self.Context.targets:
            self.build_binary(
                resources=BW_RESOURCES['BegginsExecutable'],
                description='Beggins',
            )

        if "RequestInitExecutable" in self.Context.targets:
            self.build_binary(
                resources=BW_RESOURCES['RequestInitExecutable'],
                description='RequestInit',
            )

        if "CaesarModelsExecutable" in self.Context.targets:
            self.build_binary(
                resources=BW_RESOURCES['CaesarModelsExecutable'],
                description='CaesarModels',
                musl=False,
            )

        description = self.Context.testenv_database if self.Context.testenv_database else ''
        if description:
            description = ' for ' + description

        for shard_name in target_shards:
            if shard_name == "Geo":  # We use full build for shard Geo
                continue

            # Resource will be found in child task by parent_id and shard_name is_fresh attributes
            build_shard_task = BuildBegemotCommon(
                self,
                description='{shard}. Created by RELEASE_BEGEMOT_DATA{desc}'.format(shard=shard_name, desc=description),
                binary_executor_release_type='stable',
                checkout_arcadia_from_url=self.Context.svn_url_with_rev,
                shards_to_build=[shard_name],
                use_fast_build=True,
                cypress_cache=BuildBegemotCommon.Parameters.cypress_cache.default_value if not 'Spellchecker' in shard_name else '',
                use_arc_instead_of_aapi=True,
            ).enqueue()
            build_shard_task.Requirements.disk_space = self.Parameters.wizard_data_disk_space if shard_name == 'Wizard' else None
            self.Context.build_tasks_id.append(build_shard_task.id)

        if "Geo" in target_shards:
            geoshard_resource_type = AllBegemotServices["Geo"].data_resource_type
            build_geoshard_task = BuildBegemotCommon(
                self,
                description='{shard}. Created by RELEASE_BEGEMOT_DATA{desc}'.format(shard='Geo', desc=description),
                checkout_arcadia_from_url=self.Context.svn_url_with_rev,
                shards_to_build=['Geo'],
                use_fast_build=False,
                use_full_shard_build=True,
                use_arc_instead_of_aapi=True,
            ).enqueue()
            self.Context.build_tasks_id.append(build_geoshard_task.id)


        self.Context.target_fresh_types = [
            t.name for t in
            (
                BW_RESOURCES['WizardRuntime'] * ('WizardRuntime' in self.Context.targets) +
                BW_RESOURCES['BegemotFresh'] * ('BegemotFresh' in self.Context.targets) +
                BW_RESOURCES['BegemotFreshFastBuild'] * ('BegemotFreshFastBuild' in self.Context.targets) +
                BW_RESOURCES['BegemotFreshFastBuildCommon'] * ('BegemotFreshFastBuildCommon' in self.Context.targets)
            )
        ]
        if self.Context.target_fresh_types:
            self.build_fresh('BegemotFreshFastBuild' in self.Context.targets, 'BegemotFreshFastBuildCommon' in self.Context.targets)

        logging.info('Target resources: {}'.format(self.Context.target_resources))

        if 'Bstr' in self.Context.targets:
            utilities_resource.extend(BW_RESOURCES['Bstr'])
        
        if 'EventsFiles' in self.Context.targets:
            utilities_resource.extend([
                resources.BEGEMOT_EVLOGDUMP_FOR_UNIFIED_AGENT,
            ])
            static_files.extend(
                [resource.__name__ for resource in BW_RESOURCES['EventsFiles']]
            )
                    
        if 'NannyConfigs' in self.Context.targets:
           static_files.extend(
                [resource.__name__ for resource in BW_RESOURCES['NannyConfigs']]
            )
        
        if utilities_resource:
            self.build_binary(
                resources=utilities_resource,
                description='Utilities',
            )
        if static_files:
            self.collect_static_files(
                targets=static_files,
            )
    
    def check_all_subtasks_done(self):
        for task_id in self.Context.build_tasks_id:
            task = sdk2.Task[task_id]
            logging.debug('Task #{id} status: {status}'.format(id=task_id, status=task.status))
            if task.status not in ctt.Status.Group.SUCCEED:
                raise TaskFailure("Child task #{id} failed".format(id=task_id))

    def copy_resources_from_child_tasks(self):
        target_resources = self.get_targets()
        if 'NannyConfigs' in target_resources:
            self.collect_static_files('NannyConfigs')

        if 'EventsFiles' in target_resources:
            self.collect_static_files('EventsFiles')
        res_from_child = set()

        for task_id, res_types in self.Context.build_tasks_id_with_res.iteritems():
            for res_type in res_types:
                res_from_child.add(res_type)
                logging.debug('Coping resource {} from {} task'.format(res_type, task_id))
                parent_res = sdk2.ResourceData(sdk2.Resource[res_type].find(task_id=self.id).first())
                child_res = sdk2.ResourceData(sdk2.Resource[res_type].find(task_id=task_id).first())

                if child_res.path.is_dir():
                    shutil.copytree(str(child_res.path), str(parent_res.path))
                else:
                    parent_res.path.write_bytes(child_res.path.read_bytes())
                logging.debug('Copied data from {resource_type} resource of {id} task'.format(resource_type=res_type, id=task_id))
        
        for res_type in self.Context.target_resources.keys():
            if res_type not in res_from_child:
                logging.debug("{} doesn't create in child task".format(res_type))

        for task_id, res_types in self.Context.build_tasks_id_with_res.iteritems():
            for res_type in res_types:
                res = sdk2.Resource[res_type].find(task_id=task_id)
                if res.state == ctr.State.NOT_READY:
                    sdk2.ResourceData(res).ready()
                    logging.debug('Mark resource #{} ready'.format(res.id))

    def merge_childs_output(self):
        output = {}
        for child_id in self.Context.build_tasks_id:
            task_output = sdk2.Task[child_id].Parameters.output_resources
            logging.info("Task #{id} output: {out}".format(id=child_id, out=task_output))
            output.update(task_output)
        output.update(self.Context.target_resources)
        self.Parameters.output_resources = output
                    
    def create_target_resource(self, target_name):
        if target_name not in RESOURCE_TO_CREATE:
            return
        if target_name not in BW_RESOURCES:
            # If a target is not in BW_RESOURCES - this is a begemot shard
            svc = AllBegemotServices[target_name]
            if not svc.release:
                return
            requested_resources = [
                (svc.fast_build_config_resource_type, target_name + "_fast_build") if target_name != "Geo" else
                (svc.data_resource_type, target_name),
            ]
        elif 'BegemotFresh' in target_name:
            requested_resources = []
            for res in BW_RESOURCES[target_name]:
                name = 'fast_build_config.json' if 'FastBuild' in target_name else 'fresh'
                path = os.path.join(res.dir_name, name)
                if res.name.endswith('_PACKED'):
                    path += '.tar'
                requested_resources.append((res, path))
        else:
            requested_resources = [(res, res.name) for res in BW_RESOURCES[target_name]]

        for res_type, res_path in requested_resources:
            if res_type and res_type.name not in self.Context.target_resources:
                res = sdk2.Resource[res_type.name](self, res_type.name, res_path)
                self.Context.target_resources[res_type.name] = res.id

    def on_enqueue(self):
        self.Context.targets = self.get_targets()
        self.Context.arcadia_revision = self.Parameters.needed_revision

        if not self.Context.testenv_database:
            self.Context.testenv_database = ''
        svn_branch = self.Context.testenv_database.split('-')[-1]
        self.svn_branch = None
        # allow running on trunk te base
        try:
            self.svn_branch = int(self.svn_branch)
        except TypeError or ValueError:
            pass

        for target in self.Context.targets:
            self.create_target_resource(target)
        #  res = FastDataDeployer(self, 'FastDataDeployer', 'FAST_DATA_DEPLOYER')
        #  self.Context.target_resources['FAST_DATA_DEPLOYER'] = res.id
        #  sdk2.Task.on_enqueue(self)
        logging.info('Target resources: {}'.format(self.Context.target_resources))


    def on_execute(self):
        token, session = None, None

        def get_token_and_start_client(token=None, session=None):
            if session is None:
                try:
                    token = sdk2.Vault.data('BEGEMOT', 'Begemot Nanny token')
                    session = NannyClient(api_url=NANNY_API_URL, oauth_token=token)
                    return token, session
                except Exception:
                    logging.debug("We failed during taking token from Vault and starting Nanny client")
                    return None, None
            return token, session

        def get_stable_services_list(token=None, session=None):
            dashboard_id = self.Parameters.begemot_stable_services_dashboard
            token, session = get_token_and_start_client(token, session)
            return session.get_dashboard_services(dashboard_id)
        
        def get_testing_services_list(token=None, session=None):
            re_service_category = '^/search/begemot/.*/testing/$'
            token, session = get_token_and_start_client(token, session)
            url = NANNY_API_URL + '/api/repo/ListSummaries/'
            all_services = session._get_url(url)['value']
            begemot_pattern = re.compile(re_service_category)
            return [s['serviceId'] for s in all_services if re.match(begemot_pattern, s['category'])]

        def get_files_from_service(service_id, token=None, session=None):
            token, session = get_token_and_start_client(token, session)
            response = session.get_service_runtime_attrs(service_id=service_id)
            return response['content']['resources']['sandbox_files']

        def get_url(session, url):
            try:
                r = session.get(url)
                r.raise_for_status()
            except requests.HTTPError as e:
                raise Exception("Failed to get FastBuildConfig data from url `{}`".format(url))
            except Exception as e:
                raise Exception('Failed to get url "{}"\nError: {}'.format(url, e))
            return r.json()

        def check_child_resource(res_type):
            res = sdk2.Resource[res_type].find(task_id=self.id).first()
            return res is not None

        def check_if_need_release_rule(res_id):
            res = sdk2.Resource.find(id=res_id, attrs={"released": "stable"}).first()
            return res is None

        def resource_description_dict(res_id, local_path, use_ssd, sandbox_release):
            result_dict = {
                'resource_id': res_id,
                'local_path': local_path,
                'sandbox_release': sandbox_release,
            }
            if use_ssd:
                result_dict['storage'] = '/ssd'
            return result_dict

        def get_resources_for_service(fast_build_config_name, use_ssd):
            session = requests.Session()
            retries = Retry(total=3, backoff_factor=1.0, status_forcelist=[429, 500, 502, 503, 504])
            session.mount(SANDBOX_URL, HTTPAdapter(max_retries=retries))
            res_id = self.Parameters.output_resources[fast_build_config_name]
            if res_id:
                url = SANDBOX_URL + str(res_id)
                resources = get_url(session, url)['resources']
                fb_config = [resource_description_dict(res_id, fast_build_config_name, use_ssd, sandbox_release=True)]
                files = [resource_description_dict(i["resource_id"], 'data/' + i["name"], use_ssd, sandbox_release=check_if_need_release_rule(i["resource_id"])) for i in sorted(resources)]
                return fb_config, files
            self.set_info("Found no FastBuildConfig resource in task, file name is `{}`".format(fast_build_config_name))
            return [], []

        # add up-to-date executable: BEGEMOT-2685
        def add_unified_agent_executable(file, use_ssd):
            resource_type = file['resource_type']
            last_resource_id, _ = resource_selectors.by_last_released_task(resource_type=sdk2.Resource[resource_type])
            return resource_description_dict(
                last_resource_id, file['local_path'], use_ssd, sandbox_release=check_child_resource(resource_type)
            )

        def create_release_json(services_list):
            release_json = defaultdict(list)
            for service in services_list:
                logging.debug("Starting build json description of resources of service {}".format(service))
                files = get_files_from_service(service, token=token, session=session)
                use_ssd = any([file.get('storage') == '/ssd' for file in files])
                if use_ssd:
                    logging.debug('Will use ssd for putting resources')
                else:
                    logging.debug('There are no resources on ssd now, so will not use it')
                for file in files:
                    if file['resource_type'].startswith('BEGEMOT_FAST_BUILD_CONFIG_'):
                        shard_name = file['resource_type']
                    elif file['resource_type'] == 'BEGEMOT_FAST_BUILD_RULE_DATA':
                        continue  # this is rules data
                    elif file['resource_type'] == 'BEGEMOT_FAST_DATA_CALLBACK':
                        continue  # we add it after cycle, do not add it twice
                    elif file['resource_type'] == 'BEGEMOT_BSTR_CALLBACK':
                        continue  # do not deploy old callback
                    elif file['resource_type'] == 'UNIFIED_AGENT_BIN' and file['local_path'] == 'unified_agent_setrace':
                        release_json[service].append(add_unified_agent_executable(file, use_ssd))
                    else:
                        file_name = file['resource_type']
                        file_id = self.Parameters.output_resources.get(file_name, None)
                        if not file_id:
                            logging.debug('Cannot find {} resource in target resources'.format(file_name))
                        do_release = check_child_resource(file['resource_type'])
                        if not file_id:
                            file_id = file['resource_id']
                            if file_name not in ACCEPT_MISSING_IN_RELEASE:
                                self.set_info("Found no `{}` resource in task for service `{}`".format(file_name, service))
                        release_json[service].append(resource_description_dict(file_id, file['local_path'], use_ssd, sandbox_release=do_release))
                if 'BEGEMOT_FAST_DATA_CALLBACK' in self.Parameters.output_resources:
                    release_json[service].append(resource_description_dict(self.Parameters.output_resources['BEGEMOT_FAST_DATA_CALLBACK'], 'fast_data_callback', use_ssd, sandbox_release=True))

                if shard_name not in self.Parameters.output_resources:
                    continue
                fast_build_config_file, new_files = get_resources_for_service(shard_name, use_ssd)
                logging.debug("fast_build_config_file {}".format(type(fast_build_config_file)))
                logging.debug("release_json[service] {}".format(type(release_json[service])))
                logging.debug("new_files {}".format(type(new_files)))
                release_json[service] = fast_build_config_file + release_json[service] + new_files
            return json.dumps(release_json, encoding='utf-8', indent=2)

        token, session = get_token_and_start_client()
        if session is None:
            self.set_info("Sadly, we failed starting Nanny client.")
            if token is None:
                logging.debug("Also we failed taking token from Vault.")
            nanny_imported = False
        else:
            self.set_info("Started Nanny client successfully!")
            nanny_imported = True

        self.Context.begemot_check_task_id = []
        self.description = ' for ' + str(self.type)
        db = self.Context.testenv_database if self.Context.testenv_database else ''
        if db:
            self.description += ', ' + db

        if not self.Context.build_tasks_id:
            with self.memoize_stage.create_children:
                self.create_subtasks()
                logging.debug('Child tasks with res: {}'.format(self.Context.build_tasks_id))
                raise sdk2.WaitTask(self.Context.build_tasks_id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)
        else:
            self.check_all_subtasks_done()
            self.merge_childs_output()

        if self.Parameters.need_stable_release_config or (
            nanny_imported and self.Parameters.all_resources
        ):
            stable_release_json = create_release_json(get_stable_services_list(token=token, session=session))
            testing_release_json = create_release_json(get_testing_services_list(token=token, session=session))

            with open("release_stable.json", "w") as f:
                f.write(stable_release_json)
            stable_release_res = resources.BEGEMOT_STABLE_RELEASE_CONFIG(self, self.description, "release_stable.json")
            stable_release_res.ttl = 60            
            self.Context.out_resource_id = stable_release_res.id

            with open("release_testing.json", "w") as f:
                f.write(testing_release_json)
            testing_release_res = resources.BEGEMOT_TESTING_RELEASE_CONFIG(self, self.description, "release_testing.json")
            testing_release_res.ttl = 30
