# coding: utf-8

import logging

import sandbox.common.types.resource as ctr
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.websearch.begemot.parameters as bp
import sandbox.projects.websearch.begemot.resources as br

from sandbox import sdk2
from sandbox.projects.fuzzing import resources
from sandbox.projects.fuzzing import fuzzers
from sandbox.projects.websearch.begemot.common.fast_build import ShardSyncHelper


class RunBegemotFuzzy(sdk2.Task):
    """
        BEGEMOT-1011
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 40 * 1024
        client_tags = task_env.TaskTags.startrek_client
        environments = [task_env.TaskRequirements.startrek_client]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60 * 60  # 10 hours
        count_coverage = sdk2.parameters.Bool("Count coverage")
        begemot_executable_resource = sdk2.parameters.Resource(
            "Begemot fuzzy binary",
            resource_type=br.BEGEMOT_EXECUTABLE_FUZZY,
            required=True,
        )
        fast_build_config = bp.FastBuildConfigResource(required=True)
        runs = sdk2.parameters.Integer("Runs number (0 = no fuzzing)", default=-1)
        max_total_time = sdk2.parameters.Integer("Fuzzy time (sec)", default=9000)
        last_fuzzed_queries = sdk2.parameters.Bool("Use last fuzzed queries")

        with last_fuzzed_queries.value[False]:
            begemot_queries_resource = sdk2.parameters.Resource(
                "Queries for Begemot",
                resource_type=resources.BegemotQueriesFuzzy,
                required=True,
            )

        add_queries = sdk2.parameters.Resource(
            "Additional queries for begemot",
            resource_type=resources.BegemotQueriesFuzzy,
            required=False,
        )

    def on_enqueue(self):
        shard_size = ShardSyncHelper(self.Parameters.fast_build_config).get_shard_size()
        data_and_binary_size_mb = (
            self.Parameters.begemot_executable_resource.size + shard_size
        ) >> 20
        self.Context.rss_limit_mb = self.Requirements.disk_space = data_and_binary_size_mb + 5 * 1024

    def _prepare_shard(self):
        shard_helper = ShardSyncHelper(self.Parameters.fast_build_config)
        data_path = str(self.path('data'))
        shard = shard_helper.sync_shard(data_path)

    def on_execute(self):
        if self.Parameters.last_fuzzed_queries:
            queries = list(sdk2.Resource.find(
                resource_type=resources.BegemotQueriesFuzzy,
                state=ctr.State.READY,
                attr_name="last_fuzzed_queries"
            ).limit(1))[0]
            logging.info("Last fuzzed queries resource id: %s", queries.id)
        else:
            queries = self.Parameters.begemot_queries_resource

        self._prepare_shard()

        begemot_fuzzer = fuzzers.BegemotFuzzer(
            self.Parameters.begemot_executable_resource,
            queries,
            add_queries=(self.Parameters.add_queries,),
            cmd_args={
                "rss_limit_mb": self.Context.rss_limit_mb,
                "max_total_time": self.Parameters.max_total_time,
                "runs": self.Parameters.runs,
            },
            count_coverage=self.Parameters.count_coverage,
            task=self,
        )
        with begemot_fuzzer:
            begemot_fuzzer.use_component()
        if self.Parameters.count_coverage:
            missing_info = begemot_fuzzer.output_results[fuzzers.CoverageCounting.NAME]["missing_info"]
            self.Context.coverage_info = missing_info
            self.Context.coverage_per_cent = 100.0 * missing_info["pcs_covered"] / (missing_info["pcs_instrumented"] or 1)
            self.set_info("Coverage: {:.3f} %".format(self.Context.coverage_per_cent))
            if missing_info["pcs_not_found_exist"]:
                self.set_info("WARNING: probably coverage info is not correct for this binary!")

