from sandbox import sdk2, common
from sandbox.sandboxsdk import environments
import sandbox.projects.websearch.begemot.resources as br
from requests import Session
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from . import best_split


class ShardAutobalancing(sdk2.Task):
    Result = {}

    class Parameters(sdk2.Task.Parameters):
        begemot_binary = sdk2.parameters.Resource(
            'Begemot binary to describe',
            resource_type=br.BEGEMOT_EXECUTABLE,
            required=True,
        )

    class Requirements(sdk2.Task.Requirements):
        environments = (environments.PipEnvironment('yasmapi'),)

    def on_execute(self):
        begemot_path = str(sdk2.ResourceData(self.Parameters.begemot_binary).path)
        self.Result = best_split.run(begemot_path)
        if len(self.Result['current_shards']) <= len(self.Result['new_shards']):
            self.set_info('Better split was not found')
            return
        api = common.rest.Client()
        oauth_token = api.vault.read(name='Begemot startrek token', owner='BEGEMOT', limit=1)["items"][0]["id"]
        session = Session()
        session.headers['Content-Type'] = 'application/json'
        session.headers['Authorization'] = 'OAuth {}'.format(oauth_token)
        session.mount('https://', HTTPAdapter(max_retries=Retry(total=10, backoff_factor=0.1, status_forcelist=[202])))
        response = session.post(
            'https://st-api.yandex-team.ru/v2/issues',
            data={
                'queue': 'BEGEMOT',
                'summary': 'Please rebalance shards. See task result with id {task_id}'.format(task_id=sdk2.Task.id),
                'tags': ['shard-autobalancing'],
            },
        )
        if not response.ok:
            raise BaseException('Ticket was not created')
        self.set_info('task created')

    def footer(self):
        if not self.Result:
            return
        if len(self.Result['current_shards']) <= len(self.Result['new_shards']):
            return
        response = []
        for shard_id, (shard_time, shard_size, shard_rulenames) in self.Result.items():
            response.append({
                '<h4>Shard #{i}; critical path time: {t} mcs; size: {sz} mb</h4>'.format(i=shard_id, t=shard_time, sz=shard_size): {
                    'header': [{'key': 'Rule', 'title': 'Rule'}],
                    'body': {'Rule': [name for name in shard_rulenames]},
                }
            })
        return response
