from itertools import combinations
from subprocess import Popen, PIPE
from json import loads
from os.path import exists, join
from os import rmdir, mkdir, getcwd
import sys

import constants as c
from graph import Graph
from shard import merge_two_shards, to_yamake


def get_best_split(graph):
    single_leaf_shards = [graph.get_shard_with_single_leaf(leaf_id) \
        for leaf_id in graph.get_leafs(choose_from_significant_rules=True)]
    new_shards = unite_shards(single_leaf_shards)
    return new_shards


def unite_shards(shard_list):
    while True:
        first_idx, second_idx = find_nearest_shards(shard_list)
        if first_idx is None:
            break
        shard_list[first_idx].merge_with(shard_list[second_idx])
        shard_list.pop(second_idx)
    return shard_list


def find_nearest_shards(shard_list):
    '''
    finds two shards that give best memory difference when merged
    memory difference = first_shard_memory + second_shard_memory - merged_memory
    '''
    first_idx = None
    second_idx = None
    best_delta = None

    for i, j in combinations(range(len(shard_list)), 2):
        shard = merge_two_shards(shard_list[i], shard_list[j])
        if shard.get_size() > c.MAX_SHARD_SIZE:
            continue

        delta = shard_list[i].get_size() + shard_list[j].get_size() - shard.get_size()
        if best_delta is None or delta > best_delta:
            best_delta = delta
            first_idx = i
            second_idx = j

    return first_idx, second_idx


def run(begemot_path):
    p = Popen((begemot_path, '--print-bgschema'), stdout=PIPE, stderr=PIPE)
    out, _ = p.communicate()
    bgschema = loads(out) if isinstance(out, str) else loads(out.decode())
    graph = Graph(bgschema)
    current_shards = [graph.get_existing_shard(name) for name in c.SIGNIFICANT_SHARDS]
    new_shards = get_best_split(graph)

    result = {
        'current_shards': {},
        'new_shards': {},
    }
    for i, name in enumerate(c.SIGNIFICANT_SHARDS):
        t = current_shards[i].get_longest_path_time()
        sz = current_shards[i].get_size()
        result['current_shards'][name] = (t, sz, current_shards[i].get_names())
    for i, shard in enumerate(new_shards):
        t = shard.get_longest_path_time()
        sz = shard.get_size()
        result['new_shards'][i] = (t, sz, shard.get_names())

    return result

if __name__ == '__main__':  # if started from console
    cwd = getcwd() + '/'
    idx = cwd.find('/arcadia/')
    if idx == -1:
        raise BaseException('Please run from arcadia')
    path_to_arcadia = cwd[:idx] + '/arcadia/'
    begemot_path = join(path_to_arcadia, 'search/daemons/begemot/default/begemot')
    begemot_all_services_path = join(path_to_arcadia, 'sandbox/projects/websearch/begemot')
    sys.path.append(begemot_all_services_path)  # for using BegemotAllServices

    result = run(begemot_path)

    print('existing shards:')
    for name, (t, sz, _) in result['current_shards'].items():
        print('{name} : {t} mcs; {sz} mb'.format(name=name, t=t, sz=sz))
    path = join(getcwd(), 'new_shards')
    if exists(path):
        rmdir(path)
    mkdir(path)
    print('\nnew shards:')
    for i, (t, sz, _) in result['new_shards'].items():
        print('#{i} : {t} mcs; {sz} mb'.format(i=i, t=t, sz=sz))
    for i, (_, _, ruleset) in result['new_shards'].items():
        with open(join(path, '{i}.txt'.format(i=i)), 'w') as f:
            f.write(to_yamake(ruleset))
    print('new shards\' yamake files are stored in {path}'.format(path=path))
