from functools import reduce
from re import findall
from requests import Session
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from rule_parameters import get_quantiles, get_sizes
from shard import Shard
import constants as c


def get_rulenames_from_significant_shards():
    session = Session()
    session.headers['Content-Type'] = 'application/json'
    session.mount('https://', HTTPAdapter(max_retries=Retry(total=10, backoff_factor=0.1, status_forcelist=[202])))

    try:
        from sandbox.projects.websearch.begemot.common import BegemotAllServices
    except:
        from common import BegemotAllServices
    shard_info = BegemotAllServices()
    rules = {}
    for shardname in c.SIGNIFICANT_SHARDS:
        prj = shard_info.Service[shardname].prj
        request_url = '{url}/?itype=begemot&prj={prj}&signal_pattern=begemot-WORKER-.*-TIME_dhhh'.format(url=c.YASM_URL, prj=prj)
        signals = session.get(request_url).json()['response']['result']
        current_rules = [findall(r'\w+-\w+-([A-Z][a-zA-Z0-9]*[a-z][a-zA-Z0-9]*)-TIME_dhhh', s) for s in signals]
            # e. g., "begemot-WORKER-ThesaurusDataParams-TIME_dhhh" -> "ThesaurusDataParams"
        rules[shardname] = set([s[0] for s in current_rules if s])
    return rules


def parse_bgschema(bgschema):
    id2name = {}
    dependencies = set()
    for name in bgschema['Rules']:
        info = bgschema['RuleInfos'][name]
        rid = int(info['Id']) if 'Id' in info else 0
        id2name.update({rid: name})
        if 'Dependents' in info:
            for child, dependency_type in info['Dependents'].items():
                if 'Required' in dependency_type:
                    child_id = int(bgschema['RuleInfos'][child]['Id'])
                    dependencies.add((rid, child_id))
    return id2name, dependencies


class Graph:

    def __init__(self, bgschema):
        self.id2name = {}
        self.name2id = {}
        self.shard2id = {}
        self.times = {}
        self.sizes = {}
        self.parents = {}
        self.children = {}
        self.significant_rules = set()  # all rules from significant shards
        self.shard2id = {}  # existing shards

        self.id2name, dependencies = parse_bgschema(bgschema)
        self.name2id = {name: rid for rid, name in self.id2name.items()}
        for parent_id, child_id in dependencies:
            self._add_edge(parent_id, child_id)
        shard2name = get_rulenames_from_significant_shards()
        for shard_name, rule_names in shard2name.items():
            rule_ids = set([self.name2id[name] for name in rule_names])
            self.shard2id[shard_name] = rule_ids
        self.significant_rules = reduce(lambda a, b: a | b, self.shard2id.values())

        quantiles = get_quantiles(set(self.id2name.values()))
        self.times = {self.name2id[name]: value for name, value in quantiles.items()}

        rule_sizes = get_sizes()
        self.sizes = {rid: rule_sizes[name] if name in rule_sizes else 0 for rid, name in self.id2name.items()}

    def get_parents(self, rid):
        return self.parents[rid] if rid in self.parents else set()

    def get_children(self, rid):
        return self.children[rid] if rid in self.children else set()

    def _add_edge(self, parent_id, child_id):
        if parent_id == child_id:
            return self
        new_children = self.get_children(parent_id) | set([child_id])
        self.children.update({parent_id: new_children})
        new_parents = self.get_parents(child_id) | set([parent_id])
        self.parents.update({child_id: new_parents})
        return self

    def get_leafs(self, choose_from_significant_rules=False):
        if choose_from_significant_rules:
            return set([rid for rid in self.significant_rules if not (self.get_children(rid) & self.significant_rules)])
        else:
            return set(self.id2name.keys()) - set(self.children.keys())

    def get_roots(self):
        return set(self.id2name.keys()) - set(self.parents.keys())

    def get_shard_with_single_leaf(self, leaf_id):
        try:
            from Queue import Queue
        except:
            from queue import Queue
        rule_ids = set()
        queue = Queue()
        queue.put_nowait(leaf_id)
        used = set([leaf_id])

        while not queue.empty():
            rid = queue.get_nowait()
            rule_ids.add(rid)
            for parent_id in self.get_parents(rid) - used:
                used.add(parent_id)
                queue.put_nowait(parent_id)
        return Shard(rule_ids, self)

    def get_shard(self, rule_ids):
        leaf_ids = [rid for rid in rule_ids if not self.get_children(rid) & rule_ids]
        shard = self.get_shard_with_single_leaf(leaf_ids.pop(0))
        for lid in leaf_ids:
            shard.merge_with(self.get_shard_with_single_leaf(lid))
        return shard

    def get_existing_shard(self, shard_name):
        return self.get_shard(self.shard2id[shard_name])
