import json
import os
import time

import sandbox.common.types.notification as ctn
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.svn import Arcadia


ARCADIA_URL = Arcadia.ARCADIA_TRUNK_URL


class CheckDeprecatedArguments(sdk2.Task):
    _st_client_instance = None

    class Parameters(sdk2.Parameters):
        args_file = sdk2.parameters.String(
            'File with deprecated arguments',
            description='Arcadia path to json file with deprecated arguments',
            required=True,
        )
        st_queue = sdk2.parameters.String(
            'Startrek queue',
            description='Where to create reminder tickets',
            required=True,
        )

    class Requirements(sdk2.Requirements):
        environments = [PipEnvironment('startrek_client')]

    @property
    def st_client(self):
        if self._st_client_instance is None:
            from startrek_client import Startrek
            self._st_client_instance = Startrek(token=sdk2.Vault.data('robot-search-devops_st_token'),
                                                useragent='robot-search-devops')
        return self._st_client_instance

    def get_tags(self, script_name, arg_info):
        return [
            script_name,
            arg_info['arg_name'],
            arg_info['removal_date'],
        ]

    def create_issue(self, script_name, arg_info):
        return self.st_client.issues.create(
            queue = self.Parameters.st_queue,
            assignee = arg_info['staff_login'],
            tags = self.get_tags(script_name, arg_info),
            summary = 'Removal of %s argument of script %s' % (
                arg_info['arg_name'],
                script_name
            ),
            description = (
                'Please, remove argument %s of script %s on date %s.\n'
                'Ticket created by task https://sandbox.yandex-team.ru/task/%d/view\n'
                'If you do not expected to get this ticket, you can ask mvel@' % (
                    arg_info['arg_name'],
                    script_name,
                    arg_info['removal_date'],
                    self.id,
                )
            ),
        )

    def find_or_create_issue(self, script_name, arg_info):
        query = "(Queue: %s) " % (self.Parameters.st_queue)
        query += " ".join(["AND Tags: %s" % (st_tag) for st_tag in self.get_tags(script_name, arg_info)])
        issues = self.st_client.issues.find(query)
        if len(issues) > 1:
            self.set_info(
                'Found %s issues in queue %s with tags %s' % tuple(
                    [str(len(issues)), self.Parameters.st_queue] +
                    self.get_tags(script_name, arg_info)
                )
            )
        if len(issues):
            return issues[-1]
        return self.create_issue(script_name, arg_info)

    def create_reminder_comment(self, script_name, arg_name, issue, summonees):
        issue.comments.create(
            text=(
                'Removal date has come. Please remove argument %s of script '
                '%s completely.' % (arg_name, script_name)
            ),
            summonees=summonees,
        )

    def get_elapsed_time(self, script_name, arg_info):
        try:
            expiration_time = time.mktime(time.strptime(arg_info['removal_date'], '%d.%m.%Y'))
        except:
            self.server.notification(
                subject='CheckDeprecatedArguments: invalid removal date format',
                body=(
                    'Task: https://sandbox.yandex-team.ru/task/%d/view\n'
                    'You recieved this letter as responsible for removal of '
                    'argument %s of script %s.\nThe removal date for this '
                    'argument is not given or given in incorrect format. Please, '
                    'specify this date in format `%%d.%%m.%%Y`.' % (
                        self.id,
                        arg_info['arg_name'],
                        script_name,
                    )
                ),
                recipients=[arg_info['staff_login']],
                transport=ctn.Transport.EMAIL,
            )
            return 0
        return time.time() - expiration_time

    def on_execute(self):
        parsed_url = Arcadia.parse_url(ARCADIA_URL)
        modified_path = os.path.join(parsed_url.path, self.Parameters.args_file)
        url = Arcadia.replace(ARCADIA_URL, modified_path)
        deprecated_args = json.loads(Arcadia.cat(url))
        for script_name, args_info in deprecated_args.items():
            for arg_info in args_info:
                issue = self.find_or_create_issue(script_name, arg_info)
                elapsed_time = self.get_elapsed_time(script_name, arg_info)
                if elapsed_time > 0:
                    self.create_reminder_comment(
                        script_name,
                        arg_info['arg_name'],
                        issue,
                        "mvel" if elapsed_time > 14 * 24 * 60 * 60 else None
                    )
