# coding=utf-8

import collections
import copy
import json
import logging
import os
import subprocess
import time
from datetime import datetime

import requests
import sandbox.common.types.notification as ctn
import urllib2
from sandbox import sdk2
from sandbox.projects.AddrsDeploy.tools.misc import retry
from sandbox.projects.websearch.begemot import resources as br
from sandbox.projects.websearch.begemot.common import Begemots
from sandbox.sandboxsdk import environments

CPU_USAGE_PROPORTION = "quant(portoinst-cpu_guarantee_usage_perc_hgram,90)"
NET_USAGE_PROPORTION = "perc(max(quant(portoinst-net_rx_speed_hgram, 90),quant(portoinst-net_tx_speed_hgram, 90))," \
                       "div(div(portoinst-net_guarantee_mb_summ, normal()), counter-instance_tmmv))"
YASM = "yasm.yandex-team.ru"
# Bert: 1 GPU gives us exactly 6.091 CPU, it cannot be changed
# Logger uses its own abc service
SKIP_PRJ = {'bert', 'logger'}
NETWORK_USAGE_DEFAULT_TARGET = [20, 80]


Value = collections.namedtuple("Value", "cpu net")


class RecalcCpuGuarantees(sdk2.Task):

    class Requirements(sdk2.Requirements):
        environments = (
            environments.PipEnvironment('yasmapi'),
            environments.PipEnvironment('startrek_client', custom_parameters=['requests==2.18.4']),
        )


    class Parameters(sdk2.Parameters):
        itype = sdk2.parameters.String(
            'itype',
            description='itype tag value in Golovan',
            default_value='begemot',
            required=True
        )
        prj = sdk2.parameters.String(
            'List of prj',
            description='List of prj tag values in Golovan (e.g. "merger,failcache,ethos"). Not required if itype == "begemot"',
            required=False
        )
        rps_signal = sdk2.parameters.String(
            'RPS signal',
            description='Name of Golovan signal with service RPS',
            default_value='apphost-SELF-SS-Requests_dmmm',
            required=True
        )
        use_rps_target = sdk2.parameters.Bool(
            'Recalc CPU utilization using RPS target',
            default_value=False,
            description='Multiply CPU usage values by [Target RPS]/[Real RPS]'
        )
        with use_rps_target.value[True]:
            target_rps = sdk2.parameters.Integer(
                'Target RPS',
                description='Target RPS while CPU usage is equal to target',
                required=True
            )
        auto_target_cpu = sdk2.parameters.Bool(
            'Get target CPU usage automatically (works with itype == begemot)',
            default_value=False
        )
        with auto_target_cpu.value[False]:
            target_cpu = sdk2.parameters.Integer(
                'Target CPU usage %',
                default_value=70,
                description='Critical CPU usage while RPS is equal to target',
                required=True
            )
            target_cpu_no_target_rps = sdk2.parameters.Integer(
                'Target usual CPU usage %',
                description='Mean target CPU usage for services without defined target RPS',
                default_value=30,
                required=True
            )
        quantile = sdk2.parameters.Float(
            'CPU usage quantile',
            default_value=0.9,
            required=True
        )
        secret_owner = sdk2.parameters.String(
            'Owner group',
            default_value='BEGEMOT',
            required=True
        )
        create_tickets = sdk2.parameters.Bool(
            'Create startrek tickets',
            default_value=True,
        )
        with create_tickets.value[True]:
            secret = sdk2.parameters.String(
                'Startrek oauth secret',
                default_value='Begemot startrek token',
                description='Token from https://sandbox.yandex-team.ru/admin/vault?owner={your owner group}',
                required=True
            )
            startrek_queue = sdk2.parameters.String(
                'Startrek queue for ticket creation',
                default_value='BEGEMOT',
                required=True
            )
            assignee = sdk2.parameters.String(
                'Assign tickets to',
                required=True
            )
        do_yp_reallocation = sdk2.parameters.Bool(
            'Do auto reallocation',
            default_value=False,
        )
        with do_yp_reallocation.value[True]:
            nanny_token = sdk2.parameters.String(
                'Nanny oauth token',
                default_value='Begemot Nanny token',
                description='Token from https://sandbox.yandex-team.ru/admin/vault?owner={your owner group}',
                required=True
            )
            yp_token = sdk2.parameters.String(
                'YP oauth token',
                default_value='Begemot YP token',
                description='Token from https://sandbox.yandex-team.ru/admin/vault?owner={your owner group}',
                required=True
            )
            abc = sdk2.parameters.Integer(
                'ABC service num',
                required=True
            )
            tidiness = sdk2.parameters.Resource(
                "Nanny tidiness",
                required=False,
                resource_type=br.BEGEMOT_NANNY_TIDINESS
            )
            recipients = sdk2.parameters.List(
                "Logins for email notification",
                required=False
            )
            auto_inc_threshold = sdk2.parameters.Float(
                'Increase CPU no more than by, %',
                default_value=7.0
            )
            auto_dec_threshold = sdk2.parameters.Float(
                'Decrease CPU no more than by, %',
                default_value=1.0
            )
            ignore_threshold = sdk2.parameters.Float(
                'No action for difference within, %',
                default_value=0.3
            )
            reserved_cpu = sdk2.parameters.Float(
                'CPU quota unavailable for auto reallocation, %',
                default_value=1
            )
            dry_run = sdk2.parameters.Bool(
                'Do not actually reallocate (debug mode)',
                default_value=False
            )

    def _get_golovan_request(self, geo, prj, signal, link=False, ts_from=None, ts_to=None, screenshot=False):
        tags = "itype=%s;ctype=prod,prestable;geo=%s;prj=%s" % (self.Parameters.itype, geo, prj)
        if not link:
            return tags + ":%s" % (signal)

        url = ("%s/chart/hosts=ASEARCH;" % YASM) + tags + ";signals=%s/?from=%s&to=%s&static=true" % (signal, int(ts_from * 1000), int(ts_to * 1000))
        if screenshot:
            return "https://s." + url
        return "https://" + url


    def _count_stat(self, responses, usage_sig, use_target, rps_sig=None, period=None):
        usage_values = []
        for ts, values in responses:
            if values[usage_sig] is not None:
                if not use_target:
                    usage_values.append(values[usage_sig])
                else:
                    real_usage = values[usage_sig]
                    real_rps = values[rps_sig] / period
                    if real_usage is not None and real_rps is not None and real_rps > self.Parameters.target_rps / 3.5:
                        usage_values.append(real_usage * self.Parameters.target_rps / real_rps)
                    elif real_usage is not None:
                        usage_values.append(real_usage)
                    else:
                        usage_values.append(0)

        logging.debug(usage_values)
        usage_values.sort()
        if len(usage_values) == 0:
            return 0
        return usage_values[min(len(usage_values) - 1, int(len(usage_values) * self.Parameters.quantile))]


    def _collect_golovan_stat(self, geo, shards, no_target_shards, start_time, end_time, period):
        try:
            from infra.yasm.yasmapi import GolovanRequest
        except:
            from yasmapi import GolovanRequest

        target_stat = {}
        no_target_stat = {}

        for g in geo:
            for shard in shards:
                cpu_usage_sig = self._get_golovan_request(g, shard, CPU_USAGE_PROPORTION)
                net_usage_sig = self._get_golovan_request(g, shard, NET_USAGE_PROPORTION)

                if self.Parameters.itype == 'begemot':
                    rps_shard = 'merger'
                else:
                    rps_shard = shard
                rps_sig = self._get_golovan_request(g, rps_shard, self.Parameters.rps_signal)

                gr = GolovanRequest('ASEARCH', period, start_time, end_time, [cpu_usage_sig, net_usage_sig, rps_sig],
                                    load_delay=0.1, max_retry=15, retry_delay=5)

                if shard in no_target_shards:
                    no_target_stat[(g, shard)] = Value(self._count_stat(gr, cpu_usage_sig, False),
                                                       self._count_stat(gr, net_usage_sig, False))
                else:
                    target_stat[(g, shard)] = Value(self._count_stat(gr, cpu_usage_sig, True, rps_sig, period),
                                                    self._count_stat(gr, net_usage_sig, True, rps_sig, period))

        return target_stat, no_target_stat

    def _analyze_stat(self, target_stat, no_target_stat, names, cpu_targets):
        to_ticket = {}

        info = []
        for label, stats in [
            ("Normalized usage values (%s-quantiles):", target_stat),
            ("%s-quantiles of usage:", no_target_stat),
        ]:
            if not stats:
                continue
            info.append(label % self.Parameters.quantile)
            for key, value in sorted(stats.items(), key=lambda item: item[0][1]):
                geo, service = key
                target = cpu_targets[service]
                cpu_critical = target[1]
                net_critical = NETWORK_USAGE_DEFAULT_TARGET[1]
                cpu_value = value.cpu
                net_value = value.net
                info.append('{service}, {geo}: cpu usage is {cpu_value:.2f}% (critical is {cpu_critical}%), '
                            'net usage is {net_value:.2f}% (critical is {net_critical}%)'.format(**locals()))
                if (cpu_value > cpu_critical or cpu_value < target[0] or net_value > net_critical
                    or net_value < NETWORK_USAGE_DEFAULT_TARGET[0]):
                    if service in to_ticket:
                        to_ticket[service].append((value, target, geo))
                    else:
                        to_ticket[service] = [(value, target, geo)]
            info.append("")

        self.set_info("\n" .join(info))
        return to_ticket


    @retry(tries=5, delay=2)
    def _get_chart_screenshot(self, geo, prj, signal, start_time, end_time):
        url = self._get_golovan_request(geo, prj, signal, link=True, ts_from=start_time, ts_to=end_time, screenshot=True)
        chart_url = self._get_golovan_request(geo, prj, signal, link=True, ts_from=start_time, ts_to=end_time, screenshot=False)
        response = urllib2.urlopen(url.replace(' ', ''), timeout=20)
        return chart_url, response.geturl()


    @retry(tries=5, delay=2)
    def _get_nanny_services(self, geo, prj):
        url = "https://{}/metainfo/tags/nanny?itype={}&ctype=prod,prestable&geo={}&prj={}&hosts=ASEARCH".format(YASM, self.Parameters.itype, geo, prj)
        response = requests.get(url)
        return json.loads(response.text)["response"]["result"]

    def _check_pods_groups_diff(self, pods_groups_original, diff_allowed_fields):
        pods_groups = copy.deepcopy(pods_groups_original)
        for pods_group in pods_groups:
            for field in diff_allowed_fields:
                pods_group["allocationRequest"][field] = pods_groups[0]["allocationRequest"][field]
                pods_group["grouper"][field] = pods_groups[0]["grouper"][field]
        return all([
            (pods_group["allocationRequest"] == pods_groups[0]["allocationRequest"]) and
            (pods_group["grouper"] == pods_groups[0]["grouper"])
            for pods_group in pods_groups
        ])

    def _get_reallocation_info(self, service, geo, coefs, nanny_token):
        if service is None or not service:
            return None, None, None
        YP_API_URL = "https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-sets/ListPodsGroups/"
        session = requests.Session()
        session.headers['Authorization'] = 'OAuth {}'.format(nanny_token)
        session.headers['Content-Type'] = 'application/json'

        data = {
            'serviceId': service,
            'cluster': geo.upper(),
        }
        response = session.post(YP_API_URL, data=json.dumps(data))
        link = "https://nanny.yandex-team.ru/ui/#/services/catalog/{}".format(service)
        try:
            pods_groups = response.json()["podsGroups"]
            cpu_guarantee = max([pods_group["allocationRequest"]["vcpuGuarantee"] for pods_group in pods_groups])
            net_guarantee = max([pods_group["allocationRequest"]["networkBandwidthGuaranteeMegabytesPerSec"] for pods_group in pods_groups])

            diff_allowed_fields = []
            if coefs.cpu:
                diff_allowed_fields += ["vcpuGuarantee", "vcpuLimit"]
                logging.debug("{}, cpu lack coef is {}".format(service, coefs.cpu))
            if coefs.net:
                diff_allowed_fields += ["networkBandwidthGuaranteeMegabytesPerSec"]
                logging.debug("{}, net lack coef is {}".format(service, coefs.net))
            if not self._check_pods_groups_diff(pods_groups, diff_allowed_fields):
                logging.debug("Failed to do reallocation for service {} because it has different podsets".format(service))
                return None, None, Value(None, None)

            new_cpu_guarantee = self._adjust_guarantee(cpu_guarantee, int(int(cpu_guarantee) * coefs.cpu) if coefs.cpu else None)
            new_net_guarantee = self._adjust_guarantee(net_guarantee, int(int(net_guarantee) * coefs.net) if coefs.net else None)

            if new_cpu_guarantee:
                added_cpu = sum(
                    [(new_cpu_guarantee - pods_group["allocationRequest"]["vcpuGuarantee"]) * len(pods_group["summaries"])
                    for pods_group in pods_groups]
                )
                if not self._check_reserve(geo, added_cpu, "cpu"):
                    self.set_info("{}: not enough CPU reserve for cluster {}, suggested change: {} -> {}".format(service,
                                                                                                                 geo,
                                                                                                                 cpu_guarantee,
                                                                                                                 new_cpu_guarantee))
                    new_cpu_guarantee = None

            if new_net_guarantee:
                added_net = sum(
                    [(new_net_guarantee - pods_group["allocationRequest"]["networkBandwidthGuaranteeMegabytesPerSec"]) * len(pods_group["summaries"])
                     for pods_group in pods_groups]
                )
                if not self._check_reserve(geo, added_net, "net"):
                    self.set_info("{}: not enough Net-bandwidth reserve for cluster {}, suggested change: {} -> {}".format(service,
                                                                                                                           geo,
                                                                                                                           net_guarantee,
                                                                                                                           new_net_guarantee))
                    new_net_guarantee = None

            # TODO: put suggested CPU guarantee to the link, not supported by YP now. SWAT-7000
            link += "/reallocate_yp_pods"
            return link, Value(cpu_guarantee, net_guarantee), Value(new_cpu_guarantee, new_net_guarantee)
        except Exception as e:
            logging.exception(e)
        return link, None, Value(None, None)

    def _run_tidiness(self, nanny_token, service, new_val):
        if not self.Parameters.tidiness:
            tidiness = sdk2.Resource["BEGEMOT_NANNY_TIDINESS"].find(state='READY').first()
        else:
            tidiness = self.Parameters.tidiness
        path = str(sdk2.ResourceData(tidiness).path)
        stderr = self.log_path() / 'tidiness_{}_stderr.txt'.format(service)
        stdout = self.log_path() / 'tidiness_{}_stdout.txt'.format(service)
        env = os.environ.copy()
        env['OAUTH_NANNY'] = nanny_token
        args = ['-s', service, '--commit', 'yp-reallocate']
        if new_val.cpu:
            args += ['--cpu', str(new_val.cpu)]
        if new_val.net:
            args += ['--network-guarantee', str(new_val.net)]
        args += ['--force-unsafe', 'yes']
        cmd = ' '.join([path] + args)
        if self.Parameters.dry_run:
            self.set_info("DRY_RUN. Would normally call 'tidiness {}'".format(' '.join(args)))
            return True

        with stderr.open("w") as err:
            with stdout.open("w") as out:
                p = subprocess.Popen(cmd, stderr=err, stdout=out, shell=True, env=env)
                p.wait()
                if p.returncode:
                    return False

        return True

    def _calc_diff_perc(self, old_guarantee, new_guarantee):
        return (float(new_guarantee) / old_guarantee - 1) * 100

    def _adjust_guarantee(self, old_guarantee, new_guarantee):
        if not new_guarantee:
            return None
        diff_perc = self._calc_diff_perc(old_guarantee, new_guarantee)
        logging.debug("Considering change %s -> %s (%.2f%%)", old_guarantee, new_guarantee, diff_perc)
        if diff_perc > self.Parameters.auto_inc_threshold:
            new_guarantee = int((old_guarantee + 1) * (1 + self.Parameters.auto_inc_threshold / 100) - 1)
        elif diff_perc < -self.Parameters.auto_dec_threshold:
            new_guarantee = int((old_guarantee - 1) * (1 - self.Parameters.auto_dec_threshold / 100) + 1)
        diff_perc = self._calc_diff_perc(old_guarantee, new_guarantee)
        if abs(diff_perc) < self.Parameters.ignore_threshold:
            return None
        return new_guarantee

    def _check_reserve(self, geo, added, resource_key):
        if added <= 0:
            return True

        yp_token = sdk2.Vault.data(self.Parameters.secret_owner, self.Parameters.yp_token)
        rsp = requests.post(
            url="https://{}.yp.yandex.net:8443/ObjectService/SelectObjects".format(geo),
            data=json.dumps({
                "object_type": "account",
                "filter": {
                    "query": '[/meta/id] = "abc:service:{}"'.format(self.Parameters.abc)
                },
                "selector": {
                    "paths" : [
                        "/meta/id",
                        "/status/resource_usage",
                        "/spec/resource_limits"
                    ]
                },
            }),
            headers={
                "Accept": "application/json",
                "Content-Type": "application/json",
                "Authorization": "OAuth {}".format(yp_token)
            }
        )
        usage = float(rsp.json()["results"][0]["values"][1]["per_segment"]["default"][resource_key]["capacity"])
        guarantee = float(rsp.json()["results"][0]["values"][2]["per_segment"]["default"][resource_key]["capacity"])
        max_usage = guarantee * (1 - self.Parameters.reserved_cpu / 100)
        if usage + added <= max_usage:
            return True
        return False

    def _do_reallocation(self, service, prev_val, new_val, nanny_token):
        success = self._run_tidiness(nanny_token, service, new_val)
        if not success:
            self.set_info("Failed reallocation for {}. See tidiness log for more info".format(service))
            self.server.notification(
                subject='[RecalcCPUGuarantees] {}: autoreallocation failed'.format(service),
                body='By task <a href="https://sandbox.yandex-team.ru/task/{}">{}</a>. Nanny-tidiness returned non-zero code'.format(self.id, self.id),
                recipients=self.Parameters.recipients,
                transport=ctn.Transport.EMAIL,
                urgent=False,
                type='html'
            )
            return

        body = 'By task <a href="https://sandbox.yandex-team.ru/task/{}">{}</a>.<br>'.format(self.id, self.id)
        if new_val.cpu:
            body += 'CPU guarantee changed from {} to {}.<br>'.format(prev_val.cpu, new_val.cpu)
        if new_val.net:
            body += 'Net guarantee changed from {} to {}.<br>'.format(prev_val.net, new_val.net)
        self.server.notification(
            subject='[RecalcCPUGuarantees] {}: autoreallocation started)'.format(service),
            body=body,
            recipients=self.Parameters.recipients,
            transport=ctn.Transport.EMAIL,
            urgent=False,
            type='html'
        )

    def _reallocate_services(self, stat):
        nanny_token = sdk2.Vault.data(self.Parameters.secret_owner, self.Parameters.nanny_token)
        for key in stat:
            for service in stat[key]:
                nanny_services = self._get_nanny_services(service[2], key)
                for nanny_service in nanny_services:
                    logging.debug("Processing %s...", nanny_service)
                    cpu_lack_coef = None
                    if service[0].cpu > service[1][1]:
                        cpu_lack_coef = float(service[0].cpu) / service[1][1]
                    elif service[0].cpu < service[1][0]:
                        cpu_lack_coef = float(service[0].cpu) / service[1][0]

                    net_lack_coef = None
                    if service[0].net > NETWORK_USAGE_DEFAULT_TARGET[1]:
                        net_lack_coef = float(service[0].net) / NETWORK_USAGE_DEFAULT_TARGET[1]
                    elif service[0].net < NETWORK_USAGE_DEFAULT_TARGET[0]:
                        net_lack_coef = service[0].net / NETWORK_USAGE_DEFAULT_TARGET[0]

                    coefs = Value(cpu_lack_coef, net_lack_coef)
                    _, old_guarantee, new_guarantee = self._get_reallocation_info(nanny_service, service[2], coefs, nanny_token)

                    if new_guarantee.cpu or new_guarantee.net:
                        if new_guarantee.cpu:
                            self.set_info("{}: task is going to change CPU guarantee from {} to {} ({:+.2f}%)".format(
                                nanny_service, old_guarantee.cpu, new_guarantee.cpu, (float(new_guarantee.cpu) / old_guarantee.cpu - 1) * 100
                            ))
                        if new_guarantee.net:
                            self.set_info("{}: task is going to change net guarantee from {} to {} ({:+.2f}%)".format(
                                nanny_service, old_guarantee.net, new_guarantee.net, (float(new_guarantee.net) / old_guarantee.net - 1) * 100
                            ))
                        self._do_reallocation(nanny_service, old_guarantee, new_guarantee, nanny_token)

    def _create_or_update_tickets(self, stat, names, start_time, end_time):
        # This method actually do nothing useful now. May be we will reanimate it later
        return [], [], {}

        import startrek_client
        token = sdk2.Vault.data(self.Parameters.secret_owner, self.Parameters.secret)
        client = startrek_client.Startrek(token=token, useragent='sandbox-task', base_url='https://st-api.yandex-team.ru')
        tickets_created = []
        tickets_commented = []
        tickets_ignored = {}

        for key in stat:
            task_tag = "cpu-auto"
            shard_tag = "shard:%s" % key
            charts = []
            locations = []
            date = datetime.now()
            issue = None
            need_new_issue = False
            for service in stat[key]:
                location_tag = service[2]
                locations.append(location_tag)
                chart_url, img_url = self._get_chart_screenshot(service[2], key, CPU_USAGE_PROPORTION, start_time, end_time)
                nanny_service = self._get_nanny(service[2], key)
                cpu_lack_coef = service[0] / service[1][1] if service[0] > service[1][1] else service[0] / service[1][0]
                if cpu_lack_coef <= 1:
                    continue

                nanny_token = sdk2.Vault.data(self.Parameters.secret_owner, self.Parameters.nanny_token)
                nanny_link, guarantees, _ = self._get_reallocation_info(nanny_service, service[2], cpu_lack_coef, nanny_token)

                guarantees_change = "({} -> {})".format(guarantees[0], guarantees[1]) if guarantees is not None else ""

                # to avoid '))' in link in startrek issue
                chart_url = chart_url.replace("))", "%29%29")
                charts.append("\n".join(["%s, %s [((%s link))]: <{%.2f%% от гарантии вместо %s%%. Увеличить в %.2f раз %s" % (service[2], date.date(), nanny_link, service[0], service[1], cpu_lack_coef, guarantees_change),
                    "((%s %s))}>" % (chart_url, img_url)]))
                tags = [shard_tag, task_tag, location_tag]

                query = " AND ".join([
                    "Queue: %s" % (self.Parameters.startrek_queue),
                    "Status: !Closed"] +
                    ['Tags: "%s"' % tag for tag in tags]
                )
                continue
                found = client.issues.find(query, order=['-created'])
                logging.debug(found)
                if len(found) > 0:
                    issue = found[0]
                    if ('ignored' in issue.tags):
                        tickets_ignored[issue.key] = issue
                        issue = None
                else:
                    need_new_issue = True
            continue
            if issue:
                created_date = issue.createdAt[:10]
                issue.update(summary="%s: Добавить CPU на %s" % (created_date, names[key]),
                             tags=list(set(issue.tags + locations)))
                comment = "\n".join(charts)
                issue.comments.create(text=comment)
                tickets_commented.append(issue)
            elif need_new_issue:
                summary = "%s: Добавить CPU на %s" % (date.date(), names[key])
                description = "\n".join([
                "Created by sandbox task ((https://sandbox.yandex-team.ru/task/%s/view %s))." % (self.id, self.id), "",
                "При пересчете расхода CPU с учетом таргетов обнаружилось, что потребление CPU в сервисе слишком велико. Требуется увеличить гарантии CPU.",
                "", "\n".join(charts), "",
                "Чтобы не получать оповещения об этом сервисе, выставите тег 'ignored' и не закрывайте тикет."])

                kwargs = {
                    "queue": self.Parameters.startrek_queue,
                    "assignee": self.Parameters.assignee,
                    "summary": summary,
                    "description": description,
                    "tags": [task_tag, shard_tag] + locations,
                    "type" : 'task'
                }
                tickets_created.append(client.issues.create(**kwargs))

        return tickets_created, tickets_commented, list(tickets_ignored.values())


    def _report_updated_tickets(self, tickets, msg):
        if len(tickets) > 0:
            info = msg
            for ticket in tickets:
                info = "<br>".join([info, '<a href="https://st.yandex-team.ru/%s">%s</a> %s' % (ticket.key, ticket.key, ticket.summary)])
            self.set_info(info, do_escape=False)


    def on_execute(self):
        shards = []
        no_target_shards = []
        names = {}
        cpu_targets = {}
        geo = ['sas', 'vla']
        if self.Parameters.itype == 'begemot':
            no_target_names = ['Antirobot', 'ImagesCV', 'MarketWizard', 'Megamind', 'ServiceWizard']
            for name, info in Begemots:
                if info.release and info.prj is not None and info.prj not in SKIP_PRJ:
                    shards.append(info.prj)
                    cpu_targets[info.prj] = info.cpu_range if self.Parameters.auto_target_cpu else [0, self.Parameters.target_cpu]
                    names[info.prj] = name
                    if name in no_target_names or not self.Parameters.use_rps_target:
                        no_target_shards.append(info.prj)
                        cpu_targets[info.prj] = info.cpu_range if self.Parameters.auto_target_cpu else [0, self.Parameters.target_cpu_no_target_rps]

        if len(self.Parameters.prj) > 0:
            for prj in [x.strip() for x in self.Parameters.prj.split(',')]:
                if prj in SKIP_PRJ:
                    self.set_info('Ignored prj = %s, this is hardcoded' % prj)
                    continue
                names[prj] = prj
                shards.append(prj)
                if self.Parameters.use_rps_target:
                    cpu_targets[prj] = [0, self.Parameters.target_cpu]
                else:
                    no_target_shards.append(prj)
                    cpu_targets[prj] = [0, self.Parameters.target_cpu_no_target_rps]


        period = 3600
        end_time = time.time() - period * 2
        start_time = end_time - period * 24 * 7 # 1 week

        target_stat, no_target_stat = self._collect_golovan_stat(geo, shards, no_target_shards, start_time, end_time, period)
        stat = self._analyze_stat(target_stat, no_target_stat, name, cpu_targets)

        if len(stat) == 0:
            self.set_info("No tickets created, no reallocations done.\n")
        else:
            if self.Parameters.create_tickets:
                created, updated, ignored = self._create_or_update_tickets(stat, names, start_time, end_time)
                self._report_updated_tickets(created, "Created tickets:")
                self._report_updated_tickets(updated, "Updated tickets:")
                self._report_updated_tickets(ignored, "Ignored tickets:")
            if self.Parameters.do_yp_reallocation:
                self._reallocate_services(stat)
