#!/usr/bin/env python
# -*- coding: utf-8 -*-

from datetime import datetime

import sandbox.common.rest as sb
from sandbox.common.utils import get_task_link
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.release_machine.helpers.staff_helper import StaffApi

import json
import logging


NANNY_URL = 'http://nanny.yandex-team.ru'


def check_owner_is_real(staff, login):
    info = None
    try:
        response = staff.get_persons('persons?login={}&_one=1'.format(login), None)
        info = response['result']
    except KeyError:
        logging.warning("Failed to handle staff response for %s:\n%s", login, json.dumps(response, indent=2))
    if info is None or info.get('error_message'):
        return None

    return not info.get('is_deleted') and not info['official'].get('is_robot') and not info['official'].get('is_dismissed')


def find_resource_owner(staff, info):
    owner = info['owner']
    check = check_owner_is_real(staff, owner)
    if check is None:
        return False, None
    return check, owner


def find_task_owner(staff, sandbox, task_id):
    task_info = sandbox.task[task_id].read()
    owner = task_info['owner']
    check = check_owner_is_real(staff, owner)
    if check is None:
        return False, False, owner
    return True, check, owner


def resource_age_days(info):
    time = datetime.strptime(info['time']['created'], '%Y-%m-%dT%H:%M:%SZ')
    return (datetime.now() - time).days


def check_service(service_name, nanny, sandbox, staff):
    report = []
    service_info = nanny.get_service(service_name)
    sandbox_files = service_info['runtime_attrs']['content']['resources']['sandbox_files']
    for file in sandbox_files:
        if file['resource_type'] == 'OTHER_RESOURCE':
            filename = file['local_path']
            resource_info = sandbox.resource[file['resource_id']].read()
            age = resource_age_days(resource_info)

            send_msg, owner = find_resource_owner(staff, resource_info)
            if owner is None:
                has_link, send_msg, owner = find_task_owner(staff, sandbox, file['task_id'])
            else:
                has_link = True

            report.append({'service': service_name, 'file': filename, 'age': age, 'owner': owner, 'staff_link' : has_link, 'send_email' : send_msg})
    return report


def check_other_resources(
        service_match,
        nanny_token=None,
        staff_token=None):

    nanny = NannyClient(NANNY_URL, nanny_token)
    staff = StaffApi(token=staff_token)
    sandbox = sb.Client()

    services = nanny.get_dashboard_services(service_match)
    reports = []
    for service_name in services:
        reports += check_service(service_name, nanny, sandbox, staff)

    return reports


def create_messages(reports, default_receivers, task_id):
    messages = {}
    for report in reports:
        report.update({
            'file_url': 'https://nanny.yandex-team.ru/ui/#/services/catalog/{service}/files#{file}'.format(**report),
        })
        warning_line = "The service {service} contains a file <a href={file_url}>{file}</a> with type OTHER_RESOURCE and age of {age} days, owned by {owner}.".format(**report)
        receivers = list(default_receivers)
        if report['send_email']:
            receivers.append(report['owner'])
        for rec in receivers:
            if rec not in messages:
                messages[rec] = warning_line
            else:
                messages[rec] = "<br>".join([messages[rec], warning_line])

    for rec, msg in messages.iteritems():
        messages[rec] = "<br>".join([msg,
        'OTHER_RESOURCES will not be updated by releases. They can trigger incidents like <a href="https://st.yandex-team.ru/SPI-5445">SPI-5445</a>.',
        "Please, use 'type' for 'ya upload' or replace OTHER_RESOURCES with the correct one from any release.",
        '', 'Generated by task <a href="%s">%s</a>. You are the resource owner or default receiver.' % (get_task_link(task_id), task_id)])
    return messages


def create_task_info(reports):
    if len(reports) == 0:
        return "No services with OTHER_RESOURCES.\n"
    info = "Found Nanny services with OTHER_RESOURCES:\n"
    for report in reports:
        service_url = "https://nanny.yandex-team.ru/ui/#/services/catalog/%s/" % (report['service'])
        file_url = service_url + "files#%s" % (report['file'])
        info += '<a href="%s">%s</a>: file <a href="%s">%s</a>' % (service_url, report['service'], file_url, report['file'])
        if report['owner'] is not None:
            info += ", owned by "
            if report['staff_link']:
                staff_url = "https://staff.yandex-team.ru/%s" % (report['owner'])
                info += '<a href="%s">%s</a>' % (staff_url, report['owner'])
            else:
                info += report['owner']
        info += ".\n"

    return info
