# -*- coding: utf-8 -*-
import json
import logging
import subprocess
import os

from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as sdk_constants
import sandbox.projects.common.dynamic_models.resources as models_res
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.helpers.ab_helper as rm_ab
import sandbox.sdk2 as sdk2


class BuildWebDynamicModelsFlag(rm_bt.BaseReleaseMachineTask):
    """
        * Create single flag for flags.json
        * Create ab test-id with this flag
        * make "release" action in abt
    """
    class Requirements(sdk2.Task.Requirements):
        disk_space = 5000

    class Parameters(rm_params.BaseReleaseMachineParameters):
        kill_timeout = 60 * 60  # 1 hour
        # video_mapping_tag = sdk2.parameters.Integer("Tag for video mapping file")
        debug_mode = sdk2.parameters.Bool("Debug mode", default=False)
        arcadia_patch = sdk2.parameters.String(
            "Apply patch for mappings-exporter. Doc: https://nda.ya.ru/3QTTV4",
            multiline=True,
            default="",
        )
        with sdk2.parameters.Output():
            testid = sdk2.parameters.String("Test id created; deploy prepare done; visit https://ab.yandex-team.ru/deploying/items#BULK_TESTED")

    class Context(sdk2.Task.Context):
        single_flag_res_id = 0

    def on_enqueue(self):
        super(BuildWebDynamicModelsFlag, self).on_enqueue()
        res = models_res.DynamicModelsSingleFlag(self, "Production dynamic models flag", "prod_models_flag.json")
        self.Context.single_flag_res_id = res.id

    def on_execute(self):
        super(BuildWebDynamicModelsFlag, self).on_execute()
        ab_api = rm_ab.ABApi(
            token=sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, "ab_formulas_release_token"),
            test=self.Parameters.debug_mode,
        )
        testid = getattr(self.Parameters, "testid", None)
        logging.info("MAIN: Test id: %s", testid)
        if not testid:
            single_flag = self.get_single_flag()
            testid = self.create_ab_test(ab_api, single_flag)
            self.save_data(self.Context.single_flag_res_id, single_flag)
            self.Parameters.testid = str(testid)
        else:
            self.set_info("Testid is already created: {}".format(testid))
        data_for_deploy_creation = {
            "creator": "robot-srch-releaser",
            "special_type": "fml",
            "title": "Release web-formulas and models flags",
            "testid": str(testid)
        }
        logging.info('MAIN: Going to deploy test-id')
        ab_api.deploy_test_id(data_for_deploy_creation)
        logging.info('MAIN: Finished')

    def get_single_flag(self):
        target = 'search/formula_chooser/mappings_exporter'
        binary_name = 'mappings_exporter'
        output_directory = 'build_output'
        logging.info('FormulaChooserTool: Building {} into {}'.format(target, output_directory))
        token = sdk2.Vault.data(self.owner, name='ARC_TOKEN')
        with arcadiasdk.mount_arc_path(sdk2.svn.Arcadia.ARCADIA_TRUNK_URL, use_arc_instead_of_aapi=True, arc_oauth_token=token) as arc_root:
            patch = None
            if self.Parameters.arcadia_patch:
                patch = self.Parameters.arcadia_patch
                path_to_patch = sdk2.svn.Arcadia.apply_patch(arc_root, patch, self.path())
                if patch.startswith("arc:"):
                    # When applying (arc:review-id) use zipatch  downloaded from Arcanum
                    patch = path_to_patch
            arcadiasdk.do_build(
                build_system=sdk_constants.SEMI_DISTBUILD_BUILD_SYSTEM,
                source_root=arc_root,
                targets=[target],
                results_dir=output_directory,
                clear_build=False,
                patch=patch,
            )
        logging.info('FormulaChooserTool: Finished building ' + target)
        binpath = os.path.join(os.curdir, output_directory, target, binary_name)
        logging.info('FormulaChooserTool: Start call of')
        single_flag = subprocess.check_output([binpath, "dump-web-flags", "-f"])
        logging.info('FormulaChooserTool: Finish call')
        return single_flag

    def create_ab_test(self, ab_api, single_flag):
        data = {
            "queue_id": "1",
            "title": "Web production formulas and models-bundle flags",
            "type": "ABT",
            "params": single_flag,
            "replace_token": "<testid>"
        }
        logging.info("MAIN: Try to create test id with data:\n%s", json.dumps(data, indent=2))
        self.Context.testid_created = ab_api.create_test_id(data)
        return self.Context.testid_created["testid"]

    @staticmethod
    def save_data(res_id, data):
        res_path = str(sdk2.ResourceData(sdk2.Resource[res_id]).path)
        open(res_path, "w").write(data)

    @property
    def footer(self):
        testid = getattr(self.Parameters, "testid", None)
        if testid:
            url = rm_ab.ABApi.TEST_URL if self.Parameters.debug_mode else rm_ab.ABApi.PROD_URL
            return "Test id created: <a href={url}testid/{testid}>{testid}</a>".format(url=url, testid=testid)
