import os
import logging
import tarfile

import sandbox.projects.websearch.flags_provider.resources as res

from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.projects.common.constants.constants import SEMI_DISTBUILD_BUILD_SYSTEM, RELEASE_BUILD_TYPE
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import copy_path, make_folder


FLAGS_PROVIDER_BASE_PATH = os.path.join("web", "daemons", "flags_raw_provider")
FLAGS_PROVIDER_BINARY_PATH = os.path.join(FLAGS_PROVIDER_BASE_PATH, "flags_raw_provider")
FLAGS_PROVIDER_EVLOGDUMP_PATH = os.path.join(FLAGS_PROVIDER_BASE_PATH, "tools", "event_log_dump", "event_log_dump")
FLAGS_PROVIDER_CONF_PATH = os.path.join(FLAGS_PROVIDER_BASE_PATH, "conf")

VERTICALS = ['WEB', 'SHARED']


class BuildFlagsProviderCore(ReleaseToNannyTask2, sdk2.Task):
    ''' Simple task, which compile Flags Raw Provider daemon and allow to release it. '''

    class Requirements(sdk2.Task.Requirements):
        disk_space = 100000

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.CheckGroup("Verticals", required=True) as verticals:
            for v in VERTICALS:
                setattr(verticals.values, v, verticals.Value(v, checked=True))

        with sdk2.parameters.Group("Subversion Parameters") as subversion_block:
            arcadia_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia:',
                                                     default_value=Arcadia.trunk_url(),
                                                     required=True)

    resources_map = {
        'WEB': {
            'binary': res.FlagsProviderBinaryBundleResourseType,
            'config': res.FlagsProviderConfigResourseType,
            'instancectl': res.FlagsProviderInstanceCtlResourseType
        },
        'SHARED': {
            'binary': res.FlagsProviderBinaryBundleShared,
            'config': res.FlagsProviderConfigShared,
            'instancectl': res.FlagsProviderInstanceCtlShared
        }
    }

    def on_execute(self):
        build_dir = os.path.abspath("build")
        conf_dir = os.path.abspath("conf_dir")

        svn_info = dict(sdk2.svn.Arcadia.info(self.Parameters.arcadia_url))

        revision = svn_info['entry_revision']
        arcadia_url = self.Parameters.arcadia_url

        make_folder(build_dir)
        os.chdir(build_dir)

        with sdk.mount_arc_path(arcadia_url) as arcadia_dir:
            logging.info("Arcadia repo path: " + arcadia_dir)

            copy_path(os.path.join(arcadia_dir, FLAGS_PROVIDER_CONF_PATH), conf_dir)

            logging.info("Prepare to build binaries.")
            logging.info("RELEASE_BUILD_TYPE: " + RELEASE_BUILD_TYPE)

            sdk.do_build(
                SEMI_DISTBUILD_BUILD_SYSTEM,
                arcadia_dir,
                [FLAGS_PROVIDER_BASE_PATH],
                RELEASE_BUILD_TYPE,
                clear_build=True,
                results_dir=build_dir
            )
            logging.info("Binary is built. Start to create resourse.")

        pkg_name = "flags_provider_bundle_r{}".format(revision)
        package_dir = os.path.join(build_dir, pkg_name)
        os.chdir(build_dir)
        make_folder(package_dir)

        copy_path(
            os.path.join(build_dir, FLAGS_PROVIDER_BINARY_PATH),
            os.path.join(package_dir, "flags_provider")
        )

        copy_path(
            os.path.join(build_dir, FLAGS_PROVIDER_EVLOGDUMP_PATH),
            os.path.join(package_dir, "evlogdump")
        )

        for vertical in self.Parameters.verticals:
            resource = sdk2.ResourceData(BuildFlagsProviderCore.resources_map[vertical]['binary'](
                self, "{}@{}".format("flags_provider_bundle", revision), "{}_{}.tar.gz".format(pkg_name, vertical.lower())
            ))

            with tarfile.open(str(resource.path), 'w:gz') as tar:
                tar.add(pkg_name)

            resource.ready()

            for filename in ['instancectl.conf', 'config_']:
                if filename == 'instancectl.conf':
                    resource = sdk2.ResourceData(BuildFlagsProviderCore.resources_map[vertical]['instancectl'](
                        self, "{}@{}".format(filename, revision), "instancectl_{}.conf".format(vertical.lower())
                    ))
                elif filename == 'config_':
                    filename = filename + vertical.lower() + '.json'
                    resource = sdk2.ResourceData(BuildFlagsProviderCore.resources_map[vertical]['config'](
                        self, "{}@{}".format('config.json', revision), filename
                    ))
                else:
                    raise RuntimeError('unkwnown file')

                with open(os.path.join(conf_dir, filename), 'r') as f:
                    resource.path.write_bytes(f.read())
                resource.ready()
