# -*- coding: utf-8 -*-

import json
import logging
import os
import time

from sandbox.projects import resource_types

from sandbox.projects.common import apihelpers
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common.betas.beta_api import BetaApi
from sandbox.projects.common.search import bisect
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search.bisect import BisectInfo
from sandbox.projects.common.search.settings import WebSettings
from sandbox.projects.websearch import utils as websearch_utils

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.channel import channel


class Params:

    class Version(sp.SandboxIntegerParameter):
        name = "version"
        description = "Search diff reason in version"
        required = True

    class MiddleBeta(sp.SandboxStringParameter):
        name = "middle_beta"
        description = "Middle search beta"
        required = True

    class WaitAfterBisectEnd(sp.SandboxIntegerParameter):
        name = "wait_after_bisect_end"
        description = "Interval before bisect end and component shutdown (seconds)"
        default_value = 1
        required = False

    class UseMetahost2(sp.SandboxBoolParameter):
        name = "use_metahost2"
        description = "Use metahost2 instead of metahost to redirect to local search"
        default_value = True  # TODO looks like it works better than the others, further investigation required...
        required = False

    class UseSrcRwr(sp.SandboxBoolParameter):
        name = "use_srcrwr"
        description = "Use srcrwr instead of metahost to redirect to local search"
        default_value = False
        required = False


def _get_middlesearch(
    binary_id,
    config_file,
    data_id,
    archive_model_id=None,
):
    data_path = channel.task.sync_resource(data_id)
    return sc.Middlesearch(
        is_int=False,
        work_dir=channel.task.abs_path(),
        binary=channel.task.sync_resource(binary_id),
        config_file=config_file,
        pure_dir=os.path.join(data_path, "pure"),
        rearrange_dir=os.path.join(data_path, "rearrange"),
        archive_model_path=channel.task.sync_resource(archive_model_id) if archive_model_id else None,
    )


class BisectMiddlesearchDiffReason(bisect.Bisect):
    """
        Ищем номер комита в middlesearch, вызвавший diff в выдаче на указанном запросе
        (+ указывается позиция в которой изменился документ).
        Для определения дипазона ревизий в которых нужно искать diff указывается номер версии middlesearch.
        Данные/конфиг для запусков middlsearch также указываются в параметрах задачи.

        В процессе поиска поднимаются middlesearch-и разных ревизий, через которые прогоняется указанный запрос
        (запросы перенаправляются на локально поднятый middlesearch добалением параметров к запросу-входному параметру задачи)

        На выходе - номер ревизии, изменившей выдачу + описание, какой документ на какой поменялся (url-ы).
    """

    type = "BISECT_MIDDLESEARCH_DIFF_REASON"
    input_parameters = bisect.Bisect.input_parameters + (
        Params.Version,
        Params.MiddleBeta,
        Params.WaitAfterBisectEnd,
        Params.UseMetahost2,
        Params.UseSrcRwr,
        sc.DefaultMiddlesearchParams.Data,
        sc.DefaultMiddlesearchParams.ArchiveModel,
    )

    def on_execute(self):
        ver_major = self.ctx[Params.Version.name]
        self.set_bisect_info(BisectInfo.version, ver_major)
        # put config filename to self.config_file
        self.config_file = 'middlesearch.cfg.orig'
        self.import_config(self.ctx[Params.MiddleBeta.name], self.config_file)

        # mmeta-web-248-3.hamster.yandex.ua
        ver_minor = int(self.ctx[Params.MiddleBeta.name].split('.', 1)[0].rsplit('-', 1)[-1])

        branch_path_template = 'arcadia:/arc/branches/middle/stable-{}'
        rev_min, _, _ = bisect.branch_last_rev_first_minor_last_minor(branch_path_template.format(ver_major-1))
        rev_max, minor_rev_min, minor_rev_max = bisect.branch_last_rev_first_minor_last_minor(
            branch_path_template.format(ver_major),
            ver_minor=ver_minor,
        )

        minor_build_info = bisect.get_build_info(
            'ws-middle-{}'.format(ver_major),
            'BUILD_MIDDLESEARCH_RELEASE_LINUX',
            minor_rev_min,
            minor_rev_max,
        )

        self.set_bisect_info(BisectInfo.revisions, '{}:{}'.format(rev_min, rev_max))
        build_info = bisect.get_build_info(
            'ws-middle-trunk',
            'BUILD_MIDDLESEARCH_RELEASE_LINUX',
            rev_min,
            rev_max,
        ) + minor_build_info
        self.bisect(build_info)

    @staticmethod
    def import_config(beta_name, filename):
        sdms_beta_suffix = '.hamster.yandex.'
        short_beta_name = beta_name.split(sdms_beta_suffix)[0]
        instances = BetaApi.fromurl().get_instances(short_beta_name, 'web-mmeta')
        logging.info("Instances = %s", instances)
        websearch_utils.import_config(instances, filename=filename)

    def get_docs(self, row, requrl):
        """
            Method for call from bisect algo
        """
        logging.debug(bisect.format_row(row))
        task_id = row['task_id']
        resources = channel.sandbox.list_resources(task_id=task_id, resource_type='RANKING_MIDDLESEARCH_EXECUTABLE')
        if not resources:
            eh.check_failed('Fail get resource RANKING_MIDDLESEARCH_EXECUTABLE from ' + str(task_id))
        resp = self._get_response(resources[0].id, requrl)
        try:
            resp = json.loads(resp)
        except Exception, e:
            logging.error('fail parse {}:\n{}'.format(resp, e))
            raise
        docs = resp['searchdata']['docs']
        if not docs:
            logging.debug(resp)
            self.set_info("WARNING: Empty search result, be ready for problems")
        result = []
        for i, doc in enumerate(docs, 1):
            doc_data = doc['url']
            logging.debug("#%s %s", i, doc_data)
            for attr in self.doc_attrs:
                doc_data += ' {}={}'.format(attr, repr(doc.get(attr)))
            result.append(doc_data)
        logging.debug("Docs:\n%s\n\n", json.dumps(result, indent=4))
        return result

    def _get_response(self, exe_res_id, requrl):
        mp = sc.DefaultMiddlesearchParams

        data_id = utils.get_and_check_last_resource_with_attribute(
            resource_type=resource_types.MIDDLESEARCH_DATA,
            attr_name=WebSettings.testenv_middle_resources_attr_name("mmeta", "data")
        ).id

        archive_model_id = apihelpers.get_last_resource_with_attribute(
                resource_types.DYNAMIC_MODELS_ARCHIVE, attribute_name="current_production"
        ).id

        search_component = _get_middlesearch(
            binary_id=exe_res_id,
            config_file=self.config_file,
            data_id=self.ctx.get(mp.Data.name, None) or data_id,
            archive_model_id=self.ctx.get(mp.ArchiveModel.name, None) or archive_model_id,
        )
        search_component.start()
        search_component.wait()
        url = requrl + '&json_dump=searchdata'
        if utils.get_or_default(self.ctx, Params.UseSrcRwr):
            for src in (
                'WEB',
                'WEB_MISSPELL',
            ):
                url += '&srcrwr={}%3A{}%3A{}'.format(src, self.local_host, search_component.port)
        if utils.get_or_default(self.ctx, Params.UseMetahost2):
            for src in (
                'WEB',
                'WEB_MISSPELL',
            ):
                url += '&metahost2={}%3A{}%3A{}'.format(src, self.local_host, search_component.port)
        else:
            for src in (
                'DIVERSITY_TR',
                'DIVERSITY',
                'WEB',
                'WEB_MISSPELL',
            ):
                url += '&metahost={}%3A{}%3A{}'.format(src, self.local_host, search_component.port)
        try:
            return self.read(url)
        finally:
            time_to_wait = int(utils.get_or_default(self.ctx, Params.WaitAfterBisectEnd))
            logging.info("Make a chance to remotely debug this (%s seconds)", time_to_wait)
            time.sleep(time_to_wait)

            search_component.stop()


__Task__ = BisectMiddlesearchDiffReason
