# -*- coding: utf-8 -*-

import logging
from time import time
from shutil import copytree
from os import path

from sandbox.projects import resource_types
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.parameters import LastReleasedResource

from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.utils import svn_revision
from sandbox.projects.common.utils import get_bsconfig


class CheckoutArcadiaFromUrl(SandboxArcadiaUrlParameter):
    description = 'rearrange data svn url'
    default_value = 'arcadia:/arc/branches/middle/pre-stable-31/data'


class UserdataIndexResId(LastReleasedResource):
    name = 'userdata_index_resource_id'
    description = 'userdata index with d.c2n and geoa.c2p'
    resource_type = resource_types.USERDATA_INDEX
    required = False


class BuildMiddlesearchDataBundle(ArcadiaTask):
    type = 'BUILD_MIDDLESEARCH_DATA_BUNDLE'

    input_parameters = (CheckoutArcadiaFromUrl, UserdataIndexResId)

    result_data_type = resource_types.MIDDLESEARCH_DATA
    result_data_bundle_type = resource_types.MIDDLESEARCH_DATA_BUNDLE

    def __init__(self, task_id=0):
        SandboxTask.__init__(self, task_id)
        self.ctx['build_bundle'] = False

    def on_enqueue(self):
        self.do_on_enqueue()
        SandboxTask.on_enqueue(self)

    def do_on_enqueue(self):
        self.bsconfig_id = str(int(time()))
        mdb_resource = self._create_resource(
            self.descr,
            self.abs_path('middlesearch_data_bundle-%s.tar.gz' % self.bsconfig_id),
            self.result_data_bundle_type)
        md_resource = self._create_resource(
            self.descr,
            self.abs_path('middlesearch_data-%s' % self.bsconfig_id),
            self.result_data_type)
        self.ctx['bundle_id'] = mdb_resource.id
        self.ctx['directory_id'] = md_resource.id

    def _copy_userdata_to_rearrange(self, data_dir):
        required_files = [
            'd.c2n',
            'geoa.c2p',
            'geo.stat',
            'indexuserowndat',
            'indexuserowninv',
            'indexuserownkey'
            ]
        userdata_src = self.sync_resource(self.ctx[UserdataIndexResId.name])
        for f in required_files:
            if not path.exists(path.join(userdata_src, f)):
                raise SandboxTaskFailureError("file %s doesn't exist in userdata directory" % f)
        userdata_dst = path.join(data_dir, 'rearrange', 'userdata')
        copytree(userdata_src, userdata_dst)

    def do_execute(self):
        mdb_res_path = channel.sandbox.get_resource(self.ctx['bundle_id']).path
        md_resource = channel.sandbox.get_resource(self.ctx['directory_id'])

        data_dir = md_resource.path

        Arcadia.export(url=self.ctx[CheckoutArcadiaFromUrl.name], path=data_dir)

        if self.ctx[UserdataIndexResId.name] and 'userdata_copied' not in self.ctx:
            self._copy_userdata_to_rearrange(data_dir)
            self.ctx['userdata_copied'] = 1

        rearrInfo = open(path.join(data_dir, 'version_info'), 'w')
        print >>rearrInfo, "svn url: %s" % self.ctx[CheckoutArcadiaFromUrl.name]
        print >>rearrInfo, "sandbox task: %s" % channel.sandbox.get_task(self.id).url
        rearrInfo.close()

        self.ctx['arcadia_revision'] = svn_revision(data_dir)

        bsconfig_path = get_bsconfig()
        logging.info('bsconfig_path = {}'.format(bsconfig_path))

        tarCmd = ['tar', 'cz', '-C', path.dirname(data_dir), '-f', mdb_res_path, path.basename(data_dir)]
        self._subprocess(tarCmd, wait=True)
        self._subprocess(
            "{} shard_init --torrent --rescan --logfile {} --force {}".format(
                bsconfig_path, self.log_path('bsconfig.log'), md_resource.file_name),
            check=True,
            wait=True,
            log_prefix="bs_config")

        self.mark_resource_ready(md_resource)


__Task__ = BuildMiddlesearchDataBundle
