# coding: utf-8

import logging

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import sandboxapi

from sandbox.projects.common import utils
from sandbox.projects.common.middlesearch import single_host as msh
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import settings
from sandbox.projects.common.search import production
from sandbox.projects.common.search import bugbanner
from sandbox.projects import PatchQueries as pq
import sandbox.projects.images.metasearch.task as img_meta_task
from sandbox.projects.websearch.middlesearch.PriemkaMiddlesearchBinary import params as mp
from sandbox.projects.websearch.middlesearch.PriemkaMiddlesearchBinary import res as middle_res

BIN_TYPES = ["web", "img", "video"]
MIDDLE_ATTRS = middle_res.get_testenv_attrs("mmeta")
DISK_SPACE_TO_PATCH = {
    "web": 256,  # 256Mb
    "img": 5 * 1024,  # 5Gb
    "video": 3 * 1024,  # 3Gb
}
DISK_SPACE_TO_SHOOT = {
    "web": 110 * 1024,  # 110Gb
    "img": 130 * 1024,  # 130Gb (SEARCH-7831)
    "video": 55 * 1024,  # 40Gb
}
SETTINGS_BY_TYPE = {
    "web": settings.WebSettings,
    "img": settings.ImagesSettings,
    "video": settings.VideoSettings,
}

TASK_BY_TYPE = {
    "web": "GET_MIDDLESEARCH_RESPONSES",
    "img": "IMAGES_GET_MIDDLESEARCH_RESPONSES",
    "video": "VIDEO_GET_MIDDLESEARCH_RESPONSES",
}
SHARD_ATTRS = (
    settings.META_SHARD_INSTANCE_ATTRIBUTE_NAME,
    settings.SHARD_INSTANCE_ATTRIBUTE_NAME,
    settings.SNIP_SHARD_INSTANCE_ATTRIBUTE_NAME,
)


class WebExecutableId(sp.ResourceSelector):
    name = "web_executable_res_id"
    description = "Custom middlesearch executable"
    required = False
    resource_type = mp.RES_TYPES["binary"]["web"]["mmeta"]


class WebMmetaModelsId(sp.ResourceSelector):
    name = "web_models_res_id"
    description = "Custom middlesearch models archive"
    required = False
    resource_type = mp.RES_TYPES["models"]["web"]["mmeta"]


class CheckOnBinary(sp.SandboxBoolGroupParameter):
    name = "check_on_binary"
    description = "Check on binary"
    choices = [(p, p) for p in BIN_TYPES]
    default_value = " ".join(BIN_TYPES)
    sub_fields = {
        "web": [WebExecutableId.name, WebMmetaModelsId.name]
    }


class CheckMiddlesearchExperiment(bugbanner.BugBannerTask):
    """
        Протестировать экспериментальные параметры.

        Запускает single-host таск с продакшен-средним и 2мя продакшен базовыми.
        Стреляет в них пропатченными запросами с добавлением нужных cgi.
        SEARCH-2726
    """
    type = 'CHECK_MIDDLESEARCH_EXPERIMENT'

    input_parameters = [CheckOnBinary, pq.AddCgiParameter, WebExecutableId, WebMmetaModelsId]
    execution_space = 4 * 1024  # 4 Gb
    cores = 1
    REQ_LIMIT = 3000

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.WebMiddleSearch)

        if not self.list_subtasks():
            for bin_type in utils.get_or_default(self.ctx, CheckOnBinary).split(" "):
                patched_queries = self._get_patched_queries(bin_type)
                shooter_input_ctx = self._shoot_input(bin_type, patched_queries)
                self._create_shoot_task(bin_type, shooter_input_ctx)
            utils.wait_all_subtasks_stop()
        utils.restart_broken_subtasks()

    def _shoot_input(self, bin_type, patched_queries):
        input_ctx = _get_mmeta_ctx(bin_type)
        setting = SETTINGS_BY_TYPE[bin_type]
        input_ctx.update(self._get_custom_middlesearch_data(bin_type))
        input_ctx.update(_get_base_exe_and_models(bin_type, setting))
        input_ctx.update(_get_base_configs(bin_type, setting))
        input_ctx.update(self._get_base_db(bin_type, setting))
        input_ctx.update({
            "ignore_got_error": True,
            "queries_resource_id": patched_queries,
            "get_all_factors": True,
            "process_count": 4,
            "queries_limit": self.REQ_LIMIT,
            "test_doubled_queries": False,
            "test_disabled_cache": True,
        })
        if bin_type == "img":
            # attrs for db grows in patched queries slowly
            input_ctx["early_db_selection"] = False
        return input_ctx

    def _create_shoot_task(self, binary_type, shooter_input_ctx):
        logging.info("Shooter for '%s' input ctx: %s", binary_type, shooter_input_ctx)
        self.create_subtask(
            task_type=TASK_BY_TYPE[binary_type],
            execution_space=DISK_SPACE_TO_SHOOT[binary_type],
            description="Shoot '{}' metasearch single host with experiment cgi".format(binary_type),
            input_parameters=shooter_input_ctx,
            inherit_notifications=True,
        )

    def _get_custom_middlesearch_data(self, bin_type):
        if bin_type != "web":
            return {}
        context = {}
        mmeta_exec_res_id = self.ctx.get(WebExecutableId.name)
        mmeta_models_res_id = self.ctx.get(WebMmetaModelsId.name)
        if mmeta_exec_res_id:
            context[sc.DefaultMiddlesearchParams.Binary.name] = mmeta_exec_res_id
        if mmeta_models_res_id:
            context[sc.DefaultMiddlesearchParams.ArchiveModel.name] = mmeta_models_res_id
        return context

    def _get_patched_queries(self, bin_type):
        logging.info("Get patched queries")
        queries_res_type = mp.RES_TYPES["queries"][bin_type]
        if isinstance(queries_res_type, (list, tuple)):
            queries_res_type = queries_res_type[0]

        mmeta_reqs_res = utils.get_and_check_last_resource_with_attribute(
            resource_type=queries_res_type,
            attr_name=MIDDLE_ATTRS["queries"][bin_type]["attr_name"],
            attr_value=MIDDLE_ATTRS["queries"][bin_type]["attr_value"],
        )
        add_attrs = []
        for attr_key in SHARD_ATTRS:
            shard_instance_attr = mmeta_reqs_res.attributes.get(attr_key)
            if shard_instance_attr:
                add_attrs.append("{}={}".format(attr_key, shard_instance_attr))

        patcher_input_ctx = {
            pq.QueriesParameter.name: mmeta_reqs_res.id,
            pq.AddCgiParameter.name: self.ctx[pq.AddCgiParameter.name],
            pq.ResourceAttrsParameter.name: ",".join(add_attrs),
            pq.QueriesLimitParameter.name: self.REQ_LIMIT,
        }
        logging.info("Patcher input ctx: %s", patcher_input_ctx)
        return self.create_subtask(
            task_type=pq.PatchQueries.type,
            execution_space=DISK_SPACE_TO_PATCH.get(bin_type),
            description="Patch queries with experiment cgi",
            input_parameters=patcher_input_ctx,
            inherit_notifications=True,
        ).ctx["out_resource_id"]

    @staticmethod
    def _get_base_db(bin_type, setting):
        db_res = setting.basesearch_database_resource()
        if bin_type == "web":
            base_db_res_1 = utils.get_and_check_last_resource_with_attribute(
                resource_type=db_res,
                attr_name=setting.default_basesearch_attrs_1(),
            )
            base_db_res_2 = utils.get_and_check_last_resource_with_attribute(
                resource_type=db_res,
                attr_name=setting.default_basesearch_attrs_1(),
            )
        elif bin_type == "img":
            return {}
        else:
            base_db_res_1 = utils.get_and_check_last_resource_with_attribute(
                resource_type=db_res,
                attr_name="video_priemka",
            )
            base_db_res_2 = utils.get_and_check_last_resource_with_attribute(
                resource_type=db_res,
                attr_name="video_priemka",
            )
        return {
            msh.Basesearch1Params.Database.name: base_db_res_1.id,
            msh.Basesearch2Params.Database.name: base_db_res_2.id,
        }


def _get_mmeta_ctx(bin_type):
    logging.info("Get %s mmeta production resources", bin_type)
    mmeta_exe_res = production.get_prod_binary(
        mp.RES_TYPES["binary"][bin_type]["mmeta"], mp.CLUSTERSTATE_MMETA_CONF[bin_type]
    )
    mmeta_exe_res_id = mmeta_exe_res.id
    if bin_type == "web":
        # Hack for SEARCH-7712, please remove after 248-8 is deployed
        mmeta_exe_res_id = _get_last_released_res_id(mp.RES_TYPES["binary"]["web"]["mmeta"])

    models_res_type = mp.RES_TYPES["models"][bin_type]["mmeta"]
    if isinstance(models_res_type, (list, tuple)):
        models_res_type = models_res_type[0]
    mmeta_models_res_id = _get_last_released_res_id(models_res_type)
    mmeta_data_res = utils.get_and_check_last_resource_with_attribute(
        resource_type=mp.RES_TYPES["data"][bin_type],
        attr_name=MIDDLE_ATTRS["data"][bin_type]["attr_name"],
        attr_value=MIDDLE_ATTRS["data"][bin_type]["attr_value"],
    )
    mmeta_cfg_res = utils.get_and_check_last_resource_with_attribute(
        resource_type=mp.RES_TYPES["config"][bin_type],
        attr_name=MIDDLE_ATTRS["config"][bin_type]["attr_name"],
        attr_value=MIDDLE_ATTRS["config"][bin_type]["attr_value"],
    )
    mmeta_index_attrs = MIDDLE_ATTRS["index"].get(bin_type)
    if mmeta_index_attrs:
        mmeta_index_res_id = utils.get_and_check_last_resource_with_attribute(
            resource_type=mp.RES_TYPES["index"][bin_type],
            attr_name=mmeta_index_attrs["attr_name"],
            attr_value=mmeta_index_attrs["attr_value"],
        ).id
    else:
        mmeta_index_res_id = None
    return {
        sc.DefaultMiddlesearchParams.Binary.name: mmeta_exe_res_id,
        sc.DefaultMiddlesearchParams.ArchiveModel.name: mmeta_models_res_id,
        sc.DefaultMiddlesearchParams.Data.name: mmeta_data_res.id,
        sc.DefaultMiddlesearchParams.Config.name: mmeta_cfg_res.id,
        sc.DefaultMiddlesearchParams.Index.name: mmeta_index_res_id,
    }


def _get_base_exe_and_models(bin_type, setting):
    logging.info("Get %s base production exe and models", bin_type)
    base_exe_res = production.get_prod_binary(setting.DEFAULT_BASESEARCH_BINARY_RES_TYPE, "jupiter_base-")
    base_models_res_id = _get_last_released_res_id(setting.DEFAULT_BASESEARCH_MODELS_RES_TYPE)
    if bin_type in ("web", "video"):
        return {
            msh.Basesearch1Params.Binary.name: base_exe_res.id,
            msh.Basesearch2Params.Binary.name: base_exe_res.id,
            msh.Basesearch1Params.ArchiveModel.name: base_models_res_id,
            msh.Basesearch2Params.ArchiveModel.name: base_models_res_id,
        }
    elif bin_type == "img":
        return {
            img_meta_task.BASESEARCH_PARAMS.Binary.name: base_exe_res.id,
            img_meta_task.SNIPPETIZER_PARAMS.Binary.name: base_exe_res.id,
            img_meta_task.BASESEARCH_PARAMS.ArchiveModel.name: base_models_res_id,
            img_meta_task.SNIPPETIZER_PARAMS.ArchiveModel.name: base_models_res_id,
        }


def _get_base_configs(bin_type, setting):
    cfg_res_type = setting.basesearch_config_resource()
    if bin_type == "web":
        return {
            msh.Basesearch1Params.Config.name: utils.get_and_check_last_resource_with_attribute(
                resource_type=cfg_res_type,
                attr_name=setting.default_basesearch_attrs_1(),
            ).id,
            msh.Basesearch2Params.Config.name: utils.get_and_check_last_resource_with_attribute(
                resource_type=cfg_res_type,
                attr_name=setting.default_basesearch_attrs_1(),
            ).id
        }
    cfg = _get_last_released_res_id(cfg_res_type)
    if bin_type == "img":
        return {
            img_meta_task.BASESEARCH_PARAMS.Config.name: cfg,
            img_meta_task.SNIPPETIZER_PARAMS.Config.name: cfg,
        }
    else:
        return {
            msh.Basesearch1Params.Config.name: cfg,
            msh.Basesearch2Params.Config.name: cfg,
        }


def _get_last_released_res_id(res_type):
    return utils.get_and_check_last_released_resource_id(
        resource_type=res_type,
        arch=sandboxapi.ARCH_ANY if res_type.any_arch else sandboxapi.ARCH_LINUX
    )
