# -*- coding: utf-8 -*-

import logging
import random

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp

from sandbox.projects.common import apihelpers
from sandbox.projects.common import dolbilka
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import string
from sandbox.projects import resource_types

COMBINE_GROUP = "Queries to combine"
REGS = ["ru", "kz", "ua", "by", "tr", "id"]
REG_TEMPL = 'use_{}_queries'


class TotalQueryAmount(sp.SandboxIntegerParameter):
    name = "total_query_amount"
    description = "Total query amount"
    default_value = 7000


class Attr(sp.SandboxStringParameter):
    name = 'resource_attrs'
    description = 'Add attributes to final res (ex.: attr1=v1, attr2=v2)'
    default_value = ''


def reg_to_use(reg):
    class UseRegQueries(sp.SandboxBoolParameter):
        name = REG_TEMPL.format(reg)
        description = reg.upper()
        default_value = True
        group = COMBINE_GROUP

    return UseRegQueries


class CombineMiddlesearchQueries(SandboxTask):
    """
        Takes wizarded queries as resources by given attributes
        for each region and combines them randomly.
        Resources will have almost equal proportions in final resource.
    """
    type = "COMBINE_MIDDLESEARCH_QUERIES"

    input_parameters = [
        TotalQueryAmount,
        Attr
    ] + [reg_to_use(reg) for reg in REGS]
    execution_space = 2048  # 2 Gb

    def on_enqueue(self):
        attr = string.parse_attrs(self.ctx[Attr.name])
        self.ctx["combined_queries"] = self.create_resource(
            self.descr, "combined_queries", resource_types.PLAIN_TEXT_QUERIES,
            attributes=attr,
        ).id
        self.ctx["combined_queries_plan"] = self.create_resource(
            self.descr, "combined_queries_plan", resource_types.BASESEARCH_PLAN,
            attributes=attr,
        ).id

    def on_execute(self):
        total_amount = self.ctx[TotalQueryAmount.name]
        obligatory_queries = self._get_obligatory_queries()
        res_paths_list = self._get_queries_res_paths()
        num_of_paths = len(res_paths_list)
        sample_amount = (total_amount - len(obligatory_queries)) / num_of_paths
        logging.info("Get %s random queries from every path", sample_amount)
        out_res_queries_path = channel.sandbox.get_resource(self.ctx["combined_queries"]).path
        out_res_plan_path = channel.sandbox.get_resource(self.ctx["combined_queries_plan"]).path
        with open(out_res_queries_path, "w") as out_f:
            for i, r_path in enumerate(res_paths_list):
                queries = fu.read_lines(r_path)
                logging.debug("%s queries available from '%s'", len(queries), r_path)
                if i == num_of_paths - 1:
                    sample_amount = total_amount - len(obligatory_queries) - i * sample_amount
                    logging.info("Get %s random queries from the last path", sample_amount)
                chosen = random.sample(queries, sample_amount)
                out_f.write("\n".join(chosen) + "\n")
            for q in obligatory_queries:
                out_f.write(q + "\n")
        dolbilka.convert_queries_to_plan(out_res_queries_path, out_res_plan_path)

    def _get_obligatory_queries(self):
        queries_id = apihelpers.get_last_resource_with_attribute(
            resource_types.PLAIN_TEXT_QUERIES,
            attribute_name="custom_test_queries"
        ).id
        if queries_id:
            return fu.read_lines(self.sync_resource(queries_id))
        return []

    def _get_queries_res_paths(self):
        result = []
        for reg in REGS:
            if not self.ctx.get(REG_TEMPL.format(reg)):
                logging.info("Skip reg = '%s'", reg)
                continue
            logging.info("Try to get resources for reg '%s", reg)
            queries_test = apihelpers.get_last_resource_with_attribute(
                resource_types.PLAIN_TEXT_QUERIES,
                attribute_name="TE_web_meta_wizarded_queries_test_{}".format(reg)
            )
            queries_validate = apihelpers.get_last_resource_with_attribute(
                resource_types.PLAIN_TEXT_QUERIES,
                attribute_name="TE_web_meta_wizarded_queries_validate_{}".format(reg)
            )
            queries_mobile = apihelpers.get_last_resource_with_attribute(
                resource_types.PLAIN_TEXT_QUERIES,
                attribute_name="TE_web_meta_wizarded_queries_mobile_{}".format(reg)
            )
            logging.info(
                "Resources for reg '%s': %s, %s, %s",
                reg, queries_test, queries_validate, queries_mobile
            )
            result.extend([
                self.sync_resource(queries_test.id),
                self.sync_resource(queries_validate.id),
            ])
            if queries_mobile:
                result.append(self.sync_resource(queries_mobile.id))
        logging.info("Total resources amount = %s", len(result))
        return result


__Task__ = CombineMiddlesearchQueries
