# -*- coding: utf-8 -*-

import json
from operator import itemgetter

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.projects.common.middlesearch import evlogcachehit
from sandbox.projects.common.differ import coloring
from sandbox.projects.common import error_handlers as eh


_COMPARE_TYPE_PERCENT = 'percent'
"""
    Сравнение отношения subsource_request к общему количеству запросов
"""

_COMPARE_TYPE_VALUE = 'value'
"""
    Сравнение абсолютных значений subsource_request
"""


class CachehitStats1Parameter(ResourceSelector):
    name = 'cachehit_stats_resource_1'
    description = 'First stats'
    resource_type = 'MIDDLESEARCH_CACHEHIT_STATS'
    required = True


class CachehitStats2Parameter(ResourceSelector):
    name = 'cachehit_stats_resource_2'
    description = 'Second stats'
    resource_type = 'MIDDLESEARCH_CACHEHIT_STATS'
    required = True


class CompareTypeParameter(SandboxStringParameter):
    name = 'compare_type'
    description = 'Compare as'
    choices = [('Absolute values', _COMPARE_TYPE_VALUE), ('Percents', _COMPARE_TYPE_PERCENT)]
    default_value = _COMPARE_TYPE_VALUE


class CompareMiddlesearchCachehit(SandboxTask):
    """
        Сравнивает 2 замера кешхита метапоиска
        http://wiki.yandex-team.ru/JandeksPoisk/KachestvoPoiska/Middlesearch/Cachehit
    """

    type = 'COMPARE_MIDDLESEARCH_CACHEHIT'

    cores = 1

    input_parameters = (
        CachehitStats1Parameter,
        CachehitStats2Parameter,
        CompareTypeParameter,
    )

    @property
    def footer(self):
        stats = self.ctx.get("stats")
        if not stats:
            return None
        head = [{"key": "space", "title": ""}]
        body = {"space": ["<b>Cache miss</b>", "<b>Sub_source requests</b>"]}
        for key, val in sorted(stats['all']['sources_stats'].iteritems(), key=itemgetter(0)):
            head.append({"key": key, "title": key.replace("_", " ")})
            body[key] = [
                self._show_foot_info(val["cache_miss_count"], val["cache_miss_count_diff"], self.ctx['max_diff']),
                self._show_foot_info(val["sub_source_requests"], val["sub_source_requests_diff"], self.ctx['max_diff']),
            ]
        return [
            {
                'helperName': '',
                'content': {
                    "<h4>{}</br><b>Requests: {}</b></h4>".format(self.descr, stats['all']['incoming_requests']): {
                        "header": head,
                        "body": body
                    }
                }
            }
        ]

    @staticmethod
    def _show_foot_info(count, diff, max_diff):
        return "{}</br>{}".format(count, coloring.color_diff(diff, max_diff))

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self._create_resource(self.descr, 'stats.txt', resource_types.MIDDLESEARCH_CACHEHIT_STATS)
        self.ctx[evlogcachehit.STATS_RESOURCE_ID] = resource.id

    def on_execute(self):
        self.ctx['max_diff'] = 1.0  # FIXME: it should be almost 0, see SEARCH-1434

        stats_file1 = self.sync_resource(self.ctx[CachehitStats1Parameter.name])
        stats_file2 = self.sync_resource(self.ctx[CachehitStats2Parameter.name])

        stats1 = json.loads(open(stats_file1).read())
        stats2 = json.loads(open(stats_file2).read())

        diff_stats = self._compare_stats(stats1, stats2)

        if any('SAMOHOD' in source for source in diff_stats['all']['sources_stats'].keys()):
            self.ctx['max_diff'] = 10.0

        evlogcachehit.store_stats(self, diff_stats)
        self.ctx['has_diff'], sources_stats = self.get_diff(source='all', diff_stats=diff_stats)
        self.ctx['sub_source_requests_diff'] = sources_stats['sub_source_requests_diff']
        self.ctx['cache_miss_count_diff'] = sources_stats['cache_miss_count_diff']

    def _compare_stats(self, stats1, stats2):
        diff_stats = {}
        compare_type = self.ctx.get(CompareTypeParameter.name, CompareTypeParameter.default_value)

        for region, region_stats1 in stats1.iteritems():
            region_stats2 = stats2[region]

            incoming_requests1 = region_stats1['incoming_requests']
            incoming_requests2 = region_stats2['incoming_requests']

            # ignore small diff because events in eventlog are not strongly ordered by source requests
            if compare_type == _COMPARE_TYPE_VALUE and abs(incoming_requests1 - incoming_requests2) > 10:
                eh.check_failed("diff in incoming_requests for region {}".format(region))

            diff_stats[region] = {'incoming_requests': region_stats1['incoming_requests'], 'sources_stats': {}}
            diff_source_stats = diff_stats[region]['sources_stats']

            for client_name, source_stats1 in region_stats1['sources_stats'].iteritems():
                source_stats2 = region_stats2['sources_stats'][client_name]

                diff_client_stats = diff_source_stats[client_name] = {}

                cache_miss1 = self._get_num(source_stats1['cache_miss_count'], incoming_requests1)
                cache_miss2 = self._get_num(source_stats2['cache_miss_count'], incoming_requests2)
                diff_client_stats['cache_miss_count'] = self.get_value_str(cache_miss2)
                diff_client_stats['cache_miss_count_diff'] = get_num_diff_percent(cache_miss1, cache_miss2)
                sub_source_requests1 = self._get_num(source_stats1['sub_source_requests'], incoming_requests1)
                sub_source_requests2 = self._get_num(source_stats2['sub_source_requests'], incoming_requests2)
                diff_client_stats['sub_source_requests'] = self.get_value_str(sub_source_requests2)
                diff_client_stats['sub_source_requests_diff'] = get_num_diff_percent(sub_source_requests1,
                                                                                     sub_source_requests2)

        return diff_stats

    def get_value_str(self, value):
        if self.ctx.get(CompareTypeParameter.name, CompareTypeParameter.default_value) == _COMPARE_TYPE_VALUE:
            return str(value)
        else:
            return '%s%%' % value

    def _get_num(self, sub_value, all_value):
        compare_type = self.ctx.get(CompareTypeParameter.name, CompareTypeParameter.default_value)
        if compare_type == _COMPARE_TYPE_VALUE:
            return sub_value
        return float(sub_value) / all_value * 100 if all_value else 0

    def get_short_task_result(self):
        if self.is_completed():
            if self.ctx['has_diff']:
                return "diff"
            else:
                return "no diff"
        return None

    def get_stats(self):
        """
            called from task view
        """

        return evlogcachehit.load_stats(self)

    def get_diff(self, source='all', diff_stats=None):
        """
            Returns diff results for specified source
        """

        if diff_stats is None:
            diff_stats = self.get_stats()

        if diff_stats is None:
            return False, {}

        sources_stats = diff_stats['all']['sources_stats'][source]
        sub_source_diff = sources_stats['sub_source_requests_diff']
        cache_miss_diff = sources_stats['cache_miss_count_diff']

        has_diff = max(abs(sub_source_diff), abs(cache_miss_diff)) > self.ctx['max_diff']
        return has_diff, sources_stats


def get_num_diff_percent(num1, num2):
    if num1 == 0 and num2 == 0:
        return 0
    return 100.0 * (num2 - num1) / max(num1, num2)


__Task__ = CompareMiddlesearchCachehit
