import logging

from sandbox import sdk2
import sandbox.common.types.task as ctt
import sandbox.common.errors as ce

from sandbox.projects import resource_types
from sandbox.projects.websearch.params import ResourceWithDefaultByAttr
import sandbox.projects.common.link_builder as lb
import sandbox.projects.common.binary_task.deprecated as bin_task
import sandbox.projects.common.search.bugbanner2 as bb2
import sandbox.projects.websearch.basesearch.params as bs_params
from sandbox.projects.websearch.responses.CompareSearchResponses import CompareSearchResponses
import sandbox.projects.websearch.middlesearch.base_tasks.params as ms_params
import sandbox.projects.websearch.middlesearch.resources as ms_resources
import sandbox.projects.websearch.middlesearch.GetWebMiddlesearchResponsesSingleHost as get_ms_responses


class CompareWebMiddlesearchResponsesSingleHost(bin_task.LastBinaryTaskRelease, bb2.BugBannerTask):
    """
       Task for comparing responses from two single host web middlesearch binaries
    """

    class Requirements(sdk2.Requirements):
        ram = 512
        disk_space = 512
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        requests = ResourceWithDefaultByAttr(
            "Requests",
            resource_type=[
                ms_resources.WebMiddlesearchPlainTextQueries,
                ms_resources.WebMiddlesearchApphostRequests,
            ],
            default_attr_name="TE_web_production_mmeta_reqs",
        )
        with sdk2.parameters.Group("Compare responses parameters") as web_compare_params_group:
            compare_complete_only = sdk2.parameters.Bool("Compare complete only", default=True)
            compare_fully_answered_only = sdk2.parameters.Bool("Compare fully answered only", default=True)
            ignore_diff_in_compressed_all_factors = sdk2.parameters.Bool("Ignore diff in _CompressedAllFactors field", default=False)
            ignore_diff_in_doc_ranking_factors = sdk2.parameters.Bool("Ignore diff in DocRankingFactors field", default=False)
            relev_factors_for_soft_check = sdk2.parameters.String(
                "','-separated factors indexes. Use soft difference check for given factors in _RelevFactors field", default="")
            cpu_model_tag = sdk2.parameters.CustomClientTags(
                'CPU model client tag',
                default=sdk2.parameters.ctc.Tag.INTEL_E5_2650
            )
            
        web_ms_params_group = sdk2.parameters.Group("Web middlesearch parameters")
        middlesearch = ms_params.DoubleMiddlesearchParameters
        web_bs_params_group = sdk2.parameters.Group("Web basesearch parameters")
        basesearch = bs_params.DoubleBasesearchParameters
        tasks_archive_resource = bin_task.binary_release_parameters(none=True)

    class Context(sdk2.Task.Context):
        get_responses_task1 = None
        get_responses_task2 = None
        compare_responses_task = None

    def on_execute(self):
        with self.memoize_stage.get_responses:
            self._get_responses()
        with self.memoize_stage.compare_responses:
            self._compare_responses()
        self._check_compare_status()

    def _get_responses(self):
        logging.info("Start getting responses")
        common_args = {
            "requests": self.Parameters.requests,
            "owner": self.Parameters.owner,
            "queries_limit": 10000,
            "max_retries": 2,
            "n_workers": 6,
            "get_all_factors": True,
            "use_dcfm": True,
            "max_snippets_per_request": 500,
            "sending_timeout": "5000ms",
            "n_groups_for_source_mult": 5,
            "cpu_model_tag": self.Parameters.cpu_model_tag
        }
        logging.info("Common get responses args: %s", common_args)
        task1_args = {
            "description": "Get responses for {} #1".format(self.id),
            # middlesearch
            "executable": self.Parameters.middlesearch.ms_executable1,
            "config": self.Parameters.middlesearch.ms_config1,
            "models_archive": self.Parameters.middlesearch.ms_models_archive1,
            "rearr": self.Parameters.middlesearch.ms_rearr1,
            # basesearch (first)
            "executable1": self.Parameters.basesearch.executable1,
            "config1": self.Parameters.basesearch.config1,
            "models_archive1": self.Parameters.basesearch.models_archive1,
            "database1": self.Parameters.basesearch.database1,
            "executable2": self.Parameters.basesearch.executable1,
            "config2": self.Parameters.basesearch.config1,
            "models_archive2": self.Parameters.basesearch.models_archive1,
            "database2": self.Parameters.basesearch.database1,
        }
        task1_args.update(common_args)
        logging.info("Get task 1 args: %s", task1_args)
        task1 = get_ms_responses.GetWebMiddlesearchResponsesSingleHost(self, **task1_args).enqueue().id
        task2_args = {
            "description": "Get responses for {} #2".format(self.id),
            # middlesearch
            "executable": self.Parameters.middlesearch.ms_executable2,
            "config": self.Parameters.middlesearch.ms_config2,
            "models_archive": self.Parameters.middlesearch.ms_models_archive2,
            "rearr": self.Parameters.middlesearch.ms_rearr2,
            # basesearch (first)
            "executable1": self.Parameters.basesearch.executable2,
            "config1": self.Parameters.basesearch.config2,
            "models_archive1": self.Parameters.basesearch.models_archive2,
            "database1": self.Parameters.basesearch.database2,
            "executable2": self.Parameters.basesearch.executable2,
            "config2": self.Parameters.basesearch.config2,
            "models_archive2": self.Parameters.basesearch.models_archive2,
            "database2": self.Parameters.basesearch.database2,
        }
        task2_args.update(common_args)
        logging.info("Get task 2 args: %s", task2_args)
        task2 = get_ms_responses.GetWebMiddlesearchResponsesSingleHost(self, **task2_args).enqueue().id
        self.Context.get_responses_task1 = task1
        self.Context.get_responses_task2 = task2
        raise sdk2.WaitTask([task1, task2], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def _compare_responses(self):
        logging.info("Start comparing responses")
        compare_responses_subtask = CompareSearchResponses(
            self,
            description="Compare responses for {}".format(self.id),
            owner=self.Parameters.owner,
            notifications=[],
            responses1=self._get_response_res_id(self.Context.get_responses_task1),
            responses2=self._get_response_res_id(self.Context.get_responses_task2),
            compare_complete_responses_only=self.Parameters.compare_complete_only,
            ignore_unanswered=self.Parameters.compare_fully_answered_only,
            ignore_diff_in_compressed_all_factors=self.Parameters.ignore_diff_in_compressed_all_factors,
            ignore_diff_in_doc_ranking_factors=self.Parameters.ignore_diff_in_doc_ranking_factors,
            relev_factors_for_soft_check=[int(i) for i in self.Parameters.relev_factors_for_soft_check.split(",")],
        )
        compare_responses_subtask.save()
        compare_responses_subtask.enqueue()
        self.Context.compare_responses_task = compare_responses_subtask.id
        self.server.batch.tasks.start.update([self.Context.compare_responses_task])
        raise sdk2.WaitTask([self.Context.compare_responses_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def _check_compare_status(self):
        cmp_task = sdk2.Task[self.Context.compare_responses_task]
        if cmp_task.Parameters.diff_count > 0:
            diff_resource_link = lb.resource_link(cmp_task.Parameters.diff_resource)
            self.set_info("Diff: {}".format(diff_resource_link), do_escape=False)
            raise ce.TaskFailure("There is diff it task: {}".format(self.Context.compare_responses_task))

    @staticmethod
    def _get_response_res_id(task_id):
        return sdk2.Resource[resource_types.SEARCH_PROTO_RESPONSES].find(task_id=task_id).first().id
