# -*- coding: utf-8 -*-

import logging
import random

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.channel import channel

import sandbox.projects.websearch.middlesearch.resources as ms_resources
from sandbox.projects import resource_types
from sandbox.projects.common.base_search_quality import response_saver
from sandbox.projects.common.base_search_quality import threadPool
from sandbox.projects.common.base_search_quality import settings as bss
from sandbox.projects.common.base_search_quality import basesearch_response_parser as brp
from sandbox.projects.common.middlesearch import single_host


_RESPONSE_SAVER_PARAMS = response_saver.create_response_saver_params(
    queries_resource=[
        resource_types.PLAIN_TEXT_QUERIES,
        ms_resources.WebMiddlesearchPlainTextQueries,
        resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
        resource_types.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
    ]
)


class MaxFuzzyCase(sp.SandboxIntegerParameter):
    name = 'max_fuzzy_case'
    description = 'Max fuzzy case'
    group = "Fuzzy parameters"
    default_value = 2


class ModifyRandomBits(sp.SandboxBoolParameter):
    name = 'modify_random_bits'
    description = 'Modify random bits'
    group = "Fuzzy parameters"
    default_value = True


class ProcessCount(threadPool.ProcessCount):
    # SEARCH-1716
    default_value = 4


class IgnoreGotError(_RESPONSE_SAVER_PARAMS.IgnoreGotErrorParameter):
    default_value = True


_MODIFY_BITS = 2


class FuzzyTestMiddlesearch(single_host.MiddlesearchSingleHostTask):
    """
        Patches queries with fuzzy parameters (SEARCH-1159).
        Starts 1 middlesearch and 2 basesearch instances.
    """

    type = 'FUZZY_TEST_MIDDLESEARCH'
    execution_space = (
        bss.MAX_MIDDLE_CACHE_SIZE +
        bss.RESERVED_SPACE
    )
    input_parameters = (
        (
            MaxFuzzyCase,
            ModifyRandomBits,
        ) +
        single_host.PARAMS + (
            _RESPONSE_SAVER_PARAMS.QueriesParameter,
            _RESPONSE_SAVER_PARAMS.QueriesLimitParameter,
            _RESPONSE_SAVER_PARAMS.SavePatchedQueriesParameter,
            IgnoreGotError,
            ProcessCount,
        )
    )

    def _create_dummy_response_file(self, file_name):
        brp.write_responses(file_name, responses=list())

    def on_enqueue(self):
        single_host.MiddlesearchSingleHostTask.on_enqueue(self)
        response_saver.create_resources(self)

    def _use_middlesearch_component(self, middlesearch):
        try:
            patched_queries = response_saver.get_patched_queries_resource(self)
            res = channel.sandbox.get_resource(self.ctx['out_resource_id'])

            fuzzy_cases = range(1, self.ctx[MaxFuzzyCase.name] + 1)
            if not self.ctx[ModifyRandomBits.name]:
                fuzzy_cases.remove(_MODIFY_BITS)

            logging.info("Used fuzzy cases: %s", fuzzy_cases)

            def fuzzy_patcher(q):
                return '{}&ms=proto&pron=fuzzy{}:{}'.format(
                    q,
                    fuzzy_cases[random.randint(0, len(fuzzy_cases) - 1)],
                    random.uniform(0.05, 0.5),
                )

            response_saver.save_responses(
                self.ctx,
                search_component=middlesearch,
                responses_resource=res,
                queries_patchers=[fuzzy_patcher],
                patched_queries_resource=patched_queries,
            )
        except Exception as e: # Mute TaskFailure. Motivation see: MIDDE-625 and MIDDLE-621
            self._create_dummy_response_file(res.path)
            logging.info("TaskFailure explanation")
            logging.info(e)



__Task__ = FuzzyTestMiddlesearch
