# coding: utf-8

import logging
import os

from sandbox import sdk2
import sandbox.sandboxsdk.environments as env

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import time_utils as tu
from sandbox.projects.common import string
from sandbox.projects.resource_types import MIDDLESEARCH_CACHEHIT_PLANS


class GetMiddlesearchCachehitRequests(sdk2.Task):
    """
        Task for getting requests for cachehit measuring
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 8 * 1024
        disk_space = 30 * 1024  # 30 Gb
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        number_of_plans = sdk2.parameters.Integer("Number of plans", default=3)
        number_of_requests_in_each_plan = sdk2.parameters.Integer("Number of requests in each plan", default=1000000)
        resource_attrs = sdk2.parameters.String("Add attrs to resource")

    def on_enqueue(self):
        sdk2.Task.on_enqueue(self)
        today = tu.date_ymd(sep="_")
        res = MIDDLESEARCH_CACHEHIT_PLANS(self, "Cachehit plans, ".format(today), "cachehit_plans_{}".format(today))
        self.Context.out_resource_id = res.id

    def on_execute(self):
        out_res = sdk2.Resource[self.Context.out_resource_id]
        for k, v in string.parse_attrs(self.Parameters.resource_attrs).iteritems():
            setattr(out_res, k, v)
        with env.VirtualEnvironment() as venv:
            logging.info('Installing yql')
            env.PipEnvironment('yql', version="1.0.19", venv=venv, use_wheel=True).prepare()
            env.PipEnvironment('yandex-yt', version="0.7.34.post0", venv=venv, use_wheel=True).prepare()
            env.PipEnvironment(
                'yandex-yt-yson-bindings-skynet', version="0.2.25.post1", venv=venv, use_wheel=True
            ).prepare()

            query_limits = [self.Parameters.number_of_requests_in_each_plan] * self.Parameters.number_of_plans
            if sum(query_limits) == 0:
                eh.check_failed("Zero requests are chosen in input parameters!")
            script_path = (
                os.path.dirname(os.path.dirname(__file__)) +
                "/GetMiddlesearchProdQueries/select_middlesearch_queries.py"
            )
            selected_day = tu.date_ymd(sep="")
            cmd = [
                venv.executable, script_path,
                "--day", selected_day,
                "--mtype", "mmeta",
                "--token", sdk2.Vault.data(self.owner, "yt_token_for_eventlogs"),
                "--apphostpath", str(sdk2.ResourceData(out_res).path)
            ]
            with sdk2.helpers.ProcessLog(self, logger="select_mmeta_queries_for_cachehit") as pl:
                sdk2.helpers.subprocess.Popen(" ".join(cmd), shell=True, stdout=pl.stdout, stderr=pl.stderr).wait()
