from __future__ import print_function, unicode_literals

import logging

from sandbox import sdk2
from sandbox.common import enum
from sandbox.common.types.task import Status
from sandbox.common.errors import TaskFailure
from sandbox.projects.resource_types import MIDDLESEARCH_DATA
from sandbox.projects.common import binary_task
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common import task_env
from sandbox.projects.common import time_utils
from sandbox.projects.common.search import bugbanner2 as bb2
from sandbox.projects.common.search.settings import WebSettings


CLUSTERSTATE_URL = 'http://ctrl.clusterstate.yandex-team.ru'
TRACKER_URL = 'http://vla-tracker-cajuper.n.yandex-team.ru'


class SetAttributesForTestEnv(enum.Enum):
    enum.Enum.preserve_order()

    NONE = None
    WEB_MIDDLE = None


def get_prod_timestamp(clusterstate_url):
    oldprod_url = clusterstate_url + '/web/prod/jupiter/oldprod'
    prod_state = requests_wrapper.get(oldprod_url).json()
    logging.info('Get oldprod state %s', prod_state['common'])
    return prod_state['common']['yt_state']


class GetMiddlesearchDatabase2(binary_task.LastBinaryTaskRelease, bb2.BugBannerTask):
    class Requirements(task_env.TinyRequirements):
        cores = 2
        disk_space = 80 * 1024  # 80 Gb
        ram = 16 * 1024

    class Parameters(sdk2.Task.Parameters):
        _task_binary = binary_task.binary_release_parameters(stable=True)

        tracker_url = sdk2.parameters.String('Tracker url', default=TRACKER_URL)
        clusterstate_url = sdk2.parameters.String('clusterstate url', default=CLUSTERSTATE_URL)
        tier = sdk2.parameters.String('tier', default='MsuseardataJupiterTier0')
        shard_name_format = sdk2.parameters.String('shard name format', default='rearr-jupiter-msuserdata-000-{}')
        fetch_db_timestamp = sdk2.parameters.Bool(
            "Fetch db_timestamp from clusterstate",
            default=True,
        )
        with fetch_db_timestamp.value[False]:
            db_timestamp = sdk2.parameters.String('db timestamp', default='')

        fetch_rbtorrent = sdk2.parameters.Bool(
            "Fetch rbtorrent of database from tracker (tracker_url)",
            default=True,
        )
        with fetch_rbtorrent.value[False]:
            rbtorrent = sdk2.parameters.String('rbtorrent', default='')

        with sdk2.parameters.String("Set attributes for testenv") as set_attributes_for_testenv:
            for i, opt in enumerate(SetAttributesForTestEnv):
                set_attributes_for_testenv.values[opt] = set_attributes_for_testenv.Value(
                    value=opt,
                    default=(i == 0),
                )

        with sdk2.parameters.Output:
            fetched_rbtorrent = sdk2.parameters.String('Fetchd rb torrent')
            middlesearch_database_resource = sdk2.parameters.Resource("Middlesearch database resource")

    def on_execute(self):
        import infra.callisto.deploy.tracker.client as callisto_client

        if self.Parameters.fetch_db_timestamp:
            logging.info('fetching db_timestamp from clusterstate')
            db_timestamp = get_prod_timestamp(str(self.Parameters.clusterstate_url))
        else:
            db_timestamp = self.Parameters.db_timestamp
        logging.info('db_timestamp = %s', db_timestamp)
        shard_name = str(self.Parameters.shard_name_format).format(db_timestamp)
        logging.info('shard_name = %s', shard_name)
        namespace = '/web/prod/{}/{}/'.format(db_timestamp, self.Parameters.tier)
        logging.info('namespace = %s', namespace)
        if self.Parameters.fetch_rbtorrent:
            client = callisto_client.Client(self.Parameters.tracker_url)
            rbtorrent = client.resolve_one(namespace, shard_name).rbtorrent
        else:
            rbtorrent = str(self.Parameters.rbtorrent)
        logging.info('rbtorrent = %s', rbtorrent)
        self.Parameters.fetched_rbtorrent = rbtorrent

        with self.memoize_stage.upload_source(commit_on_entrance=False):
            self.set_info('schedule REMOTE_COPY_RESOURCE task')
            self.Context.remote_copy_task_id = self._upload_database(rbtorrent=rbtorrent, shard_name=shard_name)
            self.set_info('Wait uploading middlesearch data')
            raise sdk2.WaitTask(self.Context.remote_copy_task_id, Status.Group.FINISH | Status.Group.BREAK)
        self._ensure_child_task_is_successful(self.Context.remote_copy_task_id)
        resource_id = sdk2.Task[self.Context.remote_copy_task_id].Context.result_resource_id
        self.set_info('result resource_id: {}'.format(resource_id))
        self.Parameters.middlesearch_database_resource = resource_id

    def _ensure_child_task_is_successful(self, task_id):
        statuses = [t.status for t in self.find(id=task_id)]
        if not all(s == Status.SUCCESS for s in statuses):
            raise TaskFailure('Child task {} is failed, status: {}'.format(task_id, ', '.join(statuses)))

    def _upload_database(self, rbtorrent, shard_name):
        resource_attrs = {
            'shard_name': shard_name,
        }
        if self.Parameters.set_attributes_for_testenv == SetAttributesForTestEnv.WEB_MIDDLE:
            resource_attrs.update({
                WebSettings.testenv_middle_resources_attr_name('mmeta', 'data'):
                    time_utils.date_ymd(sep="_"),  # SEARCH-4078
            })
        parameters = {
            'remote_file_name': rbtorrent,
            'remote_file_protocol': 'skynet',
            'resource_type': MIDDLESEARCH_DATA.name,
            'resource_attrs': ','.join('{}={}'.format(k, v) for k, v in resource_attrs.items()),
        }
        task = sdk2.Task['REMOTE_COPY_RESOURCE'](
            self,
            **parameters
        )
        return task.enqueue().id
