# -*- coding: utf-8 -*-

import logging

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp

import sandbox.projects.websearch.middlesearch.resources as ms_resources
from sandbox.projects.common.search.eventlog import eventlog
from sandbox.projects.common.middlesearch import single_host
from sandbox.projects.common.middlesearch.response_patcher import response_patcher
from sandbox.projects.common.base_search_quality import threadPool
from sandbox.projects.common.base_search_quality import response_saver
from sandbox.projects.common.search.components.cproxy import get_cproxy
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import metadebug
from sandbox.projects.common import utils
from sandbox.projects import resource_types


_RESPONSE_SAVER_PARAMS = response_saver.create_response_saver_params(
    queries_resource=[
        resource_types.PLAIN_TEXT_QUERIES,
        ms_resources.WebMiddlesearchPlainTextQueries,
        resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
        resource_types.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
    ]
)


class UseAsyncSearch(sp.SandboxBoolParameter):
    name = 'use_async_search'
    description = 'Use AsyncSearch'
    default_value = True


class FirstBasesearchType(sp.SandboxStringParameter):
    name = 'first_basesearch_type'
    description = '1st basesearch type'
    default_value = 'WEB'


class SecondBasesearchType(sp.SandboxStringParameter):
    name = 'second_basesearch_type'
    description = '2nd basesearch type'
    default_value = 'PLATINUM'


class GetMiddlesearchResponses(single_host.MiddlesearchSingleHostTask):
    """
        Starts 1 middlesearch and 2 basesearch instances.
        Save responses to ``PLAIN_TEXT_QUERIES`` in resources:
        ``BASESEARCH_HR_RESPONSES`` (human-readable output)
        or ``SEARCH_PROTO_RESPONSES`` (protobuf binary output).
    """

    type = 'GET_MIDDLESEARCH_RESPONSES'
    client_tags = ctc.Tag.LINUX_PRECISE & ctc.Tag.INTEL_E5_2650
    required_ram = 96 << 10

    input_parameters = (
        single_host.PARAMS +
        _RESPONSE_SAVER_PARAMS.params +
        metadebug.PARAMS.params +
        (
            UseAsyncSearch,
            FirstBasesearchType,
            SecondBasesearchType,
        ) +
        threadPool.PARAMS
    )

    def on_failure(self):
        eventlog.locate_instability_problems(self)
        single_host.MiddlesearchSingleHostTask.on_failure(self)

    def on_enqueue(self):
        single_host.MiddlesearchSingleHostTask.on_enqueue(self)

        channel.task = self
        response_saver.create_resources(self)
        self.ctx['eventlog_out_resource_id'] = self.create_resource(
            'eventlog', 'event.log', resource_types.EVENTLOG_DUMP, arch='any'
        ).id

    def _create_middlesearch_component(self):
        base_port1, base_port2 = self._get_basesearch_ports_for_middlesearch()
        params = self._get_middlesearch_additional_params()

        middlesearch = sc.get_middlesearch(
            basesearches=[
                {
                    'basesearch_type': utils.get_or_default(self.ctx, FirstBasesearchType),
                    'hosts_and_ports': [
                        # Use same instances, see SEARCH-1022 for details (production-alike)
                        ('localhost', base_port1),
                        ('localhost', base_port1),
                        ('localhost', base_port1),
                    ],
                },
                {
                    'basesearch_type': utils.get_or_default(self.ctx, SecondBasesearchType),
                    'hosts_and_ports': [
                        # Use same instances, see SEARCH-1022 for details (production-alike)
                        ('localhost', base_port2),
                        ('localhost', base_port2),
                        ('localhost', base_port2),
                    ],
                }
            ],
            event_log=channel.sandbox.get_resource(self.ctx['eventlog_out_resource_id']).path,
            max_allfactors_per_bs=50,
            disable_timeouts=True,
            **params
        )
        middlesearch.set_cache_thread_limit()  # For SEARCH-1474

        self.init_search_component(middlesearch)

        return middlesearch

    def _get_basesearch_ports_for_middlesearch(self):
        if not metadebug.use_cproxy(self.ctx):
            return single_host.BASE1_PORT, single_host.BASE2_PORT

        proxy_port1 = single_host.BASE1_PORT + 10
        proxy_port2 = single_host.BASE2_PORT + 10

        proxy = {
            proxy_port1: "localhost:{}".format(single_host.BASE1_PORT),
            proxy_port2: "localhost:{}".format(single_host.BASE2_PORT)
        }
        cproxy = get_cproxy(self, proxy)
        cproxy.start()
        cproxy.wait_port()

        return proxy_port1, proxy_port2

    def _get_middlesearch_additional_params(self):
        return {
            'use_async_search': utils.get_or_default(self.ctx, UseAsyncSearch),
            'set_n_groups_for_source_multiplier': True,
        }

    def _use_middlesearch_component(self, middlesearch):
        patched_queries_resource = response_saver.get_patched_queries_resource(self)

        def int_patcher(q):
            # int requests should have haha=da
            # kulikov@: all **search** queries to ints should contain haha=da (means search without snippets)
            return q if '&haha=da' in q else q + '&haha=da'

        memorized = {}
        # see search/common/cphash.cpp
        stabilize_relevs = set([
            "bgrtfactors",
            "dopp_url_counters_for_base",
            "dopp_url_counters_for_base_os",
            "fresh_detector_predict",
            "fresh_news_detector_predict"
        ])
        def bgrtfactors_canonizer(q):
            # relev=bgrtfactors=... is deliberately not in mmeta cache key but it does influence factors and L2 ranking
            # so mmeta response depends on whether mmeta takes a previous response from the cache (with old bgrtfactors)
            # or decides to reissue a request (with new bgrtfactors). Enforce same bgrtfactors for all requests with the same text.
            # Same goes for relev=dopp_url_counters_for_base[_os].
            # relev=fresh_[news_]detector_predict is also not in mmeta cache key;
            # diffs from these have not yet been observed, but add them as well just in case.
            # Also, rapid_clicks_l2 are calculated based on timestamp from reqid, so replace timestamp from reqids as well.
            relevpos = q.find('&relev=')
            textpos = q.find('&text=')
            if relevpos == -1 or textpos == -1:
                return q
            relevend = q.find('&', relevpos + 1)
            if relevend == -1:
                relevend = len(q)
            relevs = q[relevpos+len('&relev='):relevend].split(';')
            relevs_normal = []
            relevs_to_stabilize = []
            for r in relevs:
                pos1 = r.find('=')
                if pos1 == -1:
                    pos1 = len(r)
                pos2 = r.find('%3D')
                if pos2 == -1:
                    pos2 = len(r)
                rname = r[:min(pos1,pos2)]
                if rname in stabilize_relevs:
                    relevs_to_stabilize.append(r)
                else:
                    relevs_normal.append(r)
            relevs_to_stabilize = ';'.join(relevs_to_stabilize)
            textend = q.find('&', textpos + 1)
            if textend == -1:
                textend = len(q)
            text = q[textpos:textend]
            reqidpos = q.find('&reqid=')
            reqid_timestamp = None
            if reqidpos != -1:
                reqidend = q.find('-', reqidpos)
                if reqidend != -1:
                    try:
                        reqid_timestamp = int(q[reqidpos+len('&reqid='):reqidend])
                    except ValueError:
                        pass
            memorized_data = memorized.get(text)
            if memorized_data is None:
                memorized[text] = (relevs_to_stabilize, reqid_timestamp)
                return q
            old_relevs_to_stabilize, old_reqid_timestamp = memorized_data
            if old_relevs_to_stabilize != relevs_to_stabilize or old_reqid_timestamp != reqid_timestamp:
                logging.info('replacing "{}&reqid={}" to "{}&reqid={}" for "{}"'.format(relevs_to_stabilize, reqid_timestamp, old_relevs_to_stabilize, old_reqid_timestamp, text))
            qret = q
            if old_relevs_to_stabilize != relevs_to_stabilize:
                qret = q[:relevpos] + '&relev=' + ';'.join(relevs_normal + [old_relevs_to_stabilize]) + q[relevend:]
            if old_reqid_timestamp != reqid_timestamp:
                reqidpos = qret.find('&reqid=')
                if reqidpos == -1:
                    if old_reqid_timestamp is not None:
                        qret += '&reqid={}-synthetized'.format(old_reqid_timestamp)
                elif reqid_timestamp is None:
                    if old_reqid_timestamp is not None:
                        reqidend = qret.find('&', reqidpos + 1)
                        if reqidend == -1:
                            reqidend = len(qret)
                        qret = qret[:reqidpos] + '&reqid={}-synthetized'.format(old_reqid_timestamp) + qret[reqidend:]
                elif old_reqid_timestamp is None:
                    reqidend = qret.find('&', reqidpos + 1)
                    if reqidend == -1:
                        reqidend = len(qret)
                    qret = qret[:reqidpos] + qret[reqidend:]
                else:
                    reqidend = qret.find('-', reqidpos)
                    qret = qret[:reqidpos] + '&reqid={}'.format(old_reqid_timestamp) + qret[reqidend:]
            return qret

        queries_patchers = [
            metadebug.dump_grouping_patcher,
            metadebug.remove_eventlog_frame_dump_patcher,
            metadebug.lua_rearrange_time_limit_patcher,
            bgrtfactors_canonizer,
        ]
        if utils.get_or_default(self.ctx, sc.DefaultMiddlesearchParams.UseInt):
            queries_patchers.append(int_patcher)

        if metadebug.dump_subsource_answer(self.ctx):
            queries_patchers.append(metadebug.dump_subsource_answer_patcher)

        self.tcp_dump_proc = metadebug.start_tcpdump(
            self.ctx,
            ports=[
                single_host.BASE1_PORT,
                single_host.BASE2_PORT,
            ],
        )

        def save_responses_to_resource(res, prepare_session_func=None, test_with_doubled_queries=False):
            response_saver.save_responses(
                self.ctx,
                search_component=middlesearch,
                responses_resource=res,
                queries_patchers=queries_patchers,
                patched_queries_resource=patched_queries_resource,
                response_patchers=[response_patcher],
                prepare_session_callback=prepare_session_func,
                need_dbgrlv=False,
                test_with_doubled_queries=test_with_doubled_queries,
            )

        try:
            #  if need to test responses correctness with cache disabled:
            if self.ctx.get(_RESPONSE_SAVER_PARAMS.TestWithCacheDisabled.name):
                logging.info("Getting responses with disabled cache")
                save_responses_to_resource(
                    channel.sandbox.get_resource(self.ctx['without_cache_responses_id']),
                    prepare_session_func=middlesearch.disable_cache,
                )
            logging.info("Getting responses with empty enabled cache")
            save_responses_to_resource(
                channel.sandbox.get_resource(self.ctx['out_resource_id']),
                prepare_session_func=middlesearch.clear_cache,
                test_with_doubled_queries=utils.get_or_default(self.ctx, _RESPONSE_SAVER_PARAMS.TestWithDoubledQueries),
            )
        except Exception:
            raise
        finally:
            # any problem with responses can raise exception
            # but tcpdump should be stopped before we fail the task
            metadebug.stop_tcpdump(self.tcp_dump_proc)

    def get_short_task_result(self):
        if self.is_completed():
            return "unstable" if self.ctx.get(response_saver.UNSTABLE_OUTPUT_KEY) else ""
        return None


__Task__ = GetMiddlesearchResponses
