import logging

from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.projects import resource_types
from sandbox.projects.common import binary_task
import sandbox.projects.common.search.components.mkl as sc_mkl
import sandbox.projects.common.search.bugbanner2 as bb2
import sandbox.projects.websearch.middlesearch.base_tasks.single_host_task as ms_single_host


class GetWebMiddlesearchResponsesSingleHost(
    binary_task.LastBinaryTaskRelease,
    ms_single_host.WebMiddlesearchSingleHost,
):
    """Task for getting responses from local web middlesearch."""

    class Requirements(ms_single_host.WebMiddlesearchSingleHost.Requirements):
        cores = 16  # try using multislots
        client_tags = (ctc.Tag.GENERIC | ctc.Tag.MULTISLOT) & ctc.Tag.Group.LINUX
        Caches = sdk2.Requirements.Caches # Require caches (disallow run on multislot client)

    class Parameters(ms_single_host.WebMiddlesearchSingleHost.Parameters):
        _task_binary = binary_task.binary_release_parameters(stable=True)
        cpu_model_tag = sdk2.parameters.CustomClientTags(
            'CPU model client tag',
            default=sdk2.parameters.ctc.Tag.INTEL_E5_2650
        )

    def on_save(self):
        super(GetWebMiddlesearchResponsesSingleHost, self).on_save()
        self.Requirements.client_tags = self.Requirements.client_tags & self.Parameters.cpu_model_tag

    def on_execute(self):
        super(GetWebMiddlesearchResponsesSingleHost, self).on_execute()
        self.add_bugbanner(bb2.Banners.WebMiddleSearch)
        searchers = self.init_single_host_searchers()
        for searcher in searchers:
            sc_mkl.configure_mkl_environment(searcher)
        wbs1, wbs2, wms = searchers
        output_res = self.output_responses()
        with wbs1, wbs2, wms:
            logging.info("Launch %s", wms.name)
            self.save_responses(
                self.get_target_port(wms),
                str(sdk2.ResourceData(output_res).path),
            )

    def output_responses(self):
        if self.Parameters.make_binary:
            return resource_types.SEARCH_PROTO_RESPONSES(
                self, "Middlesearch responses, {}".format(self.Parameters.description.encode("utf-8")),
                "middlesearch_responses.bin"
            )
        else:
            return resource_types.BASESEARCH_HR_RESPONSES(
                self, "Middlesearch responses, {}".format(self.Parameters.description.encode("utf-8")),
                "middlesearch_responses.txt"
            )
