# -*- coding: utf-8 -*-

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk import parameters as sp
from sandbox.projects.common.middlesearch import single_host
from sandbox.projects.common import dolbilka
from sandbox.projects.websearch.middlesearch import resources as ms_res


class PlanParameter(sp.LastReleasedResource):
    name = 'dolbilo_plan_resource_id'
    description = 'Plan'
    resource_type = ms_res.MiddlesearchPlan


class MeasureMiddlesearchMemoryUsage(single_host.MiddlesearchSingleHostTask):
    """
        Starts 1 middlesearch instance.
        Runs dolbilka session to get rss (Resident Set Size)
        and vsz (Virtual Memory Size).
    """

    type = 'MEASURE_MIDDLESEARCH_MEMORY_USAGE'
    required_ram = 96 << 10
    execution_space = 50 * 1024  # 50G
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = (
        single_host.PARAMS + (PlanParameter,) +
        dolbilka.DolbilkaExecutor.input_task_parameters
    )

    @property
    def footer(self):
        if "memory_bytes" in self.ctx:
            return [{
                'helperName': '',
                'content': {
                    "<h4>Memory usage</h4>": {
                        k: "{:.3f} Mb".format(v / (1024*1024.0)) for k, v in self.ctx["memory_bytes"][0].items()
                    }
                }
            }]
        info = {}
        for m_type in ["rss", "vsz"]:
            info[m_type] = "{:.3} Gb".format(self.ctx.get('memory_' + m_type, [-1024*1024])[0] / (1024*1024.0))
        return [{
            'helperName': '',
            'content': {"<h4>Memory usage</h4>": info}
        }]

    def _use_middlesearch_component(self, middlesearch):
        plan = self.sync_resource(self.ctx[PlanParameter.name])
        d_executor = dolbilka.DolbilkaExecutor()
        d_executor.run_sessions(
            plan,
            middlesearch,
            run_once=True,
        )

    def get_short_task_result(self):
        if not self.is_completed():
            return None

        rss = self.ctx.get('memory_rss')
        vsz = self.ctx.get('memory_vsz')
        if rss and vsz:
            return '{:.3} / {:.3}'.format(rss[0] / float(1024 * 1024), vsz[0] / float(1024 * 1024))


__Task__ = MeasureMiddlesearchMemoryUsage
