# coding=utf-8
import logging
import json
import shutil

from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import utils
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.search import compare_middle_utils as cmu
from sandbox.projects.websearch.middlesearch import CompareMiddlesearchBinaries as cmb
from sandbox.projects.websearch.middlesearch.PriemkaMiddlesearchBinary import res, params
from sandbox.projects import CompareEventlogStats

shooting_params = cmu.create_shooting_params()
shooting_params.Nruns.default_value = 10
shooting_params.ReqCount.default_value = 5000
shooting_params.WarmupRequestCount.default_value = 0
shooting_params.Rps.default_value = 10

_COMPARE_MIDDLESEARCH_BINARIES_KEY = "compare_middlesearch_binaries_subtask_id"
_MIDDLESEARCH_COMPARE_SUBSOURCE_REQUESTS_KEY = "middlesearch_compare_subsource_requests_subtask_id"


class MiddlesearchCacheCompatibility(bugbanner.BugBannerTask):
    """
        For testenv test testenv/jobs/middlesearch/MiddleSearchCacheCompatibilityDiff.yaml

        * Запускает COMPARE_MIDDLESEARCH_BINARIES для проверки совместимости кеша (сравнивает число запросов к источникам)
        * Запускает MIDDLESEARCH_COMPARE_SUBSOURCE_REQUESTS для сравнения самих запросов к источникам (результат сравнения и есть результат testenv дифф. теста)
    """

    type = "MIDDLESEARCH_CACHE_COMPATIBILITY"


    input_parameters = (
        params.BuildTaskOld,
        params.BuildTaskNew,
        cmb.TestEnvDiffTask,
        shooting_params.Nruns,
        shooting_params.ReqCount,
        shooting_params.WarmupRequestCount,
        shooting_params.Rps,
        params.AddCgi,
    )
    execution_space = 2000  # 2 Gb

    def on_execute(self):
        with self.memoize_stage.create_children:
            self._create_compare_middlesearch_binaries_child_task()
            self._create_middlesearch_compare_subsource_requests_child_task()

        fail_on_failed_children = not self.ctx.get(cmb.TestEnvDiffTask.name, False)
        utils.check_subtasks_fails(fail_on_failed_children=fail_on_failed_children)

        self._check_cache_compatibility()
        self._forward_compare_subsource_diff_resource()
        self._set_info()

    def _create_compare_middlesearch_binaries_child_task(self):
        auto_mode = "cache_compatibility"
        bin_level = "mmeta"

        additional_cgi = self.ctx.get(params.AddCgi.name, "")
        self.ctx[params.BinType.name] = "web"
        res_keeper = res.MiddleResourceKeeper(bin_level, self.ctx, additional_cgi)

        input_ctx = {
            "notify_via": "",
            cmb.AutoModeParam.name: auto_mode,
            cmb.TestEnvDiffTask.name: self.ctx.get(cmb.TestEnvDiffTask.name, False),
            shooting_params.Nruns.name: utils.get_or_default(self.ctx, shooting_params.Nruns),
            shooting_params.ReqCount.name: utils.get_or_default(self.ctx, shooting_params.ReqCount),
            shooting_params.WarmupRequestCount.name: utils.get_or_default(self.ctx, shooting_params.WarmupRequestCount),
            shooting_params.Rps.name: utils.get_or_default(self.ctx, shooting_params.Rps),
        }
        MARKS = ["old", "new"]
        for i, mark in enumerate(MARKS):
            input_ctx.update({
                cmb.res_params[i].Binary.name: res_keeper.binary(mark),
                cmb.res_params[i].Config.name: res_keeper.config(mark),
                cmb.res_params[i].Evlogdump.name: res_keeper.evlogdump(mark),
                cmb.res_params[i].Requests.name: res_keeper.queries(mark),
                cmb.res_params[i].Data.name: res_keeper.data(mark),
                cmb.res_params[i].Index.name: res_keeper.index(mark),
                cmb.res_params[i].ArchiveModel.name: res_keeper.models(mark),
                cmb.res_params[i].UseInt.name: False,
            })
        create_task_params = {
            "task_type": "COMPARE_MIDDLESEARCH_BINARIES",
            "input_parameters": input_ctx,
            "description": "Compare mmeta binaries (auto_mode='cache_compatibility'), {}".format(self.descr),
            "arch": sandboxapi.ARCH_LINUX,
        }
        logging.info("Params for COMPARE_MIDDLESEARCH_BINARIES:\n%s", json.dumps(create_task_params, indent=2))
        self.ctx[_COMPARE_MIDDLESEARCH_BINARIES_KEY] = self.create_subtask(**create_task_params).id

    def _create_middlesearch_compare_subsource_requests_child_task(self):
        input_ctx = {
            "notify_via": "",
            params.BuildTaskOld.name: self.ctx[params.BuildTaskOld.name],
            params.BuildTaskNew.name: self.ctx[params.BuildTaskNew.name],
        }

        create_task_params = {
            "task_type": "MIDDLESEARCH_COMPARE_SUBSOURCE_REQUESTS",
            "input_parameters": input_ctx,
            "description": "Compare subsource requests, {}".format(self.descr),
            "arch": sandboxapi.ARCH_LINUX,
        }
        logging.info("Params for MIDDLESEARCH_COMPARE_SUBSOURCE_REQUESTS:\n%s", json.dumps(create_task_params, indent=2))
        self.ctx[_MIDDLESEARCH_COMPARE_SUBSOURCE_REQUESTS_KEY] = self.create_subtask(**create_task_params).id

    def _check_cache_compatibility(self):
        compare_task_id = self.ctx.get(_COMPARE_MIDDLESEARCH_BINARIES_KEY)
        eh.ensure(compare_task_id, "Compare binaries task doesn't exist")

        compare_task = channel.sandbox.get_task(compare_task_id)
        # при auto_mode="cache_compatibility" условие significant_diff==True равнозначно тому что cache altered
        self.ctx["has_diff"] = compare_task.ctx.get(cmb.KEY_SIGNIFICANT_DIFF, False)

    def _forward_compare_subsource_diff_resource(self):
        task_id = self.ctx.get(_MIDDLESEARCH_COMPARE_SUBSOURCE_REQUESTS_KEY)
        eh.ensure(task_id, "Compare subsource requests task doesn't exist")

        task = channel.sandbox.get_task(task_id)
        resource_infos = task.ctx[cmb.CompareMiddlesearchBinaries._KEY_SUBSOURCE_REQUESTS_DIFF]
        diff_resource_id = resource_infos[0][1]
        resource_from_subtask = self.sync_resource(diff_resource_id)

        # приходится пересодавать ресурс, потому что он создан дочерней задачей
        # но testenv ничего не знает про дочернюю задачу
        # то есть нужно чтобы ресурс был создан текущей задачей (чтобы в testenv была ссылка на этот ресурс с диффом)
        resource_copy = self.create_resource(
            "{}, subsource requests diff".format(self.descr), "diff",
            CompareEventlogStats.COMPARE_SUBSOURCE_REQUESTS_RESULT
        ).path
        shutil.copytree(resource_from_subtask, resource_copy)

    def _set_info(self):
        message_why = "See task {} for details why cache {}".format(
            lb.task_link(self.ctx[_COMPARE_MIDDLESEARCH_BINARIES_KEY]),
            "altered" if self.ctx["has_diff"] else "not altered"
        )
        self.set_info(message_why, do_escape=False)

        message_diff = "See task {} for links to diff of subsource requests".format(lb.task_link(self.ctx[_MIDDLESEARCH_COMPARE_SUBSOURCE_REQUESTS_KEY]))
        self.set_info(message_diff, do_escape=False)


__Task__ = MiddlesearchCacheCompatibility
