# coding=utf-8
import logging
import json

from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import utils
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.search import compare_middle_utils as cmu
from sandbox.projects.websearch.middlesearch import CompareMiddlesearchBinaries as cmb
from sandbox.projects.websearch.middlesearch.PriemkaMiddlesearchBinary import res, params
from sandbox.projects.websearch import CalcEventlogStats

shooting_params = cmu.create_shooting_params()
shooting_params.Nruns.default_value = 2
shooting_params.ReqCount.default_value = 200
shooting_params.WarmupRequestCount.default_value = 0
shooting_params.Rps.default_value = 10

_COMPARE_MIDDLESEARCH_BINARIES_KEY = "compare_middlesearch_binaries_subtask_id"


class MiddlesearchCompareSubsourceRequests(bugbanner.BugBannerTask):
    """
        Производит сравнение запросов от среднего к базовым (для двух ревизий). MIDDLE-43
    """

    type = "MIDDLESEARCH_COMPARE_SUBSOURCE_REQUESTS"


    input_parameters = (
        params.BuildTaskOld,
        params.BuildTaskNew,
        shooting_params.Nruns,
        shooting_params.ReqCount,
        shooting_params.WarmupRequestCount,
        shooting_params.Rps,
        params.AddCgi,
    )
    execution_space = 2000  # 2 Gb

    def on_execute(self):
        with self.memoize_stage.create_children:
            self._create_child_task()

        utils.check_subtasks_fails(fail_on_failed_children=False)

        compare_task_id = self.ctx.get(_COMPARE_MIDDLESEARCH_BINARIES_KEY)
        eh.ensure(compare_task_id, "Compare binaries task doesn't exist")
        compare_task = channel.sandbox.get_task(compare_task_id)

        resource_infos = compare_task.ctx[cmb.CompareMiddlesearchBinaries._KEY_SUBSOURCE_REQUESTS_DIFF]
        for compare_task_id, resource_id in resource_infos:
            message = 'Diff of subsource requests: {} (from task {})'.format(lb.resource_link(resource_id), lb.task_link(compare_task_id))
            self.set_info(message, do_escape=False)

        # for task MIDDLESEARCH_CACHE_COMPATIBILITY
        self.ctx[cmb.CompareMiddlesearchBinaries._KEY_SUBSOURCE_REQUESTS_DIFF] = resource_infos

    def _create_child_task(self):
        auto_mode = "nocache"
        bin_level = "mmeta"

        self.ctx[params.BinType.name] = "web"
        additional_cgi_params = utils.get_or_default(self.ctx, params.AddCgi)
        res_keeper = res.MiddleResourceKeeper(bin_level, self.ctx, additional_cgi_params)

        input_ctx = {
            "notify_via": "",
            cmb.AutoModeParam.name: auto_mode,
            shooting_params.Nruns.name: utils.get_or_default(self.ctx, shooting_params.Nruns),
            shooting_params.ReqCount.name: utils.get_or_default(self.ctx, shooting_params.ReqCount),
            shooting_params.WarmupRequestCount.name: utils.get_or_default(self.ctx, shooting_params.WarmupRequestCount),
            shooting_params.Rps.name: utils.get_or_default(self.ctx, shooting_params.Rps),
        }
        MARKS = ["old", "new"]
        for i, mark in enumerate(MARKS):
            input_ctx.update({
                cmb.res_params[i].Binary.name: res_keeper.binary(mark),
                cmb.res_params[i].Config.name: res_keeper.config(mark),
                cmb.res_params[i].Evlogdump.name: res_keeper.evlogdump(mark),
                cmb.res_params[i].Requests.name: res_keeper.queries(mark),
                cmb.res_params[i].Data.name: res_keeper.data(mark),
                cmb.res_params[i].Index.name: res_keeper.index(mark),
                cmb.res_params[i].ArchiveModel.name: res_keeper.models(mark),
                cmb.res_params[i].UseInt.name: False,
                CalcEventlogStats.CompareSubSourceRequests.name: True,
            })
        create_task_params = {
            "task_type": "COMPARE_MIDDLESEARCH_BINARIES",
            "input_parameters": input_ctx,
            "description": "Compare mmeta binaries (auto_mode='nocache'), {}".format(self.descr),
            "arch": sandboxapi.ARCH_LINUX,
        }
        logging.info("Params for COMPARE_MIDDLESEARCH_BINARIES:\n%s", json.dumps(create_task_params, indent=2))
        self.ctx[_COMPARE_MIDDLESEARCH_BINARIES_KEY] = self.create_subtask(**create_task_params).id


__Task__ = MiddlesearchCompareSubsourceRequests
