# -*- coding: utf-8 -*-

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters as sp

from sandbox.projects.common import utils
from sandbox.projects.common import dolbilka
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.base_search_quality import settings as bss


class RunProfilerParameter(sp.SandboxBoolParameter):
    name = 'run_profiler'
    description = 'Run profiler'
    group = 'Profiling'
    default_value = False


class UseGPerfToolsParameter(sp.SandboxBoolParameter):
    name = 'use_gperftools'
    description = 'Use GPerfTools'
    group = 'Profiling'
    default_value = False


class PlanParameter(sp.ResourceSelector):
    name = 'dolbilo_plan_resource_id'
    description = 'Plan'
    resource_type = ['MIDDLESEARCH_PLAN', 'IMAGES_MIDDLESEARCH_PLAN', 'VIDEO_MIDDLESEARCH_PLAN']
    required = True


class MiddlesearchLoad(SandboxTask):
    """
        1 middlesearch looking into production + d-executor.
        Crash test for middlesearch in release/debug modes (for SEARCH-1185, etc)
        basically for manual runs.
    """

    type = 'MIDDLESEARCH_LOAD'

    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = (
        sc.DefaultMiddlesearchParams.params + (
            # profiling
            RunProfilerParameter,
            UseGPerfToolsParameter,
            # plan
            PlanParameter,
        ) +
        dolbilka.DolbilkaExecutor.input_task_parameters
    )

    execution_space = (
        bss.MAX_MIDDLE_SHARD_SIZE +
        bss.MAX_MIDDLE_CACHE_SIZE +
        bss.RESERVED_SPACE
    )

    def on_execute(self):
        middlesearch = self._create_middlesearch_component()

        plan = self.sync_resource(self.ctx[PlanParameter.name])

        d_executor = dolbilka.DolbilkaExecutor()

        results = d_executor.run_sessions(plan, middlesearch, run_once=True)
        dolbilka.DolbilkaPlanner.fill_rps_ctx(results, self.ctx)

    def _create_middlesearch_component(self):

        use_profiler = utils.get_or_default(self.ctx, RunProfilerParameter)
        use_gperftools = utils.get_or_default(self.ctx, UseGPerfToolsParameter)

        middlesearch = sc.get_middlesearch(
            disable_timeouts=False,
            use_profiler=use_profiler or use_gperftools,
            use_gperftools=use_gperftools,
        )

        return middlesearch


__Task__ = MiddlesearchLoad
